#include <QtDeclarative>
#include <QtMaemo5>
#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include "quickwidget.hpp"

#define SETTING_QUICK_FILE "QuickWidgetFile"
#define SETTING_QUICK_SIZE "QuickWidgetSize"
#define SETTING_SAVE_WAITINTERVAL 3000

static Atom onCurrentHomescreenAtom;
static bool _atomsInitialized = false;

QuickWidget *QuickWidget::createAndShowNew(const QString& fileName, 
					   const QSize& size)
{
    QuickWidget *widget = new QuickWidget;
    widget->initView(fileName, size);
    widget->show();
    QeMaemo5DynamicWidgetHelper::globalInstance()->registerWidget(widget);
    return widget;
}

void QuickWidget::initView(const QString& fileName, const QSize& size)
{
    size_ = size;
    saveSetting(SETTING_QUICK_FILE, fileName);
    saveSetting(SETTING_QUICK_SIZE, size_);
    view_->setSource(QUrl::fromLocalFile(fileName));
    if ( view_->status() == QDeclarativeView::Error) {
        errorWidget();
    }
    else if (size_.isValid())
    {
        // use given fixed size and scale to it
        qDebug() << "scaling";
        view_->setResizeMode(QDeclarativeView::SizeRootObjectToView);
    }
    else
    {
        size_ = view_->sceneRect().size().toSize();
    }    
    connect(view_, SIGNAL(sceneResized(QSize)), this, SLOT(resizer(QSize)));

    view_->resize(size_);
    view_->show();
}

QuickWidget::QuickWidget(QWidget *parent)
    : QeMaemo5DynamicHomescreenWidget(true, parent),
    size_(QSize()),
    view_(0)
{
    if (!_atomsInitialized) {
       onCurrentHomescreenAtom = XInternAtom(QX11Info::display(),
                                             "_HILDON_APPLET_ON_CURRENT_DESKTOP",
                                             false);
       _atomsInitialized = true;
    }
    //layout()->setSizeConstraint(QLayout::SetNoConstraint);
    view_ = new QDeclarativeView(this);
    view_->setStyleSheet("background:transparent");
    view_->setAttribute(Qt::WA_TranslucentBackground);
    wrapper_ = new QuickWidgetWrapper(this);
    view_->rootContext()->setContextProperty("runtime", wrapper_);
}

QuickWidget::~QuickWidget()
{
    delete wrapper_;
}

bool QuickWidget::x11Event(XEvent *event)
{
    bool passon = QeMaemo5DynamicHomescreenWidget::x11Event(event);

    if (event->xclient.message_type == onCurrentHomescreenAtom)
    {
        bool visible = isVisibleOnCurrentHomescreen() ;
        if (visible != wrapper_->onHomeScreen())
        {
            wrapper_->setOnHomeScreen(visible);
        }
    }
    return passon;
}

bool QuickWidget::restoreWidgetState()
{
    QString fileName = loadSetting(SETTING_QUICK_FILE).toString();
    QSize size = loadSetting(SETTING_QUICK_SIZE, QSize()).value<QSize>();
    initView(fileName, size);
    return true;
}

void QuickWidget::errorWidget()
{
    const QList<QDeclarativeError>& errs = view_->errors();
    qDebug() << errs;
    QByteArray errmsg;
    QTextStream qts(&errmsg);
    qts << "import Qt 4.7" << endl
        << "Rectangle { width: 200; height: txt.height; color: \"#ff0000\"; radius: 8" << endl
        << "Text { id: txt; anchors.centerIn: parent; text: \"";
    for (int i = 0; i < errs.size(); ++i) {
        qts << errs.at(i).toString().replace(QString("\""), QString("\\\"")) <<
endl;
    }
    qts << "\"; font.pixelSize: 15; wrapMode: Text.Wrap; width: parent.width-20}" << endl
        << "}" << endl;
    qDebug() <<  errmsg;
    QDeclarativeComponent component(view_->engine());
    component.setData(errmsg, QUrl());
    qDebug() << component.errorString();
    QDeclarativeItem *errwdgt =
      qobject_cast<QDeclarativeItem*>(component.create(view_->rootContext()));
    view_->scene()->addItem(errwdgt);
    size_ = QSize(errwdgt->width(), errwdgt->height());
}
    
void QuickWidget::paintEvent(QPaintEvent *event)
{
    QWidget::paintEvent(event);
}

void QuickWidget::resizer( QSize size)
{
  if (size_.isValid())
  {
      resize(size);
  }
}

