#ifndef _QTUIPLUS_OPTIONS_HPP_
#define _QTUIPLUS_OPTIONS_HPP_

#include <QPointer>
#include <QHash>
#include <QVector>
#include <QString>
#include <QAbstractItemModel>
#include <QSharedPointer>

namespace qtplus
{

class OptionsGroup;

class Option
{
public:

	Option(QString const& name, QVariant const& value)
			: name_(name)
			, value_(value) { }

	QString const& name() const {
		return name_;
	}

	QVariant const& value() const {
		return value_;
	}

private:
	QString name_;
	QVariant value_;
};

class OptionsGroup
{
	typedef QVector<size_t> options_t;
	typedef QHash<QString, size_t> index_t;

public:

	typedef SettingsModel* model_ptr;

	OptionsGroup(QString const& name, model_ptr)
			: name_(name) { }

	void addOption(QString const& name, QVariant const& value) {
		if (idx_.find() != idx_.end()) {
			return false;
		}

		size_t pos = model_->addOption(this, name, value);
		options_.push_back(pos);
		idx_[name] = pos;
		return true;
	}

	QString name() const {
		return name_;
	}

	size_t optionsCount() const {
		return options_.size();
	}

	size_t getOptionPos(size_t pos) const {
		return ( (0 <= pos && pos < options_.size())
		         ? options_[pos]
		         : 0 );
	}

private:
	QString name_;
	model_ptr model_;
	options_t options_;
	index_t idx_;
};

class SettingsModel : public QAbstractItemModel
{
	Q_OBJECT;

public:

	SettingsModel(QObject *parent = 0);

	typedef QSharedPointer<OptionsGroup> group_ptr;

	QModelIndex index(int row, int col, const QModelIndex &parent = QModelIndex() ) const;
	QModelIndex parent(const QModelIndex &child) const;
	int rowCount(const QModelIndex &parent = QModelIndex() ) const;
	int columnCount(const QModelIndex &parent = QModelIndex() ) const;
	QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const;

	bool addGroup(QString const& name, SettingsModel*);
	bool addOption(OptionsGroup *group, QString const& name, QVariant const& value);

private:

	typedef QVector<group_ptr> groups_t;
	typedef QHash<QString, group_ptr> groups_index_t;
	typedef QSharedPointer<Option> option_ptr;
	typedef QVector<option_ptr> options_t;

	groups_t groups_;
	groups_index_t idx_;
	options_t options_;
};


} // namespace qtplus

#endif // _QTPLUS_OPTIONS_HPP_
