/***************************************************************************
 *   Copyright (C) 2010 by Ixonos Plc   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qtrapids/format.hpp>

#include "DownloadView.h"

#include <QDebug>
#include <QVariant>
#include <QColor>

namespace qtrapids
{

DownloadView::DownloadView(QWidget* parent) :
		QTreeWidget(parent),
		items_(),
		settings_()
{
	setRootIsDecorated(false); // Hide branch lines, making one-level treeview (similar to list)
	setHeaderItem(DownloadViewItem::getHeaderItem());

	connect(this, SIGNAL(itemPressed(QTreeWidgetItem*, int)),
	        this, SLOT(on_itemClicked(QTreeWidgetItem*, int)));

}


DownloadView::~DownloadView()
{
}

void DownloadView::updateItem(TorrentState const& info, ParamsMap_t other_info)
{
	DownloadItems_t::iterator p = items_.find(info.hash);
	switch (info.action) {
	case TorrentState::action_add :
		if (p == items_.end()) {
			addItem_(info, other_info);
		} else {
			qWarning() << "item with similar info hash marked as added";
			updateItem_(p.value(), info, other_info);
		}
		break;
	case TorrentState::action_update :
		if (p != items_.end()) {
			updateItem_(p.value(), info, other_info);
		} else {
			qWarning() << "item does not exist in list but information update arrived";
		}
		break;
	case TorrentState::action_remove :
		if (p != items_.end()) {
			removeItem_(p.value(), info);
		} else {
			qWarning() << "item removal request arrived but there is no such item";
		}
		break;
	default:
		qWarning() << "unknown action requested: " << info.action;
		break;
	}
}

void DownloadView::removeItem_(DownloadViewItem *item, TorrentState const& info)
{
	QString hash = item->getHash();

	int removed = items_.remove(hash);
	if (!removed)
		qDebug() << "Inconsistent download view state on item removal";

	int index = indexOfTopLevelItem(item);
	if (index >= 0) {
		takeTopLevelItem(index);
	}
}

void DownloadView::addItem_(TorrentState const& info, ParamsMap_t)
{
	DownloadViewItem *item = new DownloadViewItem
	( info.hash,
	  QStringList()
	  << info.name
	  << formatSize(info.total_size)
	  << GetStatusString((TorrentStatus::Id)info.state)
	  << formatProgress(info.progress)
	  << formatSize(info.down_rate)
	  << formatSize(info.up_rate)
	  << QString::number(info.seeds) + "/" + QString::number(info.leeches)
	  << QString::number(info.ratio)
	  << "ETA" );

	QBrush brushTmp(GetStatusColor((TorrentStatus::Id)info.state));
	item->setForeground(2, brushTmp);

	addTopLevelItem(item);
	items_[info.hash] = item;
}


void DownloadView::updateItem_(DownloadViewItem *item
                               , TorrentState const& info, ParamsMap_t)
{
	item->setData(2, Qt::DisplayRole,
	              QVariant(GetStatusString((TorrentStatus::Id)info.state)));
	item->setData(3, Qt::DisplayRole,
	              QVariant(formatProgress(info.progress)));
	item->setData(4, Qt::DisplayRole,
	              QVariant(formatSize(info.down_rate)));
	item->setData(5, Qt::DisplayRole,
	              QVariant(formatSize(info.up_rate)));
	item->setData(6, Qt::DisplayRole,
	              QString::number(info.seeds) + "/" + QString::number(info.leeches));
	item->setData(7, Qt::DisplayRole, QString::number(info.ratio));
	
	// Calculate ETA
	if (info.down_rate > 0) {
		qulonglong eta = (info.total_size - info.total_done) / info.down_rate;
		item->setData(8, Qt::DisplayRole, formatElapsedTime(eta));
	}	else {
		item->setData(8, Qt::DisplayRole, "N/A");
	}
	
	// Set color for status text
	QBrush brushTmp(GetStatusColor((TorrentStatus::Id)info.state));
	item->setForeground(2, brushTmp);
}


QString DownloadView::prepareRemoveSelected()
{
	qDebug() << "DownloadView::removeSelected() " <<  topLevelItemCount() ;

	DownloadViewItem *item = dynamic_cast<DownloadViewItem*> (currentItem());
	QString hash = item->getHash();

	item->setDisabled(true);

	qDebug() << "DownloadView::removeSelected() " <<  topLevelItemCount() ;

	return hash;
}


void DownloadView::on_itemClicked(QTreeWidgetItem * , int)
{
	/*
	  qDebug() << "DownloadView::on_itemClicked(()" << item << "," << column;
	  qDebug() << "current item" << currentItem();

	  if (item == currentItem() && item->isSelected()) {
	  item->setSelected(false);
	  }
	*/
}


void DownloadView::saveView()
{
		QTreeWidgetItem *item = headerItem();
		QList<QVariant> columns;
		
		for (int i = 0; i < item->columnCount(); ++i) {
			isColumnHidden(i) ? columns.push_back(QVariant(false)) : columns.push_back(QVariant(true));
		}
		
	settings_.setValue("downloadview_columns", QVariant(columns));
}


void DownloadView::restoreView()
{
	QTreeWidgetItem *item = headerItem();
	QVariant columns(settings_.value("downloadview_columns"));
	QList<QVariant> columnList = columns.toList();
	
	for (int i = 0; i < columnList.size(); ++i) {
		columnList.at(i).toBool() ? setColumnHidden(i, false) : setColumnHidden(i, true);
	}
}


QString DownloadView::GetStatusString(TorrentStatus::Id status)
{
	switch (status) {
		case TorrentStatus::QUEUED_FOR_CHECKING :
			return tr("Queued");
		case TorrentStatus::CHECKING_FILES :
			return tr("Checking");
		case TorrentStatus::DOWNLOADING_METADATA :
			return tr("DL meta");
		case TorrentStatus::DOWNLOADING :
			return tr("Downloading");
		case TorrentStatus::FINISHED :
			return tr("Finished");
		case TorrentStatus::SEEDING :
			return tr("Seeding");
		case TorrentStatus::ALLOCATING :
			return tr("Allocating");
		case TorrentStatus::CHECKING_RESUME_DATA :
			return tr("Checking resume");
		default:
			return tr("N/A");
	}
}


QColor DownloadView::GetStatusColor(TorrentStatus::Id status)
{
	QColor green(40,205,40);
	QColor yellow(255,174,0);

	switch (status) {
		case TorrentStatus::QUEUED_FOR_CHECKING :
		case TorrentStatus::CHECKING_FILES :
		case TorrentStatus::DOWNLOADING_METADATA :
		case TorrentStatus::ALLOCATING :
		case TorrentStatus::CHECKING_RESUME_DATA:
			return yellow;
		case TorrentStatus::DOWNLOADING :
		case TorrentStatus::FINISHED :
		case TorrentStatus::SEEDING :
			return green;
		default:
			return QColor();
	}
}

} // namespace qtrapids
