/***************************************************************************
 *   Copyright (C) 2010 by Ixonos Plc   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <QDebug>
#include <QTimer>
#include <QVariant>
#include <QColor>
#include "DownloadView.h"


DownloadView::DownloadView(QWidget* parent) :
		QTreeWidget(parent),
		items_(),
		timer_(NULL)
{
	setRootIsDecorated(false); // Hide branch lines, making one-level treeview (similar to list)
	setHeaderItem(DownloadViewItem::getHeaderItem());

	connect(this, SIGNAL(itemPressed(QTreeWidgetItem*, int)),
	        this, SLOT(on_itemClicked(QTreeWidgetItem*, int)));

	timer_ = new QTimer(this);
	connect(timer_, SIGNAL(timeout()), this, SLOT(on_timeout()));
	timer_->start(5000);

}


DownloadView::~DownloadView()
{
}


void DownloadView::newItem(qtrapids::QTorrentHandle handle)
{
#ifdef QTRAPIDS_DEBUG
	qDebug() << "DownloadView::newItem() " << items_.count(handle);
#endif

	DownloadViewItem *item = new DownloadViewItem(QStringList()
	        << handle.name()
	        << QString::number(handle.getTotalSize())
	        << GetStatusString(handle.state())
	        << QString::number(100*handle.progress())
	        << QString::number(handle.downloadRate(), 'f', 2)
	        << QString::number(handle.uploadRate(),  'f', 2)
	        << QString::number(handle.numSeeds()) + "/"
	        + QString::number(handle.numLeeches())
	        << QString::number(handle.ratio())
	        << "ETA" );

	// Set text color for status:
	QBrush brushTmp(GetStatusColor(handle.state()));
	item->setForeground(2, brushTmp);

	addTopLevelItem(item);
	items_[handle] = item;
}


void DownloadView::updateItem(qtrapids::QTorrentHandle handle)
{
	//qDebug() << "DownloadView::updateItem() "  << items_.count(handle);

	static float lastProg = 0;

	// If there are items currently downloading, update:
	if (items_.count(handle) > 0) {

		DownloadViewItem *item = items_[handle];

		// Only the changing fields are being updated:
		item->setData(2, Qt::DisplayRole,
		              QVariant(GetStatusString(handle.state())));
		item->setData(4, Qt::DisplayRole,
		              QVariant(QString::number(handle.downloadRate(), 'f', 2)));
		item->setData(5, Qt::DisplayRole,
		              QVariant(QString::number(handle.uploadRate(), 'f', 2)));
		item->setData(6, Qt::DisplayRole,
		              QString::number(handle.numSeeds()) + "/"
		              + QString::number(handle.numLeeches()));

		// Set progress if increment is 1 percent.
		float prog = handle.progress();
		if ((prog-lastProg) >= 0.01 || prog >= 1.0) {
			item->setData(3, Qt::DisplayRole,
			              QVariant(QString::number(100*prog)));
			lastProg = prog;
		}

		/// @TODO: ETA-counter adjusting,if ETA is to be implemented

		// Adjust color
		QBrush brushTmp(GetStatusColor(handle.state()));
		item->setForeground(2, brushTmp);
	}

}


qtrapids::QTorrentHandle DownloadView::removeSelected()
{
#ifdef QTRAPIDS_DEBUG
	qDebug() << "DownloadView::removeSelected() " <<  topLevelItemCount() ;
#endif

	DownloadViewItem *item = dynamic_cast<DownloadViewItem*> (currentItem());

	std::map<qtrapids::QTorrentHandle, DownloadViewItem*>::iterator listIter
	= items_.begin();
	std::map<qtrapids::QTorrentHandle, DownloadViewItem*>::iterator listEnd
	= items_.end();

	while (listIter != listEnd) {
		if (listIter->second == item) {
			break;
		}
		++listIter;
	}

	qtrapids::QTorrentHandle handle = listIter->first;
	items_.erase(listIter);


	int index = indexOfTopLevelItem(currentItem());
	if (index >= 0) {
		takeTopLevelItem(index);
	}

#ifdef QTRAPIDS_DEBUG
	qDebug() << "DownloadView::removeSelected() " <<  topLevelItemCount() ;
#endif

	return handle;
}


void DownloadView::removeItem(qtrapids::QTorrentHandle handle)
{
}


void DownloadView::setRefreshInterval(int msec)
{
	timer_->setInterval(msec);
}


void DownloadView::on_itemClicked(QTreeWidgetItem * item, int column)
{
	/*
	qDebug() << "DownloadView::on_itemClicked(()" << item << "," << column;
	qDebug() << "current item" << currentItem();

	if (item == currentItem() && item->isSelected()) {
		item->setSelected(false);
	}
	*/
}

void DownloadView::on_timeout()
{
#ifdef QTRAPIDS_DEBUG
	qDebug() << "DownloadView::on_timeout()";
#endif
	UpdateView();
}



QString DownloadView::GetStatusString(qtrapids::QTorrentHandle::State const& status) const
{
	switch (status) {
	case qtrapids::QTorrentHandle::QUEUED_FOR_CHECKING :
		return tr("Queued");
	case qtrapids::QTorrentHandle::CHECKING_FILES :
		return tr("Checking");
	case qtrapids::QTorrentHandle::DOWNLOADING_METADATA :
		return tr("DL meta");
	case qtrapids::QTorrentHandle::DOWNLOADING :
		return tr("Downloading");
	case qtrapids::QTorrentHandle::FINISHED :
		return tr("Finished");
	case qtrapids::QTorrentHandle::SEEDING :
		return tr("Seeding");
	case qtrapids::QTorrentHandle::ALLOCATING :
		return tr("Allocating");
	default:
		return tr("N/A");
	}
}


QColor DownloadView::GetStatusColor(qtrapids::QTorrentHandle::State const& status) const
{
	QColor green(40,205,40);
	QColor yellow(255,174,0);

	switch (status) {
	case qtrapids::QTorrentHandle::QUEUED_FOR_CHECKING :
	case qtrapids::QTorrentHandle::CHECKING_FILES :
	case qtrapids::QTorrentHandle::DOWNLOADING_METADATA :
	case qtrapids::QTorrentHandle::ALLOCATING :
		return yellow;
	case qtrapids::QTorrentHandle::DOWNLOADING :
	case qtrapids::QTorrentHandle::FINISHED :
	case qtrapids::QTorrentHandle::SEEDING :
		return green;
	default:
		return QColor();
	}
}

void DownloadView::UpdateView()
{
	DownloadViewItem *item = dynamic_cast<DownloadViewItem*> (currentItem());

	std::map<qtrapids::QTorrentHandle, DownloadViewItem*>::iterator listIter
	= items_.begin();
	std::map<qtrapids::QTorrentHandle, DownloadViewItem*>::iterator listEnd
	= items_.end();

	while (listIter != listEnd) {
		updateItem(listIter->first);
		++listIter;
	}
}
