/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "cityinfodisplay.h"
#include "settings.h"
#include "pixmaploader.h"
#include "utils.h"

#include <QFont>
#include <QPen>
#include <QFontMetrics>
#include <QPainter>
#include <QGraphicsSceneMouseEvent>

#define CURRENT_TEMP_FONT_SIZE  (Settings::intValue("current-temperature-font-size"))
#define TEMP_BOUND_FONT_SIZE    (Settings::intValue("bound-temperature-font-size"))
#define CITY_NAME_FONT_SIZE     (Settings::scaleHeight(50.0))
#define MAX_CITY_NAME_WIDTH     (Settings::scaleWidth(440.0))
#define CITY_NAME_TOP           (Settings::scaleHeight(235.0))

// CurrentTemperatureDisplay

CurrentTemperatureDisplay::CurrentTemperatureDisplay()
    : m_brush(Qt::white)
    , m_sign(PixmapLoader::pixmap("minus_sign"))
    , m_unit(PixmapLoader::pixmap("centigrades"))
    , m_value(-1)
    , m_pos(0.0, 0.0)
{
    m_font.setFamily("Nokia Sans");
    m_font.setBold(true);
    m_font.setPixelSize(CURRENT_TEMP_FONT_SIZE);
    m_font.setStyleStrategy(QFont::PreferAntialias);

    m_pen.setColor(Qt::white);
    m_pen.setBrush(Qt::white);
    m_pen.setJoinStyle(Qt::RoundJoin);

    setTemperature(0);
}

QRectF CurrentTemperatureDisplay::boundingRect () const
{
    return m_boundingRect;
}

void CurrentTemperatureDisplay::paint(QPainter *painter)
{
    if (m_value < 0)
        painter->drawPixmap(m_signPos, m_sign);

    painter->setPen(m_pen);
    painter->setBrush(m_brush);
    painter->setFont(m_font);
    painter->drawText(m_numberPos, m_text);

    painter->drawPixmap(m_unitPos, m_unit);
}

void CurrentTemperatureDisplay::setTemperature(int value)
{
    static const int maxValue = 300;
    value = qMax(qMin(maxValue, value), -maxValue);

    if (m_value == value)
        return;

    m_value = value;
    m_text = QString::number(qAbs(m_value));

    QFontMetrics metrics(m_font);

    int textWidth = metrics.width(m_text);
    int width = textWidth;
    int left = 0;

    if (m_value < 0) {
        left = -(width >> 1);
        m_numberPos = QPointF(left, m_unit.height());
        width += m_sign.width();
        left -= m_sign.width();
        m_signPos = QPointF(left, 0.0);
        m_unitPos = QPointF(left + width, 0.0);
        width += m_unit.width();
    } else {
        width += m_unit.width();
        int left = -(width >> 1);
        m_numberPos = QPointF(left, m_unit.height());
        m_unitPos = QPointF(left + textWidth, 0.0);
    }
    m_boundingRect = QRectF(left, 0.0, width, m_unit.height());

}

// TemperatureBoundDisplay

TemperatureBoundDisplay::TemperatureBoundDisplay(bool lowerBound)
    : m_brush(Qt::white)
    , m_icon(PixmapLoader::pixmap(lowerBound ? "icon_min" : "icon_max"))
    , m_value(-1)
    , m_lowerBound(lowerBound)
    , m_pos(0.0, 0.0)
{
    m_font.setFamily("Nokia Sans");
    m_font.setPixelSize(TEMP_BOUND_FONT_SIZE);
    m_font.setStyleStrategy(QFont::PreferAntialias);

    QColor color(255, 255, 255, 178);

    m_pen.setColor(color);
    m_pen.setBrush(color);
    m_pen.setJoinStyle(Qt::RoundJoin);

    m_brush.setColor(color);

    setValue(0);
}

QRectF TemperatureBoundDisplay::boundingRect () const
{
    return m_boundingRect;
}

void TemperatureBoundDisplay::paint(QPainter *painter)
{
    QRectF rect(m_boundingRect);
    painter->drawPixmap(rect.topLeft(), m_icon);
    rect.setLeft(rect.left() + m_icon.width());

    painter->setPen(m_pen);
    painter->setBrush(m_brush);
    painter->setFont(m_font);
    QPointF textPos(rect.left(), m_icon.height());
    painter->drawText(textPos, m_text);
}

void TemperatureBoundDisplay::setValue(int value)
{
    static const int maxValue = 300;
    value = qMax(qMin(maxValue, value), -maxValue);

    if (m_value == value)
        return;

    m_value = value;
    m_text = QString::number(m_value) + QChar(176) + 'c';

    QFontMetrics metrics(m_font);
    m_boundingRect = QRectF(0.0, 0.0, m_icon.width() + metrics.width(m_text), m_icon.height());
}


ForecastDisplay::ForecastDisplay()
    : m_line(PixmapLoader::pixmap("division_line")),
      m_minIcon(PixmapLoader::pixmap("icon_min_5day")),
      m_maxIcon(PixmapLoader::pixmap("icon_max_5day")),
      m_celsius(PixmapLoader::pixmap("centigrades_5day")),
      m_lower(0),
      m_upper(0)
{
    m_font.setFamily("Nokia Sans");
    m_font.setBold(true);
    m_font.setPixelSize(Settings::intValue("forecast-value-font-size"));
    m_font.setStyleStrategy(QFont::PreferAntialias);

    m_weekFont = m_font;
    m_weekFont.setBold(false);
    m_weekFont.setPixelSize(Settings::intValue("forecast-week-font-size"));

    m_pen.setColor(Qt::white);
    m_pen.setJoinStyle(Qt::RoundJoin);

    m_weekPen = m_pen;
    m_weekPen.setColor(QColor(255, 255, 255, 178));

    m_boundingRect = Settings::value("forecast-bounding-rect").toRect();
}

void ForecastDisplay::setValue(const QString &week, int lower, int upper)
{
    static const int maxValue = 300;

    m_week = week;
    m_lower = QString::number(qBound(-maxValue, lower, maxValue));
    m_upper = QString::number(qBound(-maxValue, upper, maxValue));

    qreal cx = m_boundingRect.center().x();

    QFontMetrics metric1(m_weekFont);
    qreal ti = metric1.height() - Settings::scaleWidth(8);

    m_linePos = QPointF(cx - m_line.width() / 2, ti);

    m_weekPos = QPointF(cx - metric1.width(m_week) / 2,
                        ti - Settings::scaleWidth(5));

    QFontMetrics metric2(m_font);
    const int upperWidth = metric2.width(m_upper);
    const int lowerWidth = metric2.width(m_lower);
    const qreal blockWidth = m_maxIcon.width() + qMax(upperWidth, lowerWidth)
        + m_celsius.width() + Settings::scaleHeight(12);

    qreal lx = (cx - blockWidth / 2);
    qreal rx = (cx + blockWidth / 2) - m_celsius.width();

    qreal tk1 = ti + Settings::scaleHeight(10);
    qreal tk2 = tk1 + m_celsius.height() + Settings::scaleHeight(15);

    m_maxIconPos = QPointF(lx, tk1);
    m_minIconPos = QPointF(lx, tk2);

    m_maxCelsiusPos = QPointF(rx, tk1);
    m_minCelsiusPos = QPointF(rx, tk2);

    rx -= Settings::scaleHeight(5);
    m_upperPos = QPointF(rx - upperWidth, tk1 + m_celsius.height());
    m_lowerPos = QPointF(rx - lowerWidth, tk2 + m_celsius.height());
}

void ForecastDisplay::paint(QPainter *painter)
{
    // paint line
    painter->drawPixmap(m_linePos, m_line);

    // paint week label
    painter->setPen(m_weekPen);
    painter->setFont(m_weekFont);
    painter->drawText(m_weekPos, m_week);

    // paint arrows
    painter->drawPixmap(m_maxIconPos, m_maxIcon);
    painter->drawPixmap(m_minIconPos, m_minIcon);

    // paint degrees
    painter->drawPixmap(m_maxCelsiusPos, m_celsius);
    painter->drawPixmap(m_minCelsiusPos, m_celsius);

    // paint values
    painter->setPen(m_pen);
    painter->setFont(m_font);
    painter->drawText(m_upperPos, m_upper);
    painter->drawText(m_lowerPos, m_lower);
}

// CityInfoDisplay

CityInfoDisplay::CityInfoDisplay(QGraphicsItem *parent)
    : QGraphicsItem(parent),
      m_isForecast(false),
      m_line(PixmapLoader::pixmap("division_line")),
      m_showUpperLower(true),
      m_lowerBound(true),
      m_upperBound(false)
{
    setCacheMode(ItemCoordinateCache);

    m_font.setFamily("Nokia Sans");
    m_font.setPixelSize(CITY_NAME_FONT_SIZE);
    m_font.setStyleStrategy(QFont::PreferAntialias);

    m_pen.setColor(Qt::white);
    m_pen.setBrush(Qt::white);
    m_pen.setJoinStyle(Qt::RoundJoin);

    m_boundingRect = Settings::value("forecast-bounding-rect").toRect();
}

void CityInfoDisplay::setTemperature(int current)
{
    setTemperature(0, 0, current, false);
}

void CityInfoDisplay::setTemperature(int current, int lower, int upper)
{
    setTemperature(lower, upper, current, true);
}

void CityInfoDisplay::setForecast(const QString &weekday, int lower, int upper)
{
    m_isForecast = true;
    m_forecast.setValue(weekday, lower, upper);
    update();
}

void CityInfoDisplay::setTemperature(int lowerBound, int upperBound, int current,
                                     bool showUpperLower)
{
    m_isForecast = false;
    m_showUpperLower = showUpperLower;
    m_lowerBound.setValue(lowerBound);
    m_upperBound.setValue(upperBound);
    m_current.setTemperature(current);

    m_current.setPos(0.0, 0.0);

    const qreal lineRight = m_line.width() >> 1;

    qreal height = m_current.boundingRect().height() + 8.0;

    m_linePos = QPointF(-lineRight, height);
    height += m_line.height() + 8.0;

    m_upperBound.setPos(-lineRight, height);
    m_lowerBound.setPos(lineRight - m_lowerBound.boundingRect().width(), height);
    update();
}

QRectF CityInfoDisplay::boundingRect () const
{
    return m_boundingRect;
}

void CityInfoDisplay::paint(QPainter *painter, const QStyleOptionGraphicsItem *opt,
                            QWidget *widget)
{
    Q_UNUSED(opt);
    Q_UNUSED(widget);

    //painter->fillRect(boundingRect(), Qt::red);

    painter->setPen(m_pen);
    painter->setFont(m_font);

    if (m_isForecast) {
        m_forecast.paint(painter);
    } else {
        painter->translate(m_current.pos());
        m_current.paint(painter);

        if (!m_showUpperLower)
            return;

        painter->translate(m_upperBound.pos() - m_current.pos());
        m_upperBound.paint(painter);

        painter->translate(m_lowerBound.pos() - m_upperBound.pos());
        m_lowerBound.paint(painter);

        painter->translate(-m_lowerBound.pos());
        painter->drawPixmap(m_linePos, m_line);
    }
}

int CityInfoDisplay::loadImages()
{
    PixmapLoader::load("minus_sign");
    PixmapLoader::load("centigrades");
    PixmapLoader::load("icon_min");
    PixmapLoader::load("icon_max");
    PixmapLoader::load("division_line");
    return 5;
}
