/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: openBossa - INdT (renato.chencarek@openbossa.org)
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** the openBossa stream from INdT (renato.chencarek@openbossa.org).
** $QT_END_LICENSE$
**
****************************************************************************/

#include "mainwindow.h"

#include "settings.h"
#include "pixmaploader.h"
#include "forecastmodel.h"
#include "datasource/bbcforecastresolver.h"
#include "datasource/yahooforecastresolver.h"

#include <QTimer>
#include <QApplication>

#ifdef Q_OS_SYMBIAN
#define CONTEXT1KEY Qt::Key_Context1
#define CONTEXT2KEY Qt::Key_Context2
#else
#define CONTEXT1KEY Qt::Key_F1
#define CONTEXT2KEY Qt::Key_F2
#endif

#define CURRENT_WEATHER_LABEL tr("Current weather")
#define WEATHER_FORECAST_LABEL tr("Weather forecast")
#define CHANGE_TO_BBC_LABEL tr("Use BBC Weather")
#define CHANGE_TO_YAHOO_LABEL tr("Use Yahoo! Weather")
#define SELECT_LABEL tr("Select")
#define ADD_LABEL tr("Add")
#define REMOVE_LABEL tr("Remove")
#define BACK_LABEL tr("Back")
#define OPTIONS_LABEL tr("Options")
#define EXIT_LABEL tr("Exit")


MainWindow::MainWindow(QGraphicsItem *parent)
    : QGraphicsWidget(parent),
      m_modelLoaded(false),
      m_pixmapLoaded(false),
      m_carroussel(0),
      m_cityPanel(0),
      m_optionsPanel(0),
      m_loadingScreen(0),
      m_cityOpened(false),
      m_optionsOpened(false),
      m_isInForecast(false)
{
    setFlag(QGraphicsItem::ItemIsFocusable);
    setFlag(QGraphicsItem::ItemHasNoContents);

    m_optionList << new QAction(WEATHER_FORECAST_LABEL, this);

    if (ForecastModel::source() == "bbc")
        m_optionList << new QAction(CHANGE_TO_YAHOO_LABEL, this);
    else
        m_optionList << new QAction(CHANGE_TO_BBC_LABEL, this);

    m_optionList << new QAction(EXIT_LABEL, this);

    if (!Settings::intValue("touch")) {
        m_optionCityList << new QAction(ADD_LABEL, this);
        m_optionCityList << new QAction(REMOVE_LABEL, this);
        connect(m_optionCityList[0], SIGNAL(triggered()), SLOT(doAddCity()));
        connect(m_optionCityList[1], SIGNAL(triggered()), SLOT(doRemoveCity()));
    }

    QAction *exitAction = new QAction(EXIT_LABEL, this);
    m_optionCityList << exitAction;
    connect(exitAction, SIGNAL(triggered()), SLOT(onOptionExit()));

    connect(m_optionList[0], SIGNAL(triggered()), SLOT(onShowForecast()));
    connect(m_optionList[1], SIGNAL(triggered()), SLOT(doChangeResource()));
    connect(m_optionList[2], SIGNAL(triggered()), SLOT(onOptionExit()));
}

void MainWindow::start(bool connected)
{
    setupToolbar();

    m_loadingScreen = new LoadingScreen(connected, this);
    m_loadingScreen->setGeometry(geometry());

    // just to show splash screen
    if (connected)
        QTimer::singleShot(1000, this, SLOT(load()));
}

void MainWindow::setupToolbar()
{
    int h = size().height();

    QPixmap bottomPixmap = PixmapLoader::pixmap("bt_options_bg");
    m_bottomBar = new QGraphicsPixmapItem(bottomPixmap, this);
    m_bottomBar->setZValue(3.0);
    m_bottomBar->setPos(0, h - bottomPixmap.height());

    m_optionsButton = new ImtkButton(QPixmap(), m_bottomBar);
    m_optionsButton->resize(bottomPixmap.width() / 2, bottomPixmap.height());

    QFont f("Nokia Sans");
    f.setPixelSize(Settings::intValue("bottom-panel-label-font-size"));

    m_leftLabel = new ImtkLabel(m_optionsButton);
    m_leftLabel->setFont(f);
    m_leftLabel->setFontColor(Settings::stringValue("bottom-panel-label-color"));
    m_leftLabel->setPos(Settings::scaleWidth(20), 0);
    m_leftLabel->resize(m_optionsButton->size());
    m_leftLabel->setAlignment(Qt::AlignVCenter);

    ImtkButton *exitButton = new ImtkButton(QPixmap(), m_bottomBar);
    exitButton->setPos(bottomPixmap.width() / 2, 0);
    exitButton->resize(bottomPixmap.width() / 2, bottomPixmap.height());

    m_rightLabel = new ImtkLabel(exitButton);
    m_rightLabel->setFont(f);
    m_rightLabel->setFontColor(Settings::stringValue("bottom-panel-label-color"));
    m_rightLabel->setText(EXIT_LABEL);
    m_rightLabel->setPos(0, 0);
    m_rightLabel->resize(exitButton->size().width() - Settings::scaleWidth(20), exitButton->size().height());
    m_rightLabel->setAlignment(Qt::AlignVCenter | Qt::AlignRight);

    connect(exitButton, SIGNAL(clicked()), this, SLOT(onRightButtonClicked()));
    connect(m_optionsButton, SIGNAL(clicked()), this, SLOT(onLeftButtonClicked()));
}

void MainWindow::load()
{
    // load data
    connect(ForecastModel::instance(), SIGNAL(loaded()), SLOT(onDataLoaded()));
    ForecastModel::instance()->load(true);

    // load pixmap
    PixmapLoader::connectToOnIdleSignal(this, SLOT(onPixmapLoaded()));

    ForecastHungItem::loadImages();
    ForecastRain::loadImages();
    ForecastSnow::loadImages();
    ForecastStars::loadImages();
    CityInfoDisplay::loadImages();
    CityCarroussel::loadImages();
    CityList::loadImages();
    AddCityTool::loadImages();
}

void MainWindow::bootEnd()
{
    if (m_loadingScreen) {
        m_loadingScreen->deleteLater();
        m_loadingScreen = 0;
    }

    setupUI();
}

void MainWindow::setupUI()
{
    m_optionsButton->show();
    m_leftLabel->setText(OPTIONS_LABEL);

    // create carroussel
    m_carroussel = new CityCarroussel();
    m_carroussel->setParentItem(this);
    m_carroussel->setGeometry(geometry());

    ForecastModel *model = ForecastModel::instance();
    connect(model, SIGNAL(itemAdded(const ForecastData&)),
            SLOT(onForecastAdded(const ForecastData&)));

    m_carroussel->updateData();

    int w = size().width();
    int h = size().height();

    // create city list panel
    m_cityPanel = new CityPanel(m_carroussel->labels(), m_carroussel);
    m_cityPanel->setZValue(10.0);
    m_cityPanel->setGeometry(0, 0, w, h - m_bottomBar->boundingRect().height());
    connect(m_cityPanel, SIGNAL(collapsed()), SLOT(onCityPanelClosed()));
    connect(m_cityPanel, SIGNAL(cityNameClicked()), SLOT(onCityPanelClicked()));
    connect(m_cityPanel->list(), SIGNAL(itemActivated(int)), SLOT(onCityItemClicked(int)));

    // create options panel
    m_optionsPanel = new OptionsPanel(this);
    m_optionsPanel->setZValue(2.0);
    m_optionsPanel->setGeometry(0, 0, w, h - m_bottomBar->boundingRect().height());
    connect(m_optionsPanel, SIGNAL(collapsed()), SLOT(onOptionsClosed()));

    if (ForecastModel::source() == "bbc") {
        m_optionList[1]->setText(CHANGE_TO_YAHOO_LABEL);
        m_sourceLogo = new QGraphicsPixmapItem(PixmapLoader::pixmap("bbc_logo"), this);
    } else {
        m_optionList[1]->setText(CHANGE_TO_BBC_LABEL);
        m_sourceLogo = new QGraphicsPixmapItem(PixmapLoader::pixmap("yahoo_logo"), this);
    }
    m_sourceLogo->setPos(10, 10);
}

void MainWindow::onForecastAdded(const ForecastData &data)
{
    m_carroussel->add(data);
}

void MainWindow::onDataLoaded()
{
    m_modelLoaded = true;
    disconnect(ForecastModel::instance(), SIGNAL(loaded()),
               this, SLOT(onDataLoaded()));

    if (m_pixmapLoaded)
        bootEnd();
}

void MainWindow::onPixmapLoaded()
{
    m_pixmapLoaded = true;
    PixmapLoader::disconnectReceiver(this);

    if (m_modelLoaded)
        bootEnd();
}

bool MainWindow::eventFilter(QObject *obj, QEvent *event)
{
    if (event->type() == QEvent::KeyPress) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

        if (m_loadingScreen) {
            if (keyEvent->key() == CONTEXT2KEY) {
                qApp->quit();
                return true;
            }
        }

        if (m_loadingScreen || !m_carroussel)
            return QGraphicsWidget::eventFilter(obj, event);

        // open options
        if (keyEvent->key() == CONTEXT1KEY) {
            onLeftButtonClicked();
            return true;
        }

        // exit application
        if (keyEvent->key() == CONTEXT2KEY) {
            onRightButtonClicked();
            return true;
        }

        // exit application
        if (keyEvent->key() == Qt::Key_Up
            && !m_cityOpened && !m_optionsOpened) {
            onCityPanelClicked();
            return true;
        }

        // keypad carroussel navigation
        if (m_carroussel && !m_cityOpened && !m_optionsOpened) {
            if (keyEvent->key() == Qt::Key_Left) {
                m_carroussel->moveRight();
                return true;
            } else if (keyEvent->key() == Qt::Key_Right) {
                m_carroussel->moveLeft();
                return true;
            }
        }
    }

    return QGraphicsWidget::eventFilter(obj, event);
}

void MainWindow::onCityPanelClicked()
{
    if (!m_cityOpened && m_carroussel->active()) {
        m_cityOpened = true;
        m_cityPanel->expand();
        updateInternalState();
    }
}

void MainWindow::onCityPanelClosed()
{
    ForecastModel::instance()->saveToDisk();

    m_cityOpened = false;
    if (!m_isInForecast)
        m_carroussel->updateData();

    updateInternalState();
}

void MainWindow::onOptionsClosed()
{
    m_optionsOpened = false;
    updateInternalState();
}

void MainWindow::onLeftButtonClicked()
{
    if (m_optionsOpened) {
        // TODO: select options item
    } else {
        m_optionsOpened = true;
        if (!m_cityOpened)
            m_optionsPanel->setOptions(m_optionList);
        else
            m_optionsPanel->setOptions(m_optionCityList);
        m_optionsPanel->expand();
        updateInternalState();
    }
}

void MainWindow::onRightButtonClicked()
{
    if (m_optionsOpened) {
        m_optionsPanel->collapse();
    } else if (m_cityOpened) {
        m_cityPanel->collapse();
    } else {
        qApp->quit();
    }
}

void MainWindow::updateInternalState()
{
    if (m_cityOpened || m_optionsOpened) {
        m_carroussel->setFrozen(true);
        m_carroussel->setActive(false);
        m_carroussel->abort();
    } else {
        m_carroussel->setFrozen(false);
        m_carroussel->setActive(true);
    }

    if (m_optionsOpened) {
        m_leftLabel->setText(SELECT_LABEL);
        m_rightLabel->setText(BACK_LABEL);
    } else if (m_cityOpened) {
        m_leftLabel->setText(OPTIONS_LABEL);
        m_rightLabel->setText(BACK_LABEL);
    } else {
        m_leftLabel->setText(OPTIONS_LABEL);
        m_rightLabel->setText(EXIT_LABEL);
    }
}

void MainWindow::onShowForecast()
{
    if (!m_isInForecast) {
        ForecastData d = m_carroussel->currentData();
        QList<ForecastData> forecasts = d.forecasts();

        if (forecasts.count() > 0)
            m_carroussel->setData(d.forecasts());

        m_optionList[0]->setText(CURRENT_WEATHER_LABEL);
        m_isInForecast = true;
    } else {
        m_carroussel->updateData();
        m_optionList[0]->setText(WEATHER_FORECAST_LABEL);
        m_isInForecast = false;
    }

    m_optionsPanel->collapse();
}

void MainWindow::onOptionExit()
{
    qApp->quit();
}

void MainWindow::doAddCity()
{
    m_optionsPanel->collapse();

    if (m_cityOpened)
        m_cityPanel->setInputFocus();
}

void MainWindow::doRemoveCity()
{
    if (m_cityOpened) {
        int index = m_cityPanel->list()->currentItem();

        if (index >= 0 && m_cityPanel->list()->itemCount() > 3)
            m_cityPanel->list()->removeItem(index);

        m_optionsPanel->collapse();
    }
}

void MainWindow::doChangeResource()
{
    bool connected = true;

    if (ForecastModel::source() == "bbc") {
        ForecastModel::setSource("yahoo");
        ForecastModel::setResolver(new YahooForecastResolver());
    } else {
        ForecastModel::setSource("bbc");
        ForecastModel::setResolver(new BbcForecastResolver());
    }

    m_loadingScreen = new LoadingScreen(connected, this);
    m_loadingScreen->setGeometry(geometry());

    m_carroussel->deleteLater();
    m_cityPanel->deleteLater();
    m_optionsPanel->deleteLater();
    delete m_sourceLogo;

    m_modelLoaded = false;
    m_carroussel = 0;
    m_cityPanel = 0;
    m_optionsPanel = 0;
    m_cityOpened = false;
    m_optionsOpened = false;
    m_isInForecast = false;
    m_optionsButton->hide();

    // just to show splash screen
    if (connected)
        QTimer::singleShot(1000, this, SLOT(load()));
}

void MainWindow::onCityItemClicked(int index)
{
    m_carroussel->setCurrentIndex(index);
}
