/* Copyright (c) 2010, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Object to handle async, complex header search from modest / tinymail
 */

/* System includes */
#include <tny-simple-list.h>
#include <tny-account.h>
#include <tny-folder.h>
#include <tny-folder-store.h>
#include <tny-folder-store-query.h>

/* Own includes */
#include "qtmm-search.h"
#include "qtmm-debug.h"
#include "qtmm-marshal.h"
#include "qtmm-folder-search.h"

enum {
    SIG_HEADERS_RECEIVED = 0,
    SIG_DONE,
    SIGNALS
};
static gulong signals[SIGNALS] = {0};

/* Private data definition */
struct _QtmMSearchPrivate {
    gboolean    disposed;

    TnyAccount  *account;
    gchar      **folder_names;
    time_t       begin_time;
    time_t       end_time;
    gboolean     sync;

    gint         progress;
    gint         max;

    GSList      *search_operations;

    TnyFolderStoreQuery *query;

    gint         op_id;

    /* This is a funky index determening if we have pending calls
     * to list folders. Apparently modest does something similar.
     */
    gint         folder_calls;

    /* Search state */
    QtmmSearchState status;
};

#define GET_PRIVATE(o) ((QtmMSearchPrivate *)           \
                        ((QtmMSearch *)(o))->priv)

G_DEFINE_TYPE (QtmMSearch, qtmm_search, G_TYPE_OBJECT);

/*
 * Private function prototypes
 */

/* Reply to getting folders */
static void
get_folders_reply (TnyFolderStore *store,
                   gboolean        cancelled,
                   TnyList        *folders,
                   GError         *error,
                   gpointer        user_data);

/* Status update for folder query */
static void
folder_status_cb (GObject   *store,
                  TnyStatus *status,
                  gpointer   user_data);

/* Reply for listing headers */
static void
headers_received_cb (GObject         *search,
                     const gchar     *folder_id,
                     const GPtrArray *headers,
                     gpointer         user_data);

/* Status update for listing headers */
static void
search_done_cb (GObject  *search,
                gpointer  user_data);

/*
 * GObject funcs
 */

/* Instance init */
static void
qtmm_search_init (QtmMSearch *search)
{
    QtmMSearchPrivate *priv;

    QTMM_DEBUG();

    priv = search->priv = G_TYPE_INSTANCE_GET_PRIVATE (
        search, QTMM_TYPE_SEARCH, QtmMSearchPrivate);

    priv->op_id = -1;
}

/* Instance disposal, unref all other GObjects here */
static void
qtmm_search_dispose (GObject *object)
{
    QtmMSearchPrivate *priv;

    QTMM_DEBUG();

    priv = GET_PRIVATE (object);

    if (priv == NULL) return;

    if (priv->disposed) {
        QTMM_DEBUG ("Already disposed");
        return;
    }

    priv->disposed = TRUE;

    if (priv->account) {
        g_object_unref (priv->account);
        priv->account = NULL;
    }

    if (priv->search_operations) {
        g_slist_foreach (priv->search_operations, (GFunc)g_object_unref, NULL);
        g_slist_free (priv->search_operations);
        priv->search_operations = NULL;
    }

    if (priv->query) {
        g_object_unref (priv->query);
        priv->query = NULL;
    }

    G_OBJECT_CLASS (qtmm_search_parent_class)->dispose (object);
}

/* Instance disposal, unref all other GObjects here */
static void
qtmm_search_finalize (GObject *object)
{
    QtmMSearchPrivate *priv;

    QTMM_DEBUG();

    priv = GET_PRIVATE (object);

    if (priv == NULL) return;

    if (priv->folder_names) {
        g_strfreev (priv->folder_names);
        priv->folder_names = NULL;
    }

    G_OBJECT_CLASS (qtmm_search_parent_class)->finalize (object);
}

/* Class (type) init */
static void
qtmm_search_class_init (QtmMSearchClass *klass)
{
    GObjectClass *object_class;

    QTMM_DEBUG();

    object_class = G_OBJECT_CLASS (klass);

    object_class->dispose     = qtmm_search_dispose;
    object_class->finalize    = qtmm_search_finalize;

    /* Signals */
    signals[SIG_HEADERS_RECEIVED] = g_signal_new (
        "headers-received",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        qtmm_marshal_VOID__INT_STRING_STRING_POINTER,
        G_TYPE_NONE, 4,
        G_TYPE_INT,
        G_TYPE_STRING,
        G_TYPE_STRING,
        G_TYPE_POINTER);

    signals[SIG_DONE] = g_signal_new (
        "done",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        g_cclosure_marshal_VOID__VOID,
        G_TYPE_NONE, 0);

    g_type_class_add_private (klass, sizeof (QtmMSearchPrivate));
}

/*
 * Public functions
 */

/* Create a new instance with appropriate search parameters */
GObject *
qtmm_search_new (TnyAccount  *account,
                 const gchar *folders[],
                 time_t       begin_time,
                 time_t       end_time,
                 gboolean     sync,
                 gint         op_id)
{
    QtmMSearchPrivate *priv;
    GObject *self;

    QTMM_DEBUG();

    g_return_val_if_fail (account != NULL, NULL);
    g_return_val_if_fail (op_id != -1, NULL);

    self = g_object_new (QTMM_TYPE_SEARCH, NULL);

    priv = GET_PRIVATE (self);
    priv->account      = g_object_ref (account);
    priv->query        = tny_folder_store_query_new ();
    tny_folder_store_query_add_item (
        priv->query, NULL, TNY_FOLDER_STORE_QUERY_OPTION_SUBSCRIBED);
    if (folders) {
        priv->folder_names = g_strdupv ((gchar **)folders);
    }
    priv->begin_time   = begin_time;
    priv->end_time     = end_time;
    priv->sync         = sync;
    priv->op_id        = op_id;

    return self;
}

/* Start the search, all results are signalled async */
gboolean
qtmm_search_start (QtmMSearch *self)
{
    QtmMSearchPrivate *priv;
    TnyList           *folders;

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    if (priv->status != QTMM_SEARCH_IDLE) {
        QTMM_DEBUG ("Can't start as the state is not idle %d", priv->status);
        return FALSE;
    }

    if (!priv->account) {
        priv->status = QTMM_SEARCH_CANCELLED;
        QTMM_DEBUG ("No account!");
        return FALSE;
    }

    QTMM_DEBUG ("Listing folders.");

    priv->status = QTMM_SEARCH_RUNNING;

    folders = tny_simple_list_new ();

    priv->folder_calls++;

    tny_folder_store_get_folders_async (
        TNY_FOLDER_STORE (priv->account),
        folders,
        priv->query,
        priv->sync,
        get_folders_reply,
        folder_status_cb,
        self);

    return TRUE;
}

/* Get the status of the search op */
QtmmSearchState
qtmm_search_get_status (QtmMSearch *search)
{
    QtmMSearchPrivate *priv;

    g_return_val_if_fail (search != NULL, QTMM_SEARCH_CANCELLED);

    priv = GET_PRIVATE (search);

    if (!priv || priv->disposed) {
        return QTMM_SEARCH_CANCELLED;
    }

    QTMM_DEBUG ("-> %d", priv->status);

    return priv->status;
}

/* Get the operation id of the search */
gint
qtmm_search_get_id (QtmMSearch *search)
{
    QtmMSearchPrivate *priv;

    g_return_val_if_fail (search != NULL, -1);

    priv = GET_PRIVATE (search);

    QTMM_DEBUG ("-> %d", priv->op_id);

    return priv->op_id;
}

/*
 * Private functions
 */

/* Reply to getting folders */
static void
get_folders_reply (TnyFolderStore *store,
                   gboolean        cancelled,
                   TnyList        *folders,
                   GError         *error,
                   gpointer        user_data)
{
    QtmMSearchPrivate *priv;
    TnyIterator *iter;
    GObject *folder = NULL;

    QTMM_DEBUG ();

    /* XXX: Potentially already disposed, mid call */
    if (!QTMM_IS_SEARCH (user_data)) return;

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    if (error) {
        QTMM_DEBUG_ERR (error, "Failed to list folders");
        goto DONE;
    }

    if (cancelled) {
        QTMM_DEBUG ("Folder listing was cancelled");
        goto DONE;
    }

    for (iter = tny_list_create_iterator (folders);
         !tny_iterator_is_done (iter);
         tny_iterator_next (iter)) {
        GObject *folder_search = NULL;

        folder = tny_iterator_get_current (iter);

        if (!folder) continue;

        if (TNY_IS_FOLDER_STORE (folder)) {
            TnyList *sub_folders = tny_simple_list_new ();
            priv->folder_calls++;

            QTMM_DEBUG ("Listing subfolders");

            tny_folder_store_get_folders_async (
                TNY_FOLDER_STORE (folder),
                sub_folders,
                priv->query,
                priv->sync,
                get_folders_reply,
                folder_status_cb,
                user_data);
        }

        if (priv->folder_names && *priv->folder_names) {
            gboolean folder_ok = FALSE;
            gchar **folder_name = priv->folder_names;
            const gchar *folder_id = tny_folder_get_id (TNY_FOLDER (folder));

            while (*folder_name) {
                if (!g_strcmp0 (folder_id, *folder_name)) {
                    QTMM_DEBUG ("Folder \"%s\" matching filter",
                                folder_id);
                    folder_ok = TRUE;
                    break;
                }
                folder_name++;
            }

            if (!folder_ok) {
                QTMM_DEBUG ("Skipping folder \"%s\" from results.",
                            folder_id);
                g_object_unref (folder);
                folder = NULL;
                continue;
            }
        }

        folder_search = qtmm_folder_search_new (
            TNY_FOLDER (folder),
            priv->begin_time,
            priv->end_time,
            priv->sync);

        g_signal_connect (
            folder_search, "headers-received",
            G_CALLBACK (headers_received_cb), user_data);
        g_signal_connect (
            folder_search, "done",
            G_CALLBACK (search_done_cb), user_data);

        priv->search_operations = g_slist_append (
            priv->search_operations, folder_search);

        g_object_unref (folder);
        folder = NULL;
    }
    g_object_unref (iter);

DONE:
    priv->folder_calls--;

    QTMM_DEBUG ("Folder calls %d", priv->folder_calls);

    if (priv->folder_calls <= 0) {
        if (priv->search_operations) {
            qtmm_folder_search_start (
                QTMM_FOLDER_SEARCH (priv->search_operations->data));
        } else {
            QTMM_DEBUG ("Didn't find any folders for the account with "
                        "given search parameters.");

            priv->status = QTMM_SEARCH_DONE;

            /* Ref and unref the search object, as it's likely the caller will
             * dispose us in the signal
             */
            g_object_ref (user_data);
            g_signal_emit (user_data, signals[SIG_DONE], 0);
            g_object_unref (user_data);
        }
    }

    g_object_unref (folders);
}

/* Status update for folder query */
static void
folder_status_cb (GObject   *store,
                  TnyStatus *status,
                  gpointer   user_data)
{
}

/* Reply for listing headers */
static void
headers_received_cb (GObject         *search,
                     const gchar     *folder_id,
                     const GPtrArray *headers,
                     gpointer         user_data)
{
    QtmMSearchPrivate *priv;

    QTMM_DEBUG();

    if (!QTMM_IS_SEARCH (user_data)) return;

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    g_signal_emit (user_data, signals[SIG_HEADERS_RECEIVED], 0,
                   priv->op_id, tny_account_get_id (priv->account),
                   folder_id, headers);
}

/* Status update for listing headers */
static void
search_done_cb (GObject  *search,
                gpointer  user_data)
{
    QtmMSearchPrivate *priv;

    QTMM_DEBUG();

    if (!QTMM_IS_SEARCH (user_data)) return;

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    priv->search_operations = g_slist_remove (
        priv->search_operations, search);

    if (priv->search_operations) {
        qtmm_folder_search_start (
            QTMM_FOLDER_SEARCH (priv->search_operations->data));
    } else {
        g_object_ref (user_data);
        g_signal_emit (user_data, signals[SIG_DONE], 0);
        g_object_unref (user_data);
    }

    g_object_unref (search);
}
