/* Copyright (c) 2010, Nokia Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * * Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * * Neither the name of the Nokia Corporation nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Errno */
#include <errno.h>

/* For open() */
#include <sys/stat.h>
#include <fcntl.h>

#include <stdlib.h>

/* GIO */
#include <gio/gio.h>

/* D-BUS */
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

/* Modest plugin headers */
#include <modest-plugin.h>
#include <modest-platform.h>
#include <modest-account-mgr.h>
#include <modest-tny-account-store.h>
#include <modest-protocol-registry.h>
#include <modest-mail-operation-queue.h>
#include <modest-plugin-ui-actions.h>

/* Tinymail */
#include <tny-combined-account.h>
#include <tny-simple-list.h>
#include <tny-header.h>
#include <tny-fs-stream.h>

#include <tny-camel-mime-part.h>
#include <tny-camel-msg.h>
#include <tny-camel-mem-stream.h>

/* Own includes */
#include "qtmm-dbus-service.h"
#include "qtmm-debug.h"
#include "qtmm-defs.h"
#include "qtmm-errors.h"
#include "qtmm-search.h"
#include "qtmm-marshal.h"

/* The generated D-BUS service header. Must come after object header */
#include "com-nokia-Qtm-Modest-Plugin-glue.h"

/* Signals */
enum {
    SIG_HEADERS_RECEIVED = 0,
    SIG_HEADERS_FETCHED,
    SIGNALS
};
static gulong signals[SIGNALS] = {0};

/* Private data definition */
struct _QtmMDBusServicePrivate {
    gboolean disposed;

    /* Modest utilities */
    ModestAccountMgr         *acc_manager;
    ModestTnyAccountStore    *acc_store;
    ModestProtocolRegistry   *protocol_registry;
    ModestMailOperationQueue *mail_queue;

    /* List of currently running operations. Not expecting too many
     * of them running at the same time, so GSList should be fast enough
     */
    GSList *search_ops;

    /* A running id of op ids */
    guint last_op_id;
};

/* Context for saving to drafts */
typedef struct {
    /* Self instance */
    QtmMDBusService       *self;

    /* Destination folder we'll transfer this message to */
    TnyFolder             *folder;

    /* D-BUS method invocation context to return values to client */
    DBusGMethodInvocation *method;
} QtmMSaveDraftContext;

#define GET_PRIVATE(o) ((QtmMDBusServicePrivate *)      \
                        ((QtmMDBusService *)(o))->priv)

G_DEFINE_TYPE (QtmMDBusService, qtmm_dbus_service, G_TYPE_OBJECT);

/*
 * Private function prototypes
 */

/* Finds a draft message based on an id */
static TnyMsg *
find_draft_message (QtmMDBusService *self,
                    const gchar *account_name,
                    const gchar *draft_id);

/* Finds a transport account from account store, based on account name */
static TnyAccount *
find_transport_account (QtmMDBusService *self,
                        const gchar *account_name);

/* Translates attachment list from list of maps to TnyMimePart list */
static GList *
build_attachments (QtmMDBusService *self,
                   const GPtrArray *inAttachments);

/* Translates inputted priority into Tny header flags */
static TnyHeaderFlags
build_priority (QtmMDBusService *self,
                const guint inPriority);

/* Translates inputted header fields into Tny header fields */
static void
build_headers (QtmMDBusService  *self,
               GHashTable       *inHeaders,
               TnyList         **header_list);

/* Decides an encoding for an attachment */
static const gchar *
decide_encoding (const gchar *mime_type);

/* Starts a queued search */
static void
start_search (QtmMDBusService *self);

/* Headers found in a search */
static void
headers_received_cb (QtmMSearch      *search,
                     gint             operation_id,
                     const gchar     *account_id,
                     const gchar     *folder_id,
                     const GPtrArray *headers,
                     gpointer         user_data);

/* Search is done */
static void
search_done_cb (QtmMSearch *search,
                gpointer    user_data);

/* Finds a body parent */
static TnyMimePart *
find_body_parent_recursive (TnyMimePart *part);

/* Finds a specific mime part from a message, based on an id */
static TnyMimePart *
find_mime_part (TnyMimePart *part,
                const gchar *mime_part_id);

/* Saves the mime part into a temporary file */
static gchar *
save_mime_part (
    TnyMimePart *mime_part,
    const gchar *account_id,
    const gchar *folder_id,
    const gchar *message_id,
    const gchar *mime_part_id,
    gint *size);

/*
 * GObject functions
 */

/* Instance init */
static void
qtmm_dbus_service_init (QtmMDBusService *service)
{
    QtmMDBusServicePrivate *priv;

    QTMM_DEBUG();

    priv = service->priv = G_TYPE_INSTANCE_GET_PRIVATE (
        service, QTMM_TYPE_DBUS_SERVICE, QtmMDBusServicePrivate);

    priv->acc_manager = modest_plugin_get_account_mgr ();

    if (!priv->acc_manager)
        QTMM_DEBUG ("Couldn't get account manager");

    priv->acc_store = modest_plugin_get_account_store ();

    if (!priv->acc_store)
        QTMM_DEBUG ("Couldn't get account store");

    priv->protocol_registry = modest_plugin_get_protocol_registry ();

    if (!priv->protocol_registry)
        QTMM_DEBUG ("Couldn't get protocol registry");

    priv->mail_queue = modest_plugin_get_mail_operation_queue ();

    if (!priv->mail_queue)
        QTMM_DEBUG ("Couldn't get mail queue");
}

/* Object constructed */
static void
qtmm_dbus_service_constructed (GObject *service)
{
    QtmMDBusServicePrivate *priv;
    DBusGConnection *connection = NULL;
    DBusConnection *bus = NULL;

    QTMM_DEBUG();

    priv = GET_PRIVATE (service);

    bus = modest_platform_get_dbus_connection ();

    if (!bus) {
        QTMM_DEBUG ("Failed to get Bus connection!");
        return;
    }
    connection = dbus_connection_get_g_connection (bus);

    dbus_g_connection_register_g_object (
        connection, QTMM_DBUS_OBJ_PATH, service);

    if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER !=
        dbus_bus_request_name (
            bus, QTMM_DBUS_COMMON_NAME,
            DBUS_NAME_FLAG_DO_NOT_QUEUE |
            DBUS_NAME_FLAG_REPLACE_EXISTING,
            NULL)) {
        QTMM_DEBUG ("Failed to acquire name %s!", QTMM_DBUS_COMMON_NAME);
    }
}

/* Instance disposal, unref all other GObjects here */
static void
qtmm_dbus_service_dispose (GObject *object)
{
    QtmMDBusServicePrivate *priv;

    QTMM_DEBUG();

    priv = GET_PRIVATE (object);

    if (priv == NULL) return;

    if (priv->disposed) {
        QTMM_DEBUG ("Already disposed");
        return;
    }

    priv->disposed = TRUE;

    /* TODO:
     * Disconnect any signals connected to the singletons
     */
    if (priv->acc_manager) {
        priv->acc_manager = NULL;
    }
    if (priv->acc_store) {
        priv->acc_store = NULL;
    }
    if (priv->protocol_registry) {
        priv->protocol_registry = NULL;
    }
    if (priv->mail_queue) {
        priv->mail_queue = NULL;
    }

    G_OBJECT_CLASS (qtmm_dbus_service_parent_class)->dispose (object);
}

/* Class (type) init */
static void
qtmm_dbus_service_class_init (QtmMDBusServiceClass *klass)
{
    GObjectClass *object_class;
    DBusGConnection *connection = NULL;

    QTMM_DEBUG();

    object_class = G_OBJECT_CLASS (klass);

    object_class->dispose     = qtmm_dbus_service_dispose;
    object_class->constructed = qtmm_dbus_service_constructed;

    /* HACK to get specialized types initted */
    connection = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
    if (connection) dbus_g_connection_unref (connection);

    /* Signals */
    signals[SIG_HEADERS_RECEIVED] = g_signal_new (
        "headers-received",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        qtmm_marshal_VOID__INT_STRING_STRING_BOXED,
        G_TYPE_NONE, 4,
        G_TYPE_INT,
        G_TYPE_STRING,
        G_TYPE_STRING,
        qtmm_header_info_get_type ());

    signals[SIG_HEADERS_FETCHED] = g_signal_new (
        "headers-fetched",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST,
        0, NULL, NULL,
        g_cclosure_marshal_VOID__INT,
        G_TYPE_NONE, 1,
        G_TYPE_INT);

    g_type_class_add_private (klass, sizeof (QtmMDBusServicePrivate));

    dbus_g_object_type_install_info (
        QTMM_TYPE_DBUS_SERVICE,
        &dbus_glib_com_nokia_Qtm_Modest_Plugin_object_info);
}

/*
 * Public functions
 */

/* Creates a new instance */
GObject *
qtmm_dbus_service_new ()
{
    GObject *service;

    QTMM_DEBUG();

    service = g_object_new (QTMM_TYPE_DBUS_SERVICE, NULL);

    return service;
}

gboolean
com_nokia_Qtm_Modest_Plugin_add_folder (
    QtmMDBusService *self,
    const gchar *inAccountId,
    const gchar *inFolderName,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    TnyStoreAccount *account = NULL;
    gchar *account_store_id = NULL;
    TnyFolder *folder = NULL;
    GError *err = NULL;

    QTMM_DEBUG ();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    QTMM_DEBUG ("Creating folder : %s", inFolderName);

    if (g_str_equal (inAccountId, "local_folders")) {
        account_store_id = g_strdup (inAccountId);
    } else {
        account_store_id = g_strconcat (inAccountId, "_store", NULL);
    }

    // Search account
    TnyList *accounts = tny_simple_list_new ();
    tny_account_store_get_accounts (TNY_ACCOUNT_STORE (priv->acc_store),
                                    accounts,
                                    TNY_ACCOUNT_STORE_STORE_ACCOUNTS);

    TnyIterator *account_iter = tny_list_create_iterator (accounts);
    while (!tny_iterator_is_done (account_iter)) {
        account = (TNY_STORE_ACCOUNT (tny_iterator_get_current (account_iter)));
        if (strcmp (tny_account_get_id (TNY_ACCOUNT (account)), account_store_id) == 0) {
            break;
        } else {
            g_object_unref (account);
            account = NULL;
        }
        tny_iterator_next (account_iter);
    }
    g_object_unref (account_iter);
    g_object_unref (accounts);
    g_free (account_store_id);

    if (account) {
        folder = tny_folder_store_create_folder (TNY_FOLDER_STORE (account), inFolderName, &err);
    }

    if (err) {
        QTMM_DEBUG ("Folder create failed : %s", err->message);
        dbus_g_method_return (context, "");
        g_error_free (err);
        err = NULL;
    } else {
        if (folder) {
            QTMM_DEBUG ("Folder was created successfully");
            dbus_g_method_return (context, tny_folder_get_id (folder));
        } else {
            QTMM_DEBUG ("Folder create failed. Account not found!");
            dbus_g_method_return (context, "");
        }
    }

    if (account)
        g_object_unref (account);

    if (folder)
        g_object_unref (folder);

    return TRUE;
}

gboolean
com_nokia_Qtm_Modest_Plugin_remove_folder (
    QtmMDBusService *self,
    const gchar *inAccountId,
    const gchar *inFolderId,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    TnyStoreAccount *account = NULL;
    gchar *account_store_id = NULL;
    TnyFolder *folder = NULL;
    GError *err = NULL;

    QTMM_DEBUG ();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    QTMM_DEBUG ("Removing folder : %s", inFolderId);

    if (g_str_equal (inAccountId, "local_folders")) {
        account_store_id = g_strdup (inAccountId);
    } else {
        account_store_id = g_strconcat (inAccountId, "_store", NULL);
    }

    // Search account
    TnyList *accounts = tny_simple_list_new ();
    tny_account_store_get_accounts (TNY_ACCOUNT_STORE (priv->acc_store),
                                    accounts,
                                    TNY_ACCOUNT_STORE_STORE_ACCOUNTS);

    TnyIterator *account_iter = tny_list_create_iterator (accounts);
    while (!tny_iterator_is_done (account_iter)) {
        account = (TNY_STORE_ACCOUNT (tny_iterator_get_current (account_iter)));
        if (strcmp (tny_account_get_id (TNY_ACCOUNT (account)), account_store_id) == 0) {
            break;
        } else {
            g_object_unref (account);
            account = NULL;
        }
        tny_iterator_next (account_iter);
    }
    g_object_unref (account_iter);
    g_object_unref (accounts);
    g_free (account_store_id);

   if (account) {
        // Search folder
        TnyList *folders = tny_simple_list_new ();
        tny_folder_store_get_folders (TNY_FOLDER_STORE (account), folders, NULL, FALSE, NULL);
        TnyIterator *folder_iter = tny_list_create_iterator (folders);
        while (!tny_iterator_is_done (folder_iter)) {
            folder = TNY_FOLDER (tny_iterator_get_current (folder_iter));
            if (strcmp (tny_folder_get_id (folder), inFolderId) == 0) {
                break;
            } else {
              g_object_unref (folder);
              folder = NULL;
            }
            tny_iterator_next (folder_iter);
        }
        g_object_unref (folder_iter);
        g_object_unref (folders);
    }

    if (folder) {
        tny_folder_store_remove_folder (TNY_FOLDER_STORE (account), folder, &err);
    }

    if (err) {
        QTMM_DEBUG ("Folder remove failed : %s", err->message);
        g_error_free (err);
        err = NULL;
        dbus_g_method_return (context, FALSE);
    } else {
        if (!account) {
            QTMM_DEBUG ("Folder remove failed. Account not found!");
            dbus_g_method_return (context, FALSE);
        } else if (!folder) {
            QTMM_DEBUG ("Folder remove failed. Folder not found!");
            dbus_g_method_return (context, FALSE);
        } else {
            QTMM_DEBUG ("Folder was removed successfully");
            dbus_g_method_return (context, TRUE);
            // "Couldn't find weak ref error" is shown,
            // if removed folder is tried to unref.
            // => folder can not be unreffed.
            folder = NULL;
        }
    }

    if (account)
        g_object_unref (account);

    if (folder)
        g_object_unref (folder);

    return TRUE;
}

#define QTM_EMAIL_MIMEPARTS_DBUS_TYPE \
    DBUS_STRUCT_BEGIN_CHAR_AS_STRING \
    DBUS_TYPE_STRING_AS_STRING /* MIME type */ \
    DBUS_TYPE_BOOLEAN_AS_STRING /* Is attachment */ \
    DBUS_TYPE_STRING_AS_STRING /* File name */ \
    DBUS_TYPE_STRING_AS_STRING /* Content id */ \
    DBUS_STRUCT_END_CHAR_AS_STRING

typedef struct {
    DBusGMethodInvocation *context;
    TnyFolder *folder;
    gchar *message_id;
} GetMessageHelper;

static void get_message_cb (TnyFolder *self,
                            gboolean cancelled,
                            TnyList *headers,
                            GError *err,
                            GetMessageHelper *helper)
{
    TnyHeader *header = NULL;
    TnyMsg *msg = NULL;

    const gchar *empty_string = "";
    const gchar *text_plain_string = "text/plain";

    // Return values
    gchar *url = NULL;
    gchar *mime_type = NULL;
    gchar *from = NULL;
    gchar *to = NULL;
    gchar *cc = NULL;
    gchar *bcc = NULL;
    gchar *replyto = NULL;
    gchar *subject = NULL;
    time_t date_received = 0;
    time_t date_sent = 0;
    guint message_size = 0;
    guint flags = 0;
    guint priority = 0;

    QTMM_DEBUG();

    if (err || cancelled) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "Failed to find a message"};

        dbus_g_method_return_error (helper->context, &error);

        g_object_unref (helper->folder);
        g_free(helper->message_id);
        g_slice_free (GetMessageHelper, helper);

        return;
    }

    TnyIterator *header_iter = tny_list_create_iterator (headers);
    while (!tny_iterator_is_done (header_iter)) {
        gchar* header_id = NULL;
        header = TNY_HEADER (tny_iterator_get_current (header_iter));
        if (header) {
            header_id = tny_header_dup_uid (header);
            if (header_id && helper->message_id && g_str_equal (header_id, helper->message_id)) {
                g_free (header_id);
                break;
            } else {
                g_free (header_id);
                g_object_unref (header);
                header = NULL;
            }
        }
        tny_iterator_next (header_iter);
    }
    g_object_unref (header_iter);

    if (header) {
        from = tny_header_dup_from (header);
        if (!from) {
            from = g_strdup (empty_string);
        }
        to = tny_header_dup_to (header);
        if (!to) {
            to = g_strdup (empty_string);
        }
        cc = tny_header_dup_cc (header);
        if (!cc) {
            cc = g_strdup (empty_string);
        }
        bcc = tny_header_dup_bcc (header);
        if (!bcc) {
            bcc = g_strdup (empty_string);
        }
        replyto = tny_header_dup_replyto (header);
        if (!replyto) {
            replyto = g_strdup (empty_string);
        }
        subject = tny_header_dup_subject (header);
        if (!subject) {
            subject = g_strdup (empty_string);
        }
        date_received = tny_header_get_date_received (header);
        date_sent     = tny_header_get_date_sent (header);
        message_size  = tny_header_get_message_size (header);
        flags         = tny_header_get_flags (header);
        priority      = tny_header_get_priority (header);

        gchar *folder_url = tny_folder_get_url_string (helper->folder);
        url = g_strconcat(folder_url, "/", helper->message_id, NULL);
        g_free (folder_url);

        if (tny_header_get_flags (header) & TNY_HEADER_FLAG_CACHED) {
            // Search message
            // Note: If message content is NOT already in file cache,
            //       message will be tried to retrieve from server
            msg = tny_folder_find_msg (helper->folder, url, NULL);
        }
    } else {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "Failed to find a message"};

        dbus_g_method_return_error (helper->context, &error);

        g_object_unref (helper->folder);
        g_free(helper->message_id);
        g_slice_free (GetMessageHelper, helper);

        return;
    }

    gboolean has_attachments = FALSE;

    if (msg) {
        const gchar *msg_mime_type = tny_mime_part_get_content_type (TNY_MIME_PART (msg));
        if (msg_mime_type) {
            mime_type = g_strdup (msg_mime_type);
            if (!g_strcmp0 (mime_type, "multipart/mixed")) {
                has_attachments = TRUE;
            }
        } else {
            mime_type = g_strdup (text_plain_string);
        }
    } else {
        mime_type = g_strdup (empty_string);
    }

    if (header)
        g_object_unref (header);

    DBusMessage *reply = dbus_g_method_get_reply (helper->context);
    DBusMessageIter iter;
    dbus_message_iter_init_append (reply, &iter);

    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &url);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &mime_type);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &from);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &to);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &cc);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &bcc);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &replyto);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_STRING, &subject);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &date_received);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &date_sent);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &message_size);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &flags);
    dbus_message_iter_append_basic (&iter, DBUS_TYPE_UINT32, &priority);

    DBusMessageIter array_iter;
    dbus_message_iter_open_container (&iter,
                                      DBUS_TYPE_ARRAY,
                                      QTM_EMAIL_MIMEPARTS_DBUS_TYPE,
                                      &array_iter);

    if (msg) {
        if (has_attachments) {
            gint attachment_counter = 1;
            TnyList *child_parts = tny_simple_list_new ();
            tny_mime_part_get_parts (TNY_MIME_PART (msg), child_parts);
            TnyIterator *child_part_iter = tny_list_create_iterator (child_parts);
            while (!tny_iterator_is_done (child_part_iter)) {
                TnyMimePart *childMimePart = TNY_MIME_PART (tny_iterator_get_current(child_part_iter));
                DBusMessageIter struct_iter;

                // Open container
                dbus_message_iter_open_container (&array_iter,
                                                  DBUS_TYPE_STRUCT,
                                                  NULL,
                                                  &struct_iter);

                const gchar *attachment_mime_type = tny_mime_part_get_content_type (childMimePart);
                if (attachment_mime_type) {
                    dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &attachment_mime_type);
                } else {
                    dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &text_plain_string);
                }

                gboolean is_attachment = tny_mime_part_is_attachment (childMimePart);
                dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_BOOLEAN, &is_attachment);

                const gchar *attachment_filename = tny_mime_part_get_filename (childMimePart);
                if (attachment_filename) {
                    dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &attachment_filename);
                } else {
                    dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &empty_string);
                }

                const gchar *content_id = tny_mime_part_get_content_id (
                    childMimePart);

                if (content_id) {
                    dbus_message_iter_append_basic (
                        &struct_iter, DBUS_TYPE_STRING, &content_id);
                } else {
                    gchar* temp_content_id = g_strdup_printf ("pos_%d", attachment_counter);
                    dbus_message_iter_append_basic (
                        &struct_iter, DBUS_TYPE_STRING, &temp_content_id);
                    g_free (temp_content_id);
                }

                // Close container
                dbus_message_iter_close_container (&array_iter, &struct_iter);

                tny_iterator_next (child_part_iter);

                g_object_unref (childMimePart);
                attachment_counter ++;
            }
            g_object_unref (child_part_iter);
            g_object_unref (child_parts);
        } else {
            TnyMimePart *msgMimePart = TNY_MIME_PART (msg);
            DBusMessageIter struct_iter;

            // Open container
            dbus_message_iter_open_container (&array_iter,
                                              DBUS_TYPE_STRUCT,
                                              NULL,
                                              &struct_iter);

            const gchar *attachment_mime_type = tny_mime_part_get_content_type (msgMimePart);
            if (attachment_mime_type) {
                dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &attachment_mime_type);
            } else {
                dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &text_plain_string);
            }

            gboolean is_attachment = FALSE;
            dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_BOOLEAN, &is_attachment);
            dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &empty_string);

            const gchar *content_id = tny_mime_part_get_content_id (msgMimePart);

            if (content_id) {
                dbus_message_iter_append_basic (&struct_iter, DBUS_TYPE_STRING, &content_id);
            } else {
                gchar* temp_content_id = g_strdup ("pos_msg");
                dbus_message_iter_append_basic (
                    &struct_iter, DBUS_TYPE_STRING, &temp_content_id);
                g_free (temp_content_id);
            }

            // Close container
            dbus_message_iter_close_container (&array_iter, &struct_iter);
        }
    }

    dbus_message_iter_close_container (&iter, &array_iter);

    if (msg)
        g_object_unref (msg);

    dbus_g_method_send_reply (helper->context, reply);

    g_free (url);
    g_free (mime_type);
    g_free (from);
    g_free (to);
    g_free (cc);
    g_free (bcc);
    g_free (replyto);
    g_free (subject);

    g_object_unref (helper->folder);
    g_free (helper->message_id);
    g_slice_free (GetMessageHelper, helper);
}


/* Get a "frame" for a single message */
gboolean
com_nokia_Qtm_Modest_Plugin_get_message (
    QtmMDBusService *self,
    const gchar *inAccountId,
    const gchar *inFolderId,
    const gchar *inMessageId,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    TnyAccount *account = NULL;
    TnyFolder *folder = NULL;

    QTMM_DEBUG();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    /* Get an account */
    account = modest_tny_account_store_get_server_account (
        priv->acc_store, inAccountId, TNY_ACCOUNT_TYPE_STORE);

    if (account) {
        // Search folder
        TnyList *folders = tny_simple_list_new ();
        tny_folder_store_get_folders (TNY_FOLDER_STORE (account), folders, NULL, FALSE, NULL);
        TnyIterator *folder_iter = tny_list_create_iterator (folders);
        while (!tny_iterator_is_done (folder_iter)) {
            folder = TNY_FOLDER (tny_iterator_get_current (folder_iter));
            if (strcmp (tny_folder_get_id (folder), inFolderId) == 0) {
                break;
            } else {
              g_object_unref (folder);
              folder = NULL;
            }
            tny_iterator_next (folder_iter);
        }
        g_object_unref (folder_iter);
        g_object_unref (folders);
    }

    if (folder) {
        // Search header
        TnyList *headers = tny_simple_list_new ();
        GetMessageHelper *helper = g_slice_new (GetMessageHelper);
        helper->context = context;
        helper->folder = folder;
        helper->message_id = g_strdup (inMessageId);

        QTMM_DEBUG("Starting to search message");
        tny_folder_get_headers_async (folder, headers, FALSE, (TnyGetHeadersCallback) get_message_cb, NULL, helper);
    } else {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "Failed to find a message"};

        dbus_g_method_return_error (context, &error);
    }

    if (account)
        g_object_unref (account);

    return TRUE;
}

/* Sends an email */
gboolean
com_nokia_Qtm_Modest_Plugin_send_email (
    QtmMDBusService *self,
    GHashTable      *inSenderInfo,
    GHashTable      *inRecipients,
    GHashTable      *inMessageData,
    const GPtrArray *inAttachments,
    const GPtrArray *inImages,
    guint            inPriority,
    GHashTable      *inHeaderFields,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    const gchar *account_name    = NULL;
    gchar       *default_account = NULL;
    const gchar *from            = NULL;
    const gchar *reply_to        = NULL;
    const gchar *to              = NULL;
    const gchar *cc              = NULL;
    const gchar *bcc             = NULL;
    const gchar *draft_id        = NULL;
    TnyMsg      *draft_msg       = NULL;
    TnyAccount  *trans_account   = NULL;
    const gchar *subject         = NULL;
    const gchar *plain_body      = NULL;
    const gchar *html_body       = NULL;
    const gchar *references      = NULL;
    GList       *attachment_list = NULL;
    GList       *image_list      = NULL;
    TnyList     *custom_headers  = NULL;
    guint        priority        = TNY_HEADER_FLAG_NORMAL_PRIORITY;
    gboolean     sending         = FALSE;

    QTMM_DEBUG();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    if (inSenderInfo) {
        account_name = g_hash_table_lookup (inSenderInfo, "account-name");
        from         = g_hash_table_lookup (inSenderInfo, "from");
        reply_to     = g_hash_table_lookup (inSenderInfo, "reply-to");
    }

    if (inRecipients) {
        to  = g_hash_table_lookup (inRecipients, "to");
        cc  = g_hash_table_lookup (inRecipients, "cc");
        bcc = g_hash_table_lookup (inRecipients, "bcc");
    }

    /* Not enough info to send */
    if (!to && !cc && !bcc) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_NO_RECIPIENTS,
            "No to, cc or bcc field set (don't know where to send)"};

        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    if (!account_name && priv->acc_manager) {
        default_account = modest_account_mgr_get_default_account (
            priv->acc_manager);
        if (default_account) account_name = default_account;
    }

    if (!account_name) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_ACCOUNT,
            "No account-name given in!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    trans_account = find_transport_account (self, account_name);

    if (inMessageData) {
        draft_id   = g_hash_table_lookup (inMessageData, "draft-id");
        subject    = g_hash_table_lookup (inMessageData, "subject");
        plain_body = g_hash_table_lookup (inMessageData, "plain-body");
        html_body  = g_hash_table_lookup (inMessageData, "html-body");
        references = g_hash_table_lookup (inMessageData, "references");
    }

    /* XXX:
     * - Doesn't work yet :(
     */
    if (draft_id) {
        trans_account = find_transport_account (self, account_name);
        if (trans_account)
            draft_msg = find_draft_message (self, account_name, draft_id);
    }

    if (inAttachments) {
        attachment_list = build_attachments (self, inAttachments);
    }

    if (inImages) {
        image_list = build_attachments (self, inImages);
    }

    if (inPriority) {
        priority = build_priority (self, inPriority);
    }

    custom_headers = tny_simple_list_new ();
    if (inHeaderFields) {
        build_headers (self, inHeaderFields, &custom_headers);
    }

    if (!trans_account) {
        sending = modest_ui_actions_on_send_custom_msg (
            account_name, from, to, cc, bcc, subject, plain_body, html_body,
            attachment_list, image_list, references, reply_to, priority,
            custom_headers);
    } else {
        sending = modest_ui_actions_send_msg_with_transport (
            TNY_TRANSPORT_ACCOUNT (trans_account), draft_msg, from, to, cc,
            bcc, subject, plain_body, html_body, attachment_list, image_list,
            references, reply_to, priority, custom_headers);
    }

    g_free (default_account);
    default_account = NULL;

    if (trans_account) {
        g_object_unref (trans_account);
        trans_account = NULL;
    }
    if (draft_msg) {
        g_object_unref (draft_msg);
        draft_msg = NULL;
    }
    if (attachment_list) {
        g_list_foreach (attachment_list, (GFunc)g_object_unref, NULL);
        g_list_free (attachment_list);
        attachment_list = NULL;
    }
    if (image_list) {
        g_list_foreach (image_list, (GFunc)g_object_unref, NULL);
        g_list_free (image_list);
        image_list = NULL;
    }
    if (custom_headers) {
        g_object_unref (custom_headers);
        custom_headers = NULL;
    }

    if (sending) {
        dbus_g_method_return (context);
    } else {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_UNKNOWN,
            "Failed to send message"};

        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    return TRUE;
}

/* Starts listing headers */
gboolean
com_nokia_Qtm_Modest_Plugin_get_headers (
    QtmMDBusService  *self,
    const gchar     **inAccountIds,
    const gchar     **inFolders,
    guint64           inBeginTime,
    guint64           inEndTime,
    gboolean          inSync,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    TnyList     *accounts = NULL;
    TnyIterator *a_iter   = NULL;
    int          out_id   = -1;

    QTMM_DEBUG ("Begin time: %llu, End time: %llu, Sync: %s, time in "
                "seconds: %llu", inBeginTime, inEndTime,
                inSync == TRUE ? "TRUE" : "FALSE", (guint64)time (NULL));

    priv = GET_PRIVATE (self);

    if (!priv->acc_store) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_UNKNOWN,
            "Internal account store is not available!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    accounts = tny_simple_list_new ();

    /* List all accounts */
    if (!inAccountIds || *inAccountIds == NULL) {
        tny_account_store_get_accounts (
            TNY_ACCOUNT_STORE (priv->acc_store),
            accounts,
            TNY_ACCOUNT_STORE_STORE_ACCOUNTS);
    } else {
        const gchar **name = inAccountIds;

        while (name && *name) {
            TnyAccount *account = NULL;

            account = modest_tny_account_store_get_server_account (
                priv->acc_store, *name, TNY_ACCOUNT_TYPE_STORE);

            if (account) {
                QTMM_DEBUG ("Found account %s", *name);
                tny_list_append (accounts, G_OBJECT (account));
                g_object_unref (account);
            }

            name++;
        }
    }

    /* Can't get accounts, can't really search anything */
    if (tny_list_get_length (accounts) == 0) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_ACCOUNT,
            "Couldn't find any accounts!"};

        QTMM_DEBUG ("Didn't find any accounts");

        dbus_g_method_return_error (context, &error);
        g_object_unref (accounts);
        return FALSE;
    }

    out_id = ++priv->last_op_id;

    for (a_iter = tny_list_create_iterator (accounts);
         !tny_iterator_is_done (a_iter);
         tny_iterator_next (a_iter)) {
        GObject *search  = NULL;
        GObject *account = NULL;

        account = tny_iterator_get_current (a_iter);

        if (!account) continue;

        search = qtmm_search_new (
            TNY_ACCOUNT (account), inFolders, inBeginTime, inEndTime, inSync,
            out_id);

        /* Typically an error should be shown? */
        if (!search) {
            g_object_unref (account);
            account = NULL;
            continue;
        }

        /* TODO:
         * - Connect signals
         * - In callback, need to remove the search from internal list and
         *   begin a new one
         * - Should we care if we already have the search in the list?
         */
        priv->search_ops = g_slist_append (priv->search_ops, search);

        g_signal_connect (
            search, "headers-received",
            G_CALLBACK (headers_received_cb), self);
        g_signal_connect (
            search, "done",
            G_CALLBACK (search_done_cb), self);

        g_object_unref (account);
        account = NULL;
    }

    /* This will check if we can even start any searches. If a search is
     * already going, this will do nothing.
     */
    start_search (self);

    if (a_iter) {
        g_object_unref (a_iter);
        a_iter = NULL;
    }

    if (accounts) {
        g_object_unref (accounts);
        accounts = NULL;
    }

    dbus_g_method_return (context, out_id);
    return TRUE;
}

/* Adds a message into a folder */
gboolean
com_nokia_Qtm_Modest_Plugin_add_message (
    QtmMDBusService *self,
    const gchar     *inFolderId,
    GHashTable      *inSenderInfo,
    GHashTable      *inRecipients,
    GHashTable      *inMessageData,
    const GPtrArray *inAttachments,
    const GPtrArray *inImages,
    guint            inPriority,
    GHashTable      *inHeaderFields,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    const gchar *account_name    = NULL;
    gchar       *default_account = NULL;
    const gchar *from            = NULL;
    const gchar *reply_to        = NULL;
    const gchar *to              = NULL;
    const gchar *cc              = NULL;
    const gchar *bcc             = NULL;
    const gchar *draft_id        = NULL;
    TnyMsg      *draft_msg       = NULL;
    const gchar *subject         = NULL;
    const gchar *plain_body      = NULL;
    const gchar *html_body       = NULL;
    const gchar *references      = NULL;
    GList       *attachment_list = NULL;
    GList       *image_list      = NULL;
    TnyList     *custom_headers  = NULL;
    guint        priority        = TNY_HEADER_FLAG_NORMAL_PRIORITY;
    gchar       *message_id      = NULL;

    TnyMsg      *message         = NULL;
    TnyMimePart *body_part       = NULL;

    TnyStoreAccount  *store_account   = NULL;
    TnyFolder        *folder          = NULL;


    QTMM_DEBUG();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    if (inSenderInfo) {
        account_name = g_hash_table_lookup (inSenderInfo, "account-name");
        from         = g_hash_table_lookup (inSenderInfo, "from");
        reply_to     = g_hash_table_lookup (inSenderInfo, "reply-to");
    }

    if (inRecipients) {
        to  = g_hash_table_lookup (inRecipients, "to");
        cc  = g_hash_table_lookup (inRecipients, "cc");
        bcc = g_hash_table_lookup (inRecipients, "bcc");
    }

    if (!account_name && priv->acc_manager) {
        default_account = modest_account_mgr_get_default_account (priv->acc_manager);
        if (default_account) {
            account_name = default_account;
        }
    }
    QTMM_DEBUG ("Default account : %s", account_name);


    if (!account_name) {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_ACCOUNT,
            "No account-name given in and no default account!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    //trans_account = find_transport_account (self, account_name);

    if (inMessageData) {
        draft_id   = g_hash_table_lookup (inMessageData, "draft-id");
        subject    = g_hash_table_lookup (inMessageData, "subject");
        plain_body = g_hash_table_lookup (inMessageData, "plain-body");
        html_body  = g_hash_table_lookup (inMessageData, "html-body");
        references = g_hash_table_lookup (inMessageData, "references");
    }

    /* XXX:
     * - Doesn't work yet :(
     */
    //if (draft_id) {
    //    trans_account = find_transport_account (self, account_name);
    //    if (trans_account)
    //        draft_msg = find_draft_message (self, account_name, draft_id);
    //}
    gchar *account_store_id = NULL;
    if (g_str_equal (account_name, "local_folders")) {
        account_store_id = g_strdup (account_name);
    } else {
        account_store_id = g_strconcat (account_name, "_store", NULL);
    }

    // Search account
    TnyList *accounts = tny_simple_list_new ();
    tny_account_store_get_accounts (TNY_ACCOUNT_STORE (priv->acc_store),
                                    accounts,
                                    TNY_ACCOUNT_STORE_STORE_ACCOUNTS);

    TnyIterator *account_iter = tny_list_create_iterator (accounts);
    while (!tny_iterator_is_done (account_iter)) {
        store_account = (TNY_STORE_ACCOUNT (tny_iterator_get_current (account_iter)));
        if (strcmp (tny_account_get_id (TNY_ACCOUNT (store_account)), account_store_id) == 0) {
            break;
        } else {
            g_object_unref (store_account);
            store_account = NULL;
        }
        tny_iterator_next (account_iter);
    }
    g_object_unref (account_iter);
    g_object_unref (accounts);
    g_free (account_store_id);

    if (store_account) {
        // Search folder
        TnyList *folders = tny_simple_list_new ();
        tny_folder_store_get_folders (TNY_FOLDER_STORE (store_account), folders, NULL, FALSE, NULL);
        TnyIterator *folder_iter = tny_list_create_iterator (folders);
        while (!tny_iterator_is_done (folder_iter)) {
            folder = TNY_FOLDER (tny_iterator_get_current (folder_iter));
            if (strcmp (tny_folder_get_id (folder), inFolderId) == 0) {
                break;
            } else {
              g_object_unref (folder);
              folder = NULL;
            }
            tny_iterator_next (folder_iter);
        }
        g_object_unref (folder_iter);
        g_object_unref (folders);
    }

    if (folder) {
        if (inAttachments) {
            attachment_list = build_attachments (self, inAttachments);
        }

        if (inImages) {
            image_list = build_attachments (self, inImages);
        }

        if (inPriority) {
            priority = build_priority (self, inPriority);
        }

        custom_headers = tny_simple_list_new ();
        if (inHeaderFields) {
            build_headers (self, inHeaderFields, &custom_headers);
        }

        /* Message formatting, creating a base object */
        message = tny_camel_msg_new ();

        if (attachment_list) {
            TnyMimePart *related = NULL;

            tny_mime_part_set_content_type (
                TNY_MIME_PART (message), "multipart/mixed");
            if (image_list) {
                related = tny_camel_mime_part_new ();
                tny_mime_part_set_content_type (related, "multipart/related");
                tny_mime_part_add_part (TNY_MIME_PART (message), related);
            } else {
                related = g_object_ref (message);
            }

            if (plain_body && html_body) {
                body_part = tny_camel_mime_part_new ();
                tny_mime_part_set_content_type (
                    body_part, "multipart/alternative");
                tny_mime_part_add_part (related, body_part);
                g_object_unref (body_part);
            }

            g_object_unref (related);
        } else if (image_list) {
            tny_mime_part_set_content_type (
                TNY_MIME_PART (message), "multipart/related");
            if (plain_body && html_body) {
                body_part = tny_camel_mime_part_new ();
                tny_mime_part_set_content_type (
                    body_part, "multipart/alternative");
                tny_mime_part_add_part (
                    TNY_MIME_PART (message), body_part);
                g_object_unref (body_part);
            }
        } else if (plain_body && html_body) {
            tny_mime_part_set_content_type (
                TNY_MIME_PART (message), "multipart/alternative");
        }

        /* New message creation, setting header data */
        TnyHeader *header;
        const gchar *content_type = NULL;
        //gint tmp_attached = 0;

        /*TnyIterator *iterator;
        iterator = tny_list_create_iterator (header_pairs);
        while (!tny_iterator_is_done (iterator)) {
            TnyPair *current = (TnyPair *) tny_iterator_get_current (iterator);
            tyn_mime_part_set_header_pair (message, tny_pair_get_name (current), tny_pair_get_value (current));
            g_object_unref (current);
            tny_iterator_next (iterator);
        }
        g_object_unref (iterator);*/


        header = tny_msg_get_header (message);

        if (from && from[0] != '\0') {
            tny_header_set_from (TNY_HEADER (header), from);
            tny_header_set_replyto (TNY_HEADER (header), from);
        }
        if (to && to[0] != '\0') {
            tny_header_set_to (TNY_HEADER (header), to);
        }
        if (cc && cc[0] != '\0') {
            tny_header_set_cc (TNY_HEADER (header), cc);
        }
        if (bcc && bcc[0] != '\0') {
            tny_header_set_bcc (TNY_HEADER (header), bcc);
        }
        if (subject && subject[0] != '\0') {
            tny_header_set_subject (TNY_HEADER (header), subject);
        }

        /* It's coming through D-BUS, so I doubt we need to check this... but
         */
        if (plain_body) {
            if (g_utf8_validate (plain_body, -1, NULL))
                content_type = "text/plain; charset=\"utf8\"";
            else
                content_type = "text/plain; charset=\"latin1\"";
        }

        tny_mime_part_set_header_pair (
            TNY_MIME_PART (message), "X-Mailer", PACKAGE " " VERSION);

        if (references)
            tny_mime_part_set_header_pair (
                TNY_MIME_PART (message), "References", references);

        if (reply_to)
            tny_mime_part_set_header_pair (
                TNY_MIME_PART (message), "In-Reply-To", reply_to);

        /* Add the body of the new mail */
        /* This is needed even if body is NULL or empty. */
        TnyMimePart *text_body_part = NULL;
        TnyStream *text_body_stream;

        /* Create the stream */
        text_body_stream = TNY_STREAM (
            tny_camel_mem_stream_new_with_buffer (
                plain_body, (plain_body ? strlen (plain_body) : 0)));

        /* Create body part */
        TnyMimePart *parent = find_body_parent_recursive (TNY_MIME_PART (message));

        if (parent != NULL) {
            text_body_part = tny_camel_mime_part_new ();
            tny_mime_part_add_part (TNY_MIME_PART (parent), text_body_part);
            g_object_unref (parent);
        } else {
            text_body_part = g_object_ref (message);
        }

        // Construct Plain Text Body Part
        tny_stream_reset (text_body_stream);
        tny_mime_part_construct (text_body_part,
                                 text_body_stream,
                                 content_type, "8bit");
        tny_stream_reset (text_body_stream);

        g_object_unref (G_OBJECT(text_body_part));

        g_object_unref (text_body_stream);

        // Add HTML body if HTML body exists
        if (html_body && html_body[0] != '\0') {
            TnyMimePart *html_body_part = NULL;
            TnyStream *html_body_stream;

            // Construct HTML Body Part
            html_body_stream = TNY_STREAM (
                tny_camel_mem_stream_new_with_buffer (
                    html_body, (html_body ? strlen (html_body) : 0)));

            TnyMimePart *html_parent = find_body_parent_recursive (TNY_MIME_PART (message));

            if (parent != NULL) {
                html_body_part = tny_camel_mime_part_new ();
                tny_mime_part_add_part (TNY_MIME_PART (html_parent), html_body_part);
                g_object_unref (html_parent);
            } else {
                html_body_part = g_object_ref (message);
            }

            tny_mime_part_construct (html_body_part,
                                     html_body_stream,
                                     "text/html; charset=utf-8",
                                     "7bit");
            tny_stream_reset (html_body_stream);

            g_object_unref (G_OBJECT(html_body_part));

            g_object_unref (html_body_stream);
        }

        // Add attachments
        if (attachment_list) {
            TnyMimePart *attachment_mime_part;
            GList *attachments_position;
            for (attachments_position = attachment_list; attachments_position; attachments_position = attachments_position->next) {
                attachment_mime_part = attachments_position->data;
                const gchar *filename = tny_mime_part_get_filename (attachment_mime_part);
                if (!filename) {
                    tny_mime_part_set_header_pair (attachment_mime_part,
                                                   "Content-Disposition",
                                                   "attachment");
                }
                tny_mime_part_set_transfer_encoding (TNY_MIME_PART (attachment_mime_part),
                                                     "base64");
                tny_mime_part_add_part (TNY_MIME_PART (message), attachment_mime_part);
            }
        }

        // Add images
        if (image_list) {
            TnyMimePart *attachment_mime_part;
            GList *attachments_position;

            for (attachments_position = attachment_list; attachments_position; attachments_position = attachments_position->next) {
                attachment_mime_part = attachments_position->data;
                tny_mime_part_set_header_pair (attachment_mime_part,
                                               "Content-Disposition",
                                               "inline");
                tny_mime_part_set_transfer_encoding (TNY_MIME_PART (attachment_mime_part),
                                                     "base64");
                tny_mime_part_add_part (TNY_MIME_PART (message), attachment_mime_part);
            }
        }

        GError *error = NULL;
        tny_folder_add_msg( folder, message, &error);

        TnyHeader *temp_header = tny_msg_get_header (message);
        message_id = tny_header_dup_uid (temp_header);
        g_object_unref (temp_header);


        if (error) {
            QTMM_DEBUG_ERR (error, "Failed to add message");
            g_clear_error (&error);
        }

        g_object_unref(header);
    }

    g_free (default_account);
    default_account = NULL;

    if (store_account) {
        g_object_unref (store_account);
        store_account = NULL;
    }
    if (folder) {
        g_object_unref (folder);
        folder = NULL;
    }
    if (draft_msg) {
        g_object_unref (draft_msg);
        draft_msg = NULL;
    }
    if (attachment_list) {
        g_list_foreach (attachment_list, (GFunc)g_object_unref, NULL);
        g_list_free (attachment_list);
        attachment_list = NULL;
    }
    if (image_list) {
        g_list_foreach (image_list, (GFunc)g_object_unref, NULL);
        g_list_free (image_list);
        image_list = NULL;
    }
    if (custom_headers) {
        g_object_unref (custom_headers);
        custom_headers = NULL;
    }

    if (message_id) {
        dbus_g_method_return (context, message_id);
        g_free (message_id);
    } else {
        dbus_g_method_return (context, "");
    }

    return TRUE;
}

/* Fetches attachment data for reading to client */
gboolean
com_nokia_Qtm_Modest_Plugin_get_mime_part (
    QtmMDBusService *self,
    const gchar *inAccountId,
    const gchar *inFolderId,
    const gchar *inMessageId,
    const gchar *inMimePartId,
    DBusGMethodInvocation *context)
{
    QtmMDBusServicePrivate *priv;
    TnyAccount  *account = NULL;
    TnyFolder   *folder  = NULL;
    TnyHeader   *header  = NULL;
    TnyMsg      *msg     = NULL;
    TnyMimePart *part    = NULL;

    QTMM_DEBUG();

    g_return_val_if_fail (QTMM_IS_DBUS_SERVICE (self), FALSE);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, FALSE);

    if (!inAccountId || inAccountId[0] == '\0') {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_ACCOUNT,
            "No account id given in!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    if (!inFolderId || inFolderId[0] == '\0') {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_FOLDER,
            "No folder id given in!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    if (!inMessageId || inMessageId[0] == '\0') {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "No message id given in!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    if (!inMimePartId || inMimePartId[0] == '\0') {
        GError error = {
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MIME_PART,
            "No mime part id given in!"};
        dbus_g_method_return_error (context, &error);
        return FALSE;
    }

    account = modest_tny_account_store_get_server_account (
        priv->acc_store, inAccountId, TNY_ACCOUNT_TYPE_STORE);

    if (!account) {
        GError *error = g_error_new (
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_ACCOUNT,
            "Failed to find account with id \"%s\"!",
            inAccountId);
        dbus_g_method_return_error (context, error);
        g_clear_error (&error);
        return FALSE;
    }

    TnyList *folders = tny_simple_list_new ();

    /* XXX:
     * - Async it
     */
    tny_folder_store_get_folders (
        TNY_FOLDER_STORE (account), folders, NULL, FALSE, NULL);
    TnyIterator *folder_iter = tny_list_create_iterator (folders);

    for (;
         !tny_iterator_is_done (folder_iter);
         tny_iterator_next (folder_iter)) {
        GObject *current;

        current = tny_iterator_get_current (folder_iter);

        if (!current) continue;

        if (strcmp (tny_folder_get_id (TNY_FOLDER (current)),
                    inFolderId) == 0) {
            folder = TNY_FOLDER (current);
            break;
        }
        g_object_unref (current);
    }
    g_object_unref (folder_iter);
    g_object_unref (folders);

    if (!folder) {
        GError *error = g_error_new (
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_FOLDER,
            "Failed to find folder with id \"%s\"!",
            inFolderId);
        dbus_g_method_return_error (context, error);
        g_clear_error (&error);
        g_object_unref (account);
        return FALSE;
    }

    /* Search for headers */
    TnyList *headers = tny_simple_list_new ();

    /* XXX:
     * - Async it
     */
    tny_folder_get_headers (folder, headers, FALSE, NULL);
    TnyIterator *header_iter = tny_list_create_iterator (headers);

    for (;
         !tny_iterator_is_done (header_iter);
         tny_iterator_next (header_iter)) {
        GObject *current = NULL;
        gchar* header_id = NULL;

        current = tny_iterator_get_current (header_iter);

        if (!current) continue;

        header_id = tny_header_dup_uid (TNY_HEADER (current));

        if (!g_strcmp0 (header_id, inMessageId)) {
            header = TNY_HEADER (current);
            g_free (header_id);
            break;
        }

        g_object_unref (current);
        g_free (header_id);
    }
    g_object_unref (header_iter);
    g_object_unref (headers);

    if (!header) {
        GError *error = g_error_new (
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "Failed to find message header with id \"%s\"!",
            inMessageId);
        dbus_g_method_return_error (context, error);
        g_clear_error (&error);
        g_object_unref (account);
        g_object_unref (folder);
        return FALSE;
    }

    /*
     * XXX:
     * - Async me
     */
    if (tny_header_get_flags (header) & TNY_HEADER_FLAG_CACHED) {
        msg = tny_folder_get_msg (folder, header, NULL);
    }
    g_object_unref (header);

    if (!msg) {
        GError *error = g_error_new (
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MESSAGE,
            "Failed to find message with id \"%s\"!",
            inMessageId);
        dbus_g_method_return_error (context, error);
        g_clear_error (&error);
        g_object_unref (account);
        g_object_unref (folder);
        return FALSE;
    }

    if (!g_strcmp0 (inMimePartId, "pos_msg")) {
        part = TNY_MIME_PART (msg);
        g_object_ref (part);
    } else if (g_str_has_prefix (inMimePartId, "pos_")) {
        TnyList     *children = NULL;
        TnyIterator *iter     = NULL;

        children = tny_simple_list_new ();

        tny_mime_part_get_parts (TNY_MIME_PART (msg), children);

        gint attachment_counter = 1;
        for (iter = tny_list_create_iterator (children);
             !tny_iterator_is_done (iter);
             tny_iterator_next (iter)) {
            GObject *current = NULL;

            current = tny_iterator_get_current (iter);

            if (current) {
                gchar *cid = g_strdup_printf ("pos_%d", attachment_counter);
                if (!g_strcmp0 (inMimePartId, cid)) {
                    part = TNY_MIME_PART (current);
                    g_object_ref (part);
                    break;
                }
                g_free (cid);
            }
            attachment_counter++;
        }

        g_object_unref (children);
        g_object_unref (iter);

    } else {
        part = find_mime_part (TNY_MIME_PART (msg), inMimePartId);
    }

    if (!part) {
        GError *error = g_error_new (
            qtmm_errors_domain (),
            QTMM_ERROR_INVALID_MIME_PART,
            "Message \"%s\" does not have a mime part with id \"%s\"!",
            inMessageId, inMimePartId);
        dbus_g_method_return_error (context, error);
        g_clear_error (&error);
        g_object_unref (account);
        g_object_unref (folder);
        g_object_unref (msg);
        return FALSE;
    }

    const gchar *filename = tny_mime_part_get_filename (part);
    gboolean expunge = FALSE;
    gchar *dynamic_path = NULL;
    gint size = -1;

    if (!filename) {
        dynamic_path = save_mime_part (
            part, inAccountId, inFolderId, inMessageId, inMimePartId, &size);
        filename = dynamic_path;

        if (!filename) filename = "";
        else expunge = TRUE;
    }

    const gchar *mimetype = tny_mime_part_get_content_type (part);

    if (!mimetype) {
        /* XXX:
         * - Don't know what to do here, perhaps we should just return empty
         *   and let caller handle the issue
         */
        mimetype = "";
    }

    gboolean is_attachment = tny_mime_part_is_attachment (part);

    dbus_g_method_return (
        context, filename, mimetype, size, is_attachment, expunge);

    g_object_unref (account);
    g_object_unref (folder);
    g_object_unref (msg);
    g_object_unref (part);
    g_free (dynamic_path);

    return TRUE;
}

/*
 * Private function implementation
 */

/* Finds a draft message based on an id */
static TnyMsg *
find_draft_message (QtmMDBusService *self,
                    const gchar *account_name,
                    const gchar *draft_id)
{
    TnyAccount *account;
    QtmMDBusServicePrivate *priv;

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, NULL);
    g_return_val_if_fail (priv->acc_store != NULL, NULL);

    if (!draft_id || !account_name) return NULL;

    QTMM_DEBUG ("Seeking account %s", account_name);

    account = modest_tny_account_store_get_tny_account_by (
        priv->acc_store, MODEST_TNY_ACCOUNT_STORE_QUERY_ID, account_name);

    if (!account) return NULL;


    g_object_unref (account);
    return NULL;
}

/* Finds a transport account from account store, based on account name */
static TnyAccount *
find_transport_account (QtmMDBusService *self,
                        const gchar *account_name)
{
    QtmMDBusServicePrivate *priv;
    TnyAccount *account;

    QTMM_DEBUG ("Seeking account %s", account_name);

    priv = GET_PRIVATE (self);

    g_return_val_if_fail (priv != NULL, NULL);
    g_return_val_if_fail (priv->acc_store != NULL, NULL);

    /* Seek an account based on a name */
    account = modest_tny_account_store_get_tny_account_by (
        priv->acc_store, MODEST_TNY_ACCOUNT_STORE_QUERY_ID, account_name);

    if (!account)
        account = modest_tny_account_store_get_tny_account_by (
            priv->acc_store, MODEST_TNY_ACCOUNT_STORE_QUERY_URL, account_name);

    if (account) {
        TnyTransportAccount *trans = NULL;
        TnyAccountType type;

        type = tny_account_get_account_type (account);

        if (type == TNY_ACCOUNT_TYPE_TRANSPORT) {
            return account;
        } else if (type == TNY_ACCOUNT_TYPE_COMBINED) {
            trans = tny_combined_account_get_transport_account (
                TNY_COMBINED_ACCOUNT (account));
            g_object_unref (account);
        }

        if (trans) return TNY_ACCOUNT (trans);
    }

    return modest_tny_account_store_get_server_account (
        priv->acc_store, account_name, TNY_ACCOUNT_TYPE_TRANSPORT);
}

/* Translates attachment list from list of maps to TnyMimePart list */
static GList *
build_attachments (QtmMDBusService *self,
                   const GPtrArray *inAttachments)
{
    GList       *list       = NULL;
    GHashTable  *att_data   = NULL;
    TnyMimePart *attachment = NULL;
    guint i = 0;

    QTMM_DEBUG();

    for (i = 0; i < inAttachments->len; i++) {
        gint file_desc = 0;
        const gchar *data = NULL;
        const gchar *filename = NULL;
        gchar *mime_type = NULL;
        const gchar *encoding = "base64";

        att_data = g_ptr_array_index (inAttachments, i);

        /* Impossible? */
        if (!att_data) continue;

        attachment = tny_camel_mime_part_new ();

        if ((filename = g_hash_table_lookup (att_data, "filename")) != NULL) {
            tny_mime_part_set_filename (attachment, filename);
	    QTMM_DEBUG ("filename: %s", filename);
        }

        if (!filename || filename[0] == '\0') {
            QTMM_DEBUG ("Cannot construct an attachment without file name");
            g_object_unref (attachment);
            continue;
        }

        if ((data = g_hash_table_lookup (att_data, "mime-type"))
            != NULL) {
            tny_mime_part_set_content_type (attachment, data);
            mime_type = g_strdup (data);
        } else {
            GFile *a_file = NULL;
            GFileInfo *a_file_info = NULL;
            GError *error = NULL;

            /* Filename existence is checked before in the loop */
            if (g_str_has_prefix (filename, "file://")) {
                a_file = g_file_new_for_uri (filename);
            } else {
                a_file = g_file_new_for_path (filename);
            }

            a_file_info = g_file_query_info (
                a_file,
                G_FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE, 0,
                NULL, &error);

            if (error) {
                QTMM_DEBUG_ERR (error, "Failed to get file info");
                g_clear_error (&error);
            }

            if (a_file_info) {
                mime_type = g_strdup (
                    g_file_info_get_attribute_string (
                        a_file_info,
                        G_FILE_ATTRIBUTE_STANDARD_FAST_CONTENT_TYPE));

                tny_mime_part_set_content_type (attachment, mime_type);

                g_object_unref (a_file_info);
                a_file_info = NULL;
            }

            if (a_file) {
                g_object_unref (a_file);
                a_file = NULL;
            }
        }

        if (!mime_type || mime_type[0] == '\0') {
            QTMM_DEBUG ("Cannot create an attachment without mime-type");
            g_object_unref (attachment);
            continue;
        }

        if ((data = g_hash_table_lookup (att_data, "content-location"))
            != NULL) {
            tny_mime_part_set_content_location (attachment, data);
        }
        if ((data = g_hash_table_lookup (att_data, "content-id")) != NULL) {
            tny_mime_part_set_content_id (attachment, data);
        }
        if ((data = g_hash_table_lookup (att_data, "description")) != NULL) {
            tny_mime_part_set_description (attachment, data);
        }
        if ((encoding = g_hash_table_lookup (att_data, "transfer-encoding"))
            != NULL) {
            tny_mime_part_set_transfer_encoding (attachment, encoding);
        } else {
            encoding = decide_encoding (mime_type);

            tny_mime_part_set_transfer_encoding (attachment, encoding);
        }

        file_desc = open (filename, O_RDONLY);

        if (file_desc != -1) {
            TnyStream *stream = tny_fs_stream_new (file_desc);
            tny_mime_part_construct (
                attachment, stream, mime_type, encoding);
            g_object_unref (stream);
        } else {
            QTMM_DEBUG ("Failed to open %s: %s", filename, strerror (errno));
            g_object_unref (attachment);
            g_free (mime_type);
            continue;
        }

        list = g_list_prepend (list, attachment);
        g_free (mime_type);
    }

    return list;
}

/* Translates inputted priority into Tny header flags */
static TnyHeaderFlags
build_priority (QtmMDBusService *self,
                const guint inPriority)
{
    TnyHeaderFlags flags = 0;

    switch (inPriority) {
    default:
    case QTMM_PRIORITY_NORMAL:
        flags = TNY_HEADER_FLAG_NORMAL_PRIORITY;
        break;
    case QTMM_PRIORITY_LOW:
        flags = TNY_HEADER_FLAG_LOW_PRIORITY;
        break;
    case QTMM_PRIORITY_HIGH:
        flags = TNY_HEADER_FLAG_HIGH_PRIORITY;
        break;
    }

    return flags;
}

/* Translates inputted header fields into Tny header fields */
static void
build_headers (QtmMDBusService  *self,
               GHashTable       *inHeaders,
               TnyList         **header_list)
{
    /* TODO: Implement meh! */
}

/* Decides an encoding for an attachment */
static const gchar *
decide_encoding (const gchar *mime_type)
{
    if (!g_strcmp0 (mime_type, "text/plain")) {
        return "8bit";
    }

    return "base64";
}

/* Starts a queued search */
static void
start_search (QtmMDBusService *self)
{
    QtmMDBusServicePrivate *priv;
    GSList *iter;

    QTMM_DEBUG();

    priv = GET_PRIVATE (self);

    /* This loop seems a bit odd, but in truth we don't need to
     * check anything else but the first instance to be sure we're
     * searching everything through.
     *
     * Signalling will cause us to dispose the object when it's completed
     * in one way or another, at the same time removing the operation from
     * queue.
     */
    while ((iter = priv->search_ops) != NULL) {
        QtmmSearchState status = QTMM_SEARCH_IDLE;
        GObject *operation = iter->data;

        /* Shouldn't happen */
        if (!operation) {
            QTMM_DEBUG ("No search op in list");
            priv->search_ops = g_slist_delete_link (priv->search_ops, iter);
            continue;
        }

        status = qtmm_search_get_status (QTMM_SEARCH (operation));

        /* There's already an operation running, no need to start
         * anything
         */
        if (status == QTMM_SEARCH_RUNNING) {
            break;
        } else if (status == QTMM_SEARCH_IDLE) {
            /* Start a search, terminate the loop */
            qtmm_search_start (QTMM_SEARCH (operation));
            break;
        }

        /* This is a fallback, we shouldn't have any completed or cancelled
         * operations in the list, but let's be sure...
         */
        priv->search_ops = g_slist_remove (priv->search_ops, operation);
        g_object_unref (operation);
    }
}

/* Headers found in a search */
static void
headers_received_cb (QtmMSearch      *search,
                     gint             operation_id,
                     const gchar     *account_id,
                     const gchar     *folder_id,
                     const GPtrArray *headers,
                     gpointer         user_data)
{
    QtmMDBusServicePrivate *priv;

    QTMM_DEBUG();

    if (!QTMM_IS_DBUS_SERVICE (user_data)) return;

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    QTMM_DEBUG ("op id: %d, folder_id: %s, account_id: %s",
                operation_id, folder_id, account_id);

    g_signal_emit (user_data, signals[SIG_HEADERS_RECEIVED], 0,
                   operation_id, account_id, folder_id, headers);
}

/* Search is done */
static void
search_done_cb (QtmMSearch *search,
                gpointer    user_data)
{
    QtmMDBusServicePrivate *priv;
    gboolean operation_done = TRUE, found = FALSE;
    gint operation_id = -1;
    GSList *iter;

    QTMM_DEBUG();

    if (!QTMM_IS_DBUS_SERVICE (user_data)) return;

    priv = GET_PRIVATE (user_data);

    if (!priv || priv->disposed) return;

    operation_id = qtmm_search_get_id (search);

    for (iter = priv->search_ops; iter != NULL; iter = iter->next) {
        QtmMSearch *other = iter->data;
        QtmmSearchState other_state = QTMM_SEARCH_IDLE;

        if (other == search) {
            found = TRUE;
            continue;
        }
        if (operation_id != qtmm_search_get_id (other)) continue;

        other_state = qtmm_search_get_status (other);

        if (other_state == QTMM_SEARCH_IDLE ||
            other_state == QTMM_SEARCH_RUNNING) {
            QTMM_DEBUG ("More operations still running with id %d",
                        operation_id);
            operation_done = FALSE;
            break;
        }
    }

    if (found) {
        QTMM_DEBUG ("Removing operation from list");
        priv->search_ops = g_slist_remove (priv->search_ops, search);
        g_object_unref (search);
    }

    if (operation_done) {
        QTMM_DEBUG ("Emitting \"HeadersFetched\"");
        g_signal_emit (user_data, signals[SIG_HEADERS_FETCHED], 0,
                       operation_id);
    }

    /* And finally, fire next operation */
    start_search (QTMM_DBUS_SERVICE (user_data));
}

/* Finds a body parent */
static TnyMimePart *
find_body_parent_recursive (TnyMimePart *part)
{
    const gchar *msg_content_type = NULL;
    msg_content_type = tny_mime_part_get_content_type (part);

    if ((msg_content_type != NULL) &&
        (!g_ascii_strcasecmp (msg_content_type, "multipart/alternative"))) {
        return g_object_ref (TNY_MIME_PART (part));
    } else if ((msg_content_type != NULL) &&
             (g_str_has_prefix (msg_content_type, "multipart/"))) {
        TnyIterator *iter = NULL;
        TnyMimePart *alternative_part = NULL;
        TnyMimePart *related_part = NULL;
        TnyList *parts = TNY_LIST (tny_simple_list_new ());
        tny_mime_part_get_parts (part, parts);
        iter = tny_list_create_iterator (parts);

        while (!tny_iterator_is_done (iter)) {
            TnyMimePart *part = TNY_MIME_PART (tny_iterator_get_current (iter));
            if (part && !g_ascii_strcasecmp(tny_mime_part_get_content_type (part), "multipart/alternative"))
            {
                alternative_part = part;
                break;
            } else if (part && !g_ascii_strcasecmp (tny_mime_part_get_content_type (part), "multipart/related")) {
                related_part = part;
                break;
            }

            if (part)
                g_object_unref (part);

            tny_iterator_next (iter);
        }
        g_object_unref (iter);
        g_object_unref (parts);
        if (related_part) {
            TnyMimePart *result;
            result = find_body_parent_recursive (related_part);
            g_object_unref (related_part);
            return result;
        } else if (alternative_part) {
            return alternative_part;
        } else {
            return g_object_ref (part);
        }
    } else {
        return NULL;
    }
}

/* Finds a specific mime part from a message, based on an id */
static TnyMimePart *
find_mime_part (TnyMimePart *part,
                const gchar *mime_part_id)
{
    TnyMimePart *match    = NULL;
    TnyList     *children = NULL;
    TnyIterator *iter     = NULL;

    /* Ref is done to ensure that the return value is always reffed,
     * and as such, it is always safe to unref the return value, if it's
     * not null.
     */
    if (!g_strcmp0 (tny_mime_part_get_content_id (part), mime_part_id))
        return g_object_ref (part);

    children = tny_simple_list_new ();

    tny_mime_part_get_parts (part, children);

    for (iter = tny_list_create_iterator (children);
         !tny_iterator_is_done (iter);
         tny_iterator_next (iter)) {
        GObject *current = NULL;

        current = tny_iterator_get_current (iter);

        if (!current) continue;

        match = find_mime_part (TNY_MIME_PART (current), mime_part_id);

        /* In truth, match can be the same as current, but it's reffed in
         * the recursive call, so we can unref the current here...
         */
        g_object_unref (current);

        /* And if we found a match, end recursion / loop */
        if (match) break;
    }

    g_object_unref (children);
    g_object_unref (iter);

    return match;
}

/* Saves the mime part into a temporary file */
static gchar *
save_mime_part (
    TnyMimePart *mime_part,
    const gchar *account_id,
    const gchar *folder_id,
    const gchar *message_id,
    const gchar *mime_part_id,
    gint *size)
{
    QTMM_DEBUG();

    gchar *file_path = g_build_filename (
        g_get_tmp_dir (),
        PACKAGE,
        NULL);

    GFile *file = g_file_new_for_path (file_path);
    GError *error = NULL;
    g_file_make_directory (file, NULL, &error);
    g_object_unref (file);

    if (error) {
        if (error->code != G_IO_ERROR_EXISTS) {
            QTMM_DEBUG_ERR (error, "Failed to create directory \"%s\"",
                            file_path);
            g_free (file_path);
            g_clear_error (&error);
            return NULL;
        }
        g_clear_error (&error);
    }

    gchar *path = g_build_filename (file_path, "mime_part_XXXXXX", NULL);
    g_free (file_path);

    gint fd = mkstemp (path);

    /* Opening failed */
    if (fd == -1) {
        QTMM_DEBUG ("Failed to open file \"%s\": %s", path, strerror (errno));
        g_free (path);
        return NULL;
    }

    TnyStream *stream = tny_fs_stream_new (fd);

    /* XXX: Might be worthwhile to async this as well */
    gssize written = tny_mime_part_decode_to_stream (
        mime_part, stream, &error);
    g_object_unref (stream);

    if (error) {
        QTMM_DEBUG_ERR (error, "Failed to decode mime part to stream");
        g_clear_error (&error);
        g_free (path);
        return NULL;
    }

    if (size) {
        *size = written;
    }

    return path;
}
