#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QDir>
#include <QSettings>
#include <QDebug>
#include <QProcess>
#include <QPixmap>
#include <QTimer>
#include <QMenuBar>
#include <QMaemo5InformationBox>

QString MainWindow::ScreenshotPath = "/home/user/.config/sakya/qtlscreenshots";

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    QDir().mkpath(MainWindow::ScreenshotPath);
    m_ScrenshotMaker = NULL;
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5LandscapeOrientation);
#endif
    getThemesList();
    connect(ui->lstThemes, SIGNAL(currentRowChanged(int)), this, SLOT(onListRowChanged(int)));
    connect(ui->lstThemes, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(onListDoubleClicked(QModelIndex)));

    createMainMenu();
    createContextMenu();
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::createMainMenu()
{
    QMenu *menu = menuBar()->addMenu(tr("QtLockscreenConfig"));
    QAction* action = new QAction(tr("Rebuild screenshots cache"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(clearCacheSlot()));
    menu->addAction(action);
    action = new QAction(tr("About"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(aboutSlot()));
    menu->addAction(action);
}

void MainWindow::createContextMenu()
{
    m_ThemeMenu = new QMenu(this);
    QAction* action = new QAction(tr("Preview"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(previewThemeSlot()));
    m_ThemeMenu->addAction(action);
    action = new QAction(tr("Update screenshot"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(updateScreenshotThemeSlot()));
    m_ThemeMenu->addAction(action);
    action = new QAction(tr("Set as active theme"), this);
    connect(action, SIGNAL(triggered()), this, SLOT(setActiveThemeSlot()));
    m_ThemeMenu->addAction(action);

    ui->lstThemes->setContextMenuPolicy(Qt::CustomContextMenu);
    connect(ui->lstThemes, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(themeMenuSlot(QPoint)));
}

void MainWindow::getThemesList()
{
    m_Themes.clear();

    QString dir = "/opt/qtlockscreen";

    QDir themesDir(dir);
    QStringList filters = QStringList() << "*.ini";
    themesDir.setNameFilters(filters);
    themesDir.setSorting(QDir::Name);
    QStringList list = themesDir.entryList();

    foreach (QString theme, list){
        qDebug() << theme;
        QString themeFile = QString("%1/%2").arg(dir).arg(theme);
        QSettings s(themeFile, QSettings::IniFormat, this);

        QString name = s.value("Main/ThemeName", "").toString();
        QString author = s.value("Main/ThemeAuthor", "").toString();

        if (!name.isEmpty()){
            qDebug() << "  " << name << author;
            Theme* t = new Theme(this);
            t->name = name;
            t->author = author;
            t->file = themeFile;
            m_Themes.append(t);

            QString filename = QString("%1/%2.png").arg(MainWindow::ScreenshotPath).arg(t->name);
            t->screenshot = QPixmap(filename);
        }
    }

    ui->lstThemes->clear();
    foreach (Theme* t, m_Themes){
        ui->lstThemes->addItem(t->name);
    }

    if (m_Themes.count() > 0)
        ui->lstThemes->setCurrentRow(0);
    else
        ui->lblScreenshot->setText(tr("No theme found"));

    m_ScrenshotMaker = new ScreenshotMaker(this, m_Themes);
    connect(m_ScrenshotMaker, SIGNAL(screenshot(Theme*,QString)), this, SLOT(onNewScreenshot(Theme*, QString)));
    connect(m_ScrenshotMaker, SIGNAL(finished()), this, SLOT(onScreenshotMakerFinished()));
    m_ScrenshotMaker->start();
}

void MainWindow::onListDoubleClicked(QModelIndex)
{
    setActiveThemeSlot();
}

void MainWindow::onListRowChanged(int row)
{
    if (row < 0 || row >= m_Themes.count())
        return;

    Theme* theme = m_Themes.at(row);
    if (!theme->screenshot.isNull()){
        ui->lblScreenshot->setPixmap(theme->screenshot);
    }else{
        ui->lblScreenshot->setPixmap(QPixmap());
    }
}

void MainWindow::onScreenshotMakerFinished()
{
    onListRowChanged(ui->lstThemes->currentRow());
}

void MainWindow::onNewScreenshot(Theme* theme, QString filename)
{
    QPixmap s = QPixmap(filename);
    if (!s.isNull()){
        theme->screenshot = s.scaledToWidth(s.width() / 2);
        theme->screenshot.save(QString("%1/%2.png").arg(MainWindow::ScreenshotPath).arg(theme->name));
    }

    Theme* currTheme = m_Themes.at(ui->lstThemes->currentRow());
    if (currTheme == theme)
        onListRowChanged(ui->lstThemes->currentRow());
}

void MainWindow::themeMenuSlot(QPoint point)
{
    m_ThemeMenu->exec(mapToGlobal(point));
}

void MainWindow::previewThemeSlot()
{
    setEnabled(false);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QApplication::processEvents();

    Theme* t = m_Themes.at(ui->lstThemes->currentRow());

    QProcess p;
    p.start(QString("/usr/bin/qtlockscreen -d -c \"%2\"").arg(t->file));
    p.waitForFinished(-1);

    setEnabled(true);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void MainWindow::setActiveThemeSlot()
{
    int row = ui->lstThemes->currentRow();
    if (row < 0 || row >= m_Themes.count())
        return;

    Theme* t = m_Themes.at(row);
    if (confirmMessage(this, tr("Confirm"), tr("Set '%1' as the active theme?").arg(t->name)) != QMessageBox::Yes)
        return;

    setEnabled(false);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QApplication::processEvents();

    QProcess p;
    p.start("rm -f /home/user/.config/sakya/QtLockscreen.ini");
    p.waitForFinished(-1);

    p.start(QString("ln -s \"%1\" /home/user/.config/sakya/QtLockscreen.ini").arg(t->file));
    p.waitForFinished(-1);

    p.start("pkill -HUP -f /usr/bin/qtlockscreen");
    p.waitForFinished(-1);

    /*p.start("sh -c \"if [ ! -n \\\"$(pidof qtlockscreen)\\\" ]; then echo '1'; fi\"");
    p.waitForFinished(-1);
    QString res = p.readAll();
    qDebug() << "Result:" << res;*/

    setEnabled(true);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

    infoMessage(this, tr("Active theme changed"));
}

void MainWindow::updateScreenshotThemeSlot()
{
    setEnabled(false);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    QApplication::processEvents();


    Theme* t = m_Themes.at(ui->lstThemes->currentRow());
    QString tempFile = "/tmp/qtl.png";

    QProcess p;
    p.start(QString("/usr/bin/qtlockscreen -s \"%1\" -c \"%2\"").arg(tempFile).arg(t->file));
    p.waitForFinished(-1);
    onNewScreenshot(t, tempFile);

    setEnabled(true);
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void MainWindow::clearCacheSlot()
{
    if (confirmMessage(this, tr("Confirm"), tr("Rebuild the screenshots cache?")) != QMessageBox::Yes)
        return;

    if (m_ScrenshotMaker != NULL){
        m_ScrenshotMaker->terminate();
        delete m_ScrenshotMaker;
        m_ScrenshotMaker = NULL;
    }

    foreach (Theme* t, m_Themes){
        t->screenshot = QPixmap();
    }
    onListRowChanged(ui->lstThemes->currentRow());

    QProcess p;
    QString cmd = QString("sh -c \"rm -f %1/*.png\"").arg(MainWindow::ScreenshotPath);
    qDebug() << cmd;
    p.start(cmd);
    p.waitForFinished(-1);

    m_ScrenshotMaker = new ScreenshotMaker(this, m_Themes);
    connect(m_ScrenshotMaker, SIGNAL(screenshot(Theme*,QString)), this, SLOT(onNewScreenshot(Theme*, QString)));
    m_ScrenshotMaker->start();
}


QMessageBox::StandardButton MainWindow::confirmMessage(QWidget* parent, QString title, QString message)
{
    QMessageBox msgBox(parent);
    msgBox.setIcon(QMessageBox::Question);
    if (title.isEmpty())
        title = QObject::tr("Confirm");
    msgBox.setWindowTitle(title);
    msgBox.setText(message);
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    return (QMessageBox::StandardButton)msgBox.exec();
}

void MainWindow::infoMessage(QWidget* parent, QString message)
{
    QMaemo5InformationBox* box = new QMaemo5InformationBox(parent);
    QObject::connect(box, SIGNAL(clicked()), box, SLOT(accept()));

    QLabel* text = new QLabel(box);
    text->setMinimumHeight(100);
    text->setText(message);
    text->setAlignment(Qt::AlignCenter);

    box->setWidget(text);
    box->exec();
}

void MainWindow::aboutSlot()
{
    QString msg = QString("%1 v.%2\n(c) 2011 Paolo Iommarini").arg(QApplication::applicationName())
                                                             .arg(QApplication::applicationVersion());
    QMessageBox msgBox(this);
    msgBox.setIcon(QMessageBox::Information);
    msgBox.setWindowTitle(tr("About"));
    msgBox.setText(msg);
    msgBox.setStandardButtons(QMessageBox::Ok);
    msgBox.exec();
}
