#include <stdlib.h>
#include <QApplication>

#include "engine.h"
#include "reversigame.h"

int const MAX_AI_LEVEL = 7;

inline
int random_int(int max_value)
{
    return random() % max_value;
}

inline
ChipColor opponentColorFor(ChipColor color)
{
    if(color == Empty)
        return color;
    else
        return ( color == White ? Black : White );
}

// (Re)initialize the stack so that is empty, and at the same time resize it to 'size'.

void SquareStack::init(int size)
{
    resize(size);

    top_ = 0;
    for (int i = 0; i < size; i++)
        squarestack_[i].setXY(0,0);
}

// Some special values used in the search.
static const int LARGEINT      = 99999;
static const int ILLEGAL_VALUE = 8888888;
static const int BC_WEIGHT     = 3;

Engine::Engine(int strength, int seed):
    strength_(strength)
{
    init(strength, seed);
}

void
Engine::init(int strength, int seed)
{
    strength_ = strength;
    score_.set(White, 0);
    score_.set(Black, 0);
    bc_score_.set(White, 0);
    bc_score_.set(Black, 0);

    srandom(seed);
    setupBcBoard();
    setupBits();
} /* Engine::init(int strength = 1, int seed = 0) */

// Set up the board control values that will be used in evaluation of the position.
void
Engine::setupBcBoard()
{
    for (unsigned i=1; i < 9; i++)
        for (unsigned j=1; j < 9; j++)
        {
            if (i == 2 || i == 7)
                bc_board_[i][j] = -1;
            else
                bc_board_[i][j] = 0;

            if (j == 2 || j == 7)
                bc_board_[i][j] -= 1;
        }

  bc_board_[1][1] = 2;
  bc_board_[8][1] = 2;
  bc_board_[1][8] = 2;
  bc_board_[8][8] = 2;

  bc_board_[1][2] = -1;
  bc_board_[2][1] = -1;
  bc_board_[1][7] = -1;
  bc_board_[7][1] = -1;
  bc_board_[8][2] = -1;
  bc_board_[2][8] = -1;
  bc_board_[8][7] = -1;
  bc_board_[7][8] = -1;
}//setupBcBoard

void
Engine::setupBits()
{
    quint64 bits = 1;

    // Store a 64 bit unsigned it with the corresponding bit set for each square.
    for (unsigned i=1; i < FIELDSIZE + 1; i++)
        for (unsigned j=1; j < FIELDSIZE + 1; j++) {
            coord_bit_[i][j] = bits;
            bits *= 2;
        }

    // Store a bitmap consisting of all neighbors for each square.
    for (unsigned i=1; i < FIELDSIZE + 1; i++)
        for (unsigned j=1; j < FIELDSIZE + 1; j++) {
            neighbor_bits_[i][j] = 0;

            for (int xinc=-1; xinc<=1; xinc++)
                for (int yinc=-1; yinc<=1; yinc++) {
                    if (xinc != 0 || yinc != 0)
                        if (i + xinc > 0 && i + xinc < FIELDSIZE + 1 && j + yinc > 0 && j + yinc < FIELDSIZE + 1)
                            neighbor_bits_[i][j] |= coord_bit_[i + xinc][j + yinc];
                }
        }//for j
}//setupBits

ReversiPos
Engine::computeMove(const ReversiGame& game)
{
    // Suppose that we should give a heuristic evaluation.  If we are
    // close to the end of the game we can make an exhaustive search,
    // but that case is determined further down.
    exhaustive_ = false;

    // Get the color to calculate the move for.
    ChipColor color = game.currentPlayer();
    ChipColor opponentColor = game.opponentPlayer();
    
    // Figure out the current score
    score_.set(White, game.score(White));
    score_.set(Black, game.score(Black));

    unsigned total_score = score_.score(White) + score_.score(Black);
    // Treat the first move as a special case (we can basically just pick a move at random).
    if (total_score == 4)
        return computeFirstMove();

    // Let there be room for 3000 changes during the recursive search.
    // This is more than will ever be needed.
    squarestack_.init(3000);

    // Get the search depth.  If we are close to the end of the game,
    // the number of possible moves goes down, so we can search deeper
    // without using more time.
    depth_ = strength_;
    if (total_score + depth_ + 3 >= CHIPS_COUNT)
        depth_ = CHIPS_COUNT - total_score;
    else if (total_score + depth_ + 4 >= CHIPS_COUNT)
        depth_ += 2;
    else if (total_score + depth_ + 5 >= CHIPS_COUNT)
        depth_++;

    // If we are very close to the end, we can even make the search
    // exhaustive.
    if (total_score + depth_ >= CHIPS_COUNT)
        exhaustive_ = true;

    // The evaluation is a linear combination of the score (number of
    // pieces) and the sum of the scores for the squares (given by
    // m_bc_score).  The earlier in the game, the more we use the square
    // values and the later in the game the more we use the number of
    // pieces.
    coeff_ = 100 - (100* (total_score + depth_ - 4)) / 60;

    // Initialize the board that we use for the search.
    for (uint x = 0; x < FIELDSIZE + 2; ++x)
        for (uint y = 0; y < FIELDSIZE + 2; ++y) {
            if (1 <= x && x <= FIELDSIZE && 1 <= y && y <= FIELDSIZE)
                board_[x][y] = game.chipColor(y-1, x-1);
            else
                board_[x][y] = Empty;
        }//for y

    // Initialize a lot of stuff that we will use in the search.

    // Initialize m_bc_score to the current bc score.  This is kept
    // up-to-date incrementally so that way we won't have to calculate
    // it from scratch for each evaluation.
    bc_score_.set(White, calcBcScore(White));
    bc_score_.set(Black, calcBcScore(Black));

    quint64 colorbits    = computeOccupiedBits(color);
    quint64 opponentbits = computeOccupiedBits(opponentColor);

    int maxval = -LARGEINT;
    int max_x = 0;
    int max_y = 0;
    MoveAndValue moves[60];
    int number_of_moves = 0;
    int number_of_maxval = 0;

    quint64 null_bits;
    null_bits = 0;

    // The main search loop.  Step through all possible moves and keep
    // track of the most valuable one.  This move is stored in
    // (max_x, max_y) and the value is stored in maxval.
    nodes_searched_ = 0;
    for (unsigned x = 1; x < FIELDSIZE + 1; x++) {
        for (unsigned y = 1; y < FIELDSIZE + 1; y++) {
            // Don't bother with non-empty squares and squares that aren't
            // neighbors to opponent pieces.
            if (board_[x][y] != Empty || (neighbor_bits_[x][y] & opponentbits) == null_bits)
                continue;

            int val = computeMove2(x, y, color, 1, maxval, colorbits, opponentbits);
            if (val != ILLEGAL_VALUE)
            {
                moves[number_of_moves++].setXYV(x, y, val);

                // If the move is better than all previous moves, then record this fact...
                if (val > maxval)
                {
                    // ...except that we want to make the computer miss some
                    // good moves so that beginners can play against the program
                    // and not always lose. But in max level always record better move.
                    if (maxval == -LARGEINT || strength_ == MAX_AI_LEVEL || random_int(MAX_AI_LEVEL) < strength_ || random_int(2) != 1)
                    {
                        maxval = val;
                        max_x  = x;
                        max_y  = y;

                        number_of_maxval = 1;
                    }
                }//if (val > maxval)
                else
                    if (val == maxval)
                        number_of_maxval++;
            }

        }//for y
    }//for x
    // If there are more than one best move, the pick one randomly.
    if (number_of_maxval > 1)
    {
        int  r = random_int(number_of_maxval) + 1;
        int  i;
        for (i = 0; i < number_of_moves; ++i) {
            if (moves[i].value_ == maxval && --r <= 0)
                break;
        }
        max_x = moves[i].x_;
        max_y = moves[i].y_;
    }//if (number_of_maxval > 1)

    return (maxval != -LARGEINT) ? ReversiPos(max_y-1, max_x-1) : ReversiPos(FIELDSIZE + 1, FIELDSIZE + 1);
}//computeMove

// Play a move at (xplay, yplay) and generate a value for it.  If we
// are at the maximum search depth, we get the value by calling
// EvaluatePosition(), otherwise we get it by performing an alphabeta
// search.

int
Engine::computeMove2(int xplay, int yplay, ChipColor color, int level,  int      cutoffval,
                    quint64  colorbits, quint64 opponentbits)
{
    int               number_of_turned = 0;
    SquareStackEntry  mse;
    ChipColor         opponent = opponentColorFor(color);

    nodes_searched_++;

    // Put the piece on the board and incrementally update scores and bitmaps.
    board_[xplay][yplay] = color;
    colorbits |= coord_bit_[xplay][yplay];
    score_.inc(color);
    bc_score_.add(color, bc_board_[xplay][yplay]);
    // Loop through all 8 directions and turn the pieces that can be turned.
    for (int xinc = -1; xinc <= 1; xinc++)
    {
        for (int yinc = -1; yinc <= 1; yinc++) {
            if (xinc == 0 && yinc == 0)
                continue;

            int x, y;
            for (x = xplay + xinc, y = yplay + yinc; board_[x][y] == opponent; x += xinc, y += yinc)
                ;

            // If we found the end of a turnable row, then go back and turn
            // all pieces on the way back.  Also push the squares with
            // turned pieces on the squarestack so that we can undo the move
            // later.
            if (board_[x][y] == color)
                for (x -= xinc, y -= yinc; x != xplay || y != yplay; x -= xinc, y -= yinc)
                {
                    board_[x][y] = color;
                    colorbits |= coord_bit_[x][y];
                    opponentbits &= ~coord_bit_[x][y];

                    squarestack_.push(x, y);

                    bc_score_.add(color, bc_board_[x][y]);
                    bc_score_.sub(opponent, bc_board_[x][y]);
                    number_of_turned++;
                }
        }//for yinc
    } //for xinc 
    int retval = -LARGEINT;

    // If we managed to turn at least one piece, then (xplay, yplay) was
    // a legal move.  Now find out the value of the move.
    if (number_of_turned > 0) 
    {

        // First adjust the number of pieces for each side.
        score_.add(color, number_of_turned);
        score_.sub(opponent, number_of_turned);

        // If we are at the bottom of the search, get the evaluation.
        if (level >= depth_)
            retval = evaluatePosition(color); // Terminal node
        else
        {
            int maxval = tryAllMoves(opponent, level, cutoffval, opponentbits, colorbits);

            if (maxval != -LARGEINT)
                retval = -maxval;
            else {
                // No possible move for the opponent, it is colors turn again:
                retval = tryAllMoves(color, level, -LARGEINT, colorbits, opponentbits);

                if (retval == -LARGEINT)
                {
                    // No possible move for anybody => end of game:
                    int finalscore = score_.score(color) - score_.score(opponent);

                    if (exhaustive_)
                        retval = finalscore;
                    else {
                        // Take a sure win and avoid a sure loss (may not be optimal):

                        if (finalscore > 0)
                            retval = LARGEINT - 65 + finalscore;
                        else if (finalscore < 0)
                            retval = -(LARGEINT - 65 + finalscore);
                        else
                            retval = 0;
                    }
                }//if (retval == -LARGEINT)
            }//else if (maxval != -LARGEINT)
        }
        score_.add(opponent, number_of_turned);
        score_.sub(color, number_of_turned);
    }//if (number_of_turned > 0)

    // Undo the move.  Start by unturning the turned pieces.
    for (int i = number_of_turned; i > 0; i--) {
        mse = squarestack_.pop();
        bc_score_.add(opponent, bc_board_[mse.x_][mse.y_]);
        bc_score_.sub(color, bc_board_[mse.x_][mse.y_]);
        board_[mse.x_][mse.y_] = opponent;
    }

    // Now remove the new piece that we put down.
    board_[xplay][yplay] = Empty;
    score_.sub(color, 1);
    bc_score_.sub(color, bc_board_[xplay][yplay]);

    // Return a suitable value.
    if (number_of_turned < 1)
        return ILLEGAL_VALUE;
    else
        return retval;
} /* computeMove2 */

// Calculate a heuristic value for the current position.  If we are at
// the end of the game, do this by counting the pieces.  Otherwise do
// it by combining the score using the number of pieces, and the score
// using the board control values.
int
Engine::evaluatePosition(ChipColor color)
{
    int retval;

    ChipColor opponent = opponentColorFor(color);

    int score_color    = score_.score(color);
    int score_opponent = score_.score(opponent);

    if (exhaustive_)
        retval = score_color - score_opponent;
    else {
        retval = (100 - coeff_) * (score_.score(color) - score_.score(opponent)) +
                coeff_ * BC_WEIGHT * (bc_score_.score(color) - bc_score_.score(opponent));
    }
    return retval;
} /* Engine::evaluatePosition(ChipColor color) */

// Generate all legal moves from the current position, and do a search
// to see the value of them.  This function returns the value of the
// most valuable move, but not the move itself.
int
Engine::tryAllMoves(ChipColor opponent, int level, int cutoffval, quint64  opponentbits, quint64 colorbits)
{
    int maxval = -LARGEINT;

    // Keep GUI alive by calling the event loop.
    qApp->processEvents();

    quint64  null_bits;
    null_bits = 0;

    for (unsigned x = 1; x < FIELDSIZE + 1; x++) {
        for (unsigned y = 1; y < FIELDSIZE + 1; y++) {
            if (board_[x][y] == Empty && (neighbor_bits_[x][y] & colorbits) != null_bits)
            {
                int val = computeMove2(x, y, opponent, level+1, maxval, opponentbits, colorbits);

                if (val != ILLEGAL_VALUE && val > maxval)
                {
                    maxval = val;
                    if (maxval > -cutoffval)
                        break;
                }
            }
        }//for y

        if (maxval > -cutoffval)
            break;
    }//for x

    //if (interrupted())
        //return -LARGEINT;

    return maxval;
} /* Engine::tryAllMoves(ChipColor opponent, int level, int cutoffval, quint64  opponentbits, quint64 colorbits) */

ReversiPos
Engine::computeFirstMove()
{
    unsigned x[] = {2, 3, 4, 5};
    unsigned y[] = {3, 2, 5, 4};
    // First step always have black player
    unsigned v = random_int(4);
    return ReversiPos(x[v], y[v]);
} //computeFirstMove

int
Engine::calcBcScore(ChipColor color)
{
    int sum = 0;

    for (unsigned i=1; i < FIELDSIZE + 1; i++)
        for (unsigned j=1; j < FIELDSIZE + 1; j++)
            if (board_[i][j] == color)
                sum += bc_board_[i][j];
    return sum;
} /* Engine::calcBcScore(ChipColor color) */

quint64
Engine::computeOccupiedBits(ChipColor color)
{
    quint64 retval = 0;

    for (unsigned i=1; i < FIELDSIZE + 1; i++)
        for (unsigned j=1; j < FIELDSIZE + 1; j++)
            if (board_[i][j] == color) retval |= coord_bit_[i][j];

    return retval;
} /* Engine::computeOccupiedBits(ChipColor color) */
