#include <QGraphicsSceneMouseEvent>
#include <QWidget>
#include <QTimer>
#include <QPixmap>

#include "reversiscene.h"
#include "reversiitem.h"
#include "reversidefs.h"
#include "reversigame.h"
#include "pixmaps/EmptyCell.xpm"

ReversiScene::ReversiScene(ReversiGame &reversiGame):
        reversiGame_(reversiGame), cellWidth_(0), cellHeight_(0),
        isNetworkMode_(false)
{
    //temporary pixmap for calculating width and heigh of cell
    QPixmap p(EmptyCell_xpm);
    cellWidth_ = p.width();
    cellHeight_ = p.height();

    pixmapsFactory_ = new PixmapsFactory();

    animTimer_ = new QTimer(this);
    for (unsigned i = 0, x = 0; i < FIELDSIZE; ++i, x += cellWidth_)
        for(unsigned j = 0, y = 0; j < FIELDSIZE; j++, y += cellHeight_)
        {
            ReversiItem *ri = new ReversiItem(this, pixmapsFactory_);
            ri->setPos(x, y);
        }
    fieldWidth_ = FIELDSIZE * cellWidth_;
    fieldHeight_ = FIELDSIZE * cellHeight_;

    connect(&reversiGame_, SIGNAL(stepComplete()), SLOT(slotStepComplete()));
    connect(&reversiGame_, SIGNAL(newGame(bool)), SLOT(slotNewGame(bool)));
    connect(animTimer_, SIGNAL(timeout()), SLOT(slotAnimationStep()));
}

ReversiScene::~ReversiScene()
{
    delete animTimer_;
    delete pixmapsFactory_;
}

void ReversiScene::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    int x = int(event->scenePos().x() / cellWidth_);
    int y = int(event->scenePos().y() / cellHeight_);
    currentPlayer_ = reversiGame_.currentPlayer();
    if (isNetworkMode_)
        emit playerStep(x, y);
    else
        reversiGame_.makeStep(x, y);
}

void ReversiScene::slotNewGame(bool networkGame)
{
    if (animTimer_->isActive())
        animTimer_->stop();
    currentPlayer_ = reversiGame_.currentPlayer();
    isNetworkMode_ = networkGame;
    syncBoard();
}

void ReversiScene::syncBoard()
{
    for (unsigned i = 0; i < FIELDSIZE; ++i)
        for(unsigned j = 0; j < FIELDSIZE; j++)
        {
            ReversiItem *it = static_cast<ReversiItem*>(itemAt(i * cellWidth_, j * cellHeight_));
            it->setColor(reversiGame_.chipColor(i, j));
        }
    update(0, 0, fieldWidth_, fieldHeight_);
}

void ReversiScene::slotStepComplete()
{
    int animStartStep;
    int i;

    const QList<QPoint> &changedCells = reversiGame_.changedCells();
    int cell_count = changedCells.count();

    if (reversiGame_.opponentPlayer() == White)
    {
        animDirection_ = -1;
        animStartStep = 13;
    }
    else
    {
        animDirection_ = 1;
        animStartStep = 3;
    }
    // inital animation step for all cells
    for (i = 0; i < cell_count; ++i)
        animCellStep_[i] = animStartStep;
    animFirstCell_ = animLastCell_ = 1;

    //set first chip
    int x = changedCells.at(0).x();
    int y = changedCells.at(0).y();
    ReversiItem *it = static_cast<ReversiItem*>(itemAt(x * cellWidth_, y * cellHeight_));
    it->setColor(reversiGame_.opponentPlayer());
    update(x * cellWidth_, y * cellHeight_, cellWidth_, cellHeight_);

    //start animation
    animTimer_->start(50);
}//slotStepComplete

void ReversiScene::slotAnimationStep()
{
    int i;
    const QList<QPoint> &changedCells = reversiGame_.changedCells();
    int cell_count = changedCells.count();
    //set new step of animation for every cell
    for (i = animFirstCell_; i <= animLastCell_; ++i)
    {
        animCellStep_[i] += animDirection_;
        int x = changedCells.at(i).x();
        int y = changedCells.at(i).y();
        ReversiItem *it = static_cast<ReversiItem*>(itemAt(x * cellWidth_, y * cellHeight_));
        it->setColor(animCellStep_[i]);
        update(x * cellWidth_, y * cellHeight_, cellWidth_, cellHeight_);
    }
    // if cell fully flip, move to next
    if ((animDirection_ == 1 && animCellStep_[animFirstCell_] == 12) ||
        (animDirection_ == -1 && animCellStep_[animFirstCell_] == 3))
        ++animFirstCell_;
    if (animLastCell_ - animFirstCell_ < 10 and animLastCell_ < cell_count - 1)
        ++animLastCell_;
     //if all cells flipped - stop animation
    if (animFirstCell_ >= cell_count)
    {
        animTimer_->stop();

        currentPlayer_ = reversiGame_.opponentPlayer();
        for(int i = 0; i < changedCells.size(); ++i)
        {
            int x = changedCells.at(i).x();
            int y = changedCells.at(i).y();
            ReversiItem *it = static_cast<ReversiItem*>(itemAt(x * cellWidth_, y * cellHeight_));
            it->setColor(currentPlayer_);
            update(x * cellWidth_, y * cellHeight_, cellWidth_, cellHeight_);
        }
    reversiGame_.startNextTurn();
    }
} //slotAnimationStep
