//    QtBatteryWidget
//    Copyright (C) 2011 Paolo Iommarini
//    sakya_tg@yahoo.it
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#include "batterywidget.h"
#include <QPainter>

BatteryWidget::BatteryWidget(QWidget *parent) :
    QWidget(parent)
{
    m_Background = NULL;
    m_State = NULL;
    m_Critical = NULL;
    m_Low = NULL;
    m_Normal = NULL;

    m_Min = 0;
    m_Max = 100;
    SetValue(0);

    m_Margins = QMargins(0, 0, 0, 0);
    setContentsMargins(0,0,0,0);
    m_Charging = false;
}

BatteryWidget::~BatteryWidget()
{
    m_State = NULL;
    if (m_Background)
        delete m_Background;
    if (m_Critical)
        delete m_Critical;
    if (m_Low)
        delete m_Low;
    if (m_Normal)
        delete m_Normal;
}

void BatteryWidget::SetRange(int min, int max)
{
    m_Min = min;
    m_Max = max;
}

int BatteryWidget::Maximum()
{
    return m_Max;
}

int BatteryWidget::Minimum()
{
    return m_Min;
}

int BatteryWidget::Value()
{
    return m_Value;
}

void BatteryWidget::SetValue(int value, bool updateLabel)
{
    m_Value = value;

    int h = size().height() - m_Margins.top() - m_Margins.bottom();
    int w = size().width() - m_Margins.left() - m_Margins.right();
    m_InsideSize = QSize(w, h);
    m_Perc = (float)m_Value / (float)(m_Max - m_Min) * 100.0;

    if (updateLabel){
        m_ValueStr = QString("%1%").arg(QString::number(m_Perc, 'f', 1));
        QFontMetrics fm(font());
        m_ValueStrSize = fm.size(Qt::TextSingleLine, m_ValueStr);

        if (m_Perc >= m_LowLevel)
            m_State = m_Normal;
        else if (m_Perc >= m_CriticalLevel)
            m_State = m_Low;
        else
            m_State = m_Critical;
    }else{
        m_State = m_Normal;
    }
    update();
}

void BatteryWidget::SetBackgroundImage(QPixmap background)
{
    if (m_Background)
        delete m_Background;
    m_Background = NULL;

    if (!background.isNull())
        m_Background = new QPixmap(background);
}

void BatteryWidget::SetCriticalLevelImage(QPixmap image)
{
    m_Critical = new QPixmap(image);
}

void BatteryWidget::SetLowLevelImage(QPixmap image)
{
    m_Low = new QPixmap(image);
}

void BatteryWidget::SetNormalLevelImage(QPixmap image)
{
    m_Normal = new QPixmap(image);
}

void BatteryWidget::SetLevels(int low, int critical)
{
    m_CriticalLevel = critical;
    m_LowLevel = low;
}

void BatteryWidget::SetInsideMargins(int left, int top, int right, int bottom)
{
    m_Margins = QMargins(left, top, right, bottom);
}

void BatteryWidget::resizeEvent(QResizeEvent* event)
{
    SetValue(m_Value);
    QWidget::resizeEvent(event);
}

void BatteryWidget::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);
    painter.save();
    if (m_Background)
        painter.drawPixmap(0, 0, m_Background->scaled(size()));

    if (m_State){
        int h = m_InsideSize.height();
        int w = m_InsideSize.width();
        w = (float)w / 100.0 * m_Perc;

        painter.drawPixmap(m_Margins.left(), m_Margins.top(), m_State->scaled(w, h));
        if (m_Charging && m_Critical){
            int tw = m_InsideSize.width() - w;
            painter.drawPixmap(m_Margins.left() + w, m_Margins.top(), m_Critical->scaled(tw, h));
        }
    }

    painter.setFont(font());
    painter.drawText(m_Margins.left(), m_Margins.top(), m_InsideSize.width(), m_InsideSize.height(), Qt::AlignCenter, m_ValueStr);

    painter.restore();
    QWidget::paintEvent(event);
}

void BatteryWidget::SetIsCharging(bool charging)
{
    m_Charging = charging;
}
