/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtGui module of the unofficial Maemo Qt Toolkit.
**
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the either Technology Preview License Agreement or the
** Beta Release License Agreement.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License versions 2.0 or 3.0 as published by the Free
** Software Foundation and appearing in the file LICENSE.GPL included in
** the packaging of this file.  Please review the following information
** to ensure GNU General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.  In addition, as a special
** exception, Nokia gives you certain additional rights. These rights
** are described in the Nokia Qt GPL Exception version 1.3, included in
** the file GPL_EXCEPTION.txt in this package.
**
** Qt for Windows(R) Licensees
** As a special exception, Nokia, as the sole copyright holder for Qt
** Designer, grants users of the Qt/Eclipse Integration plug-in the
** right for the Qt/Eclipse Integration to link to functionality
** provided by Qt Designer and its related libraries.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
****************************************************************************/

#include <qgtkstyle.h>
#include <qmaemo5style.h>
#include <private/qmaemo5style_p.h>

#include <private/qapplication_p.h>

#include <QtCore/QLibrary>
#include <QtGui/QStyleOption>
#include <QtGui/QLineEdit>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QAbstractScrollArea>
#include <QtGui/QAbstractItemView>
#include <QtGui/QScrollBar>
#include <QtGui/QTextEdit>
#include <QtGui/QButtonGroup>
#include <QtGui/QPushButton>
#include <QtGui/QRadioButton>
#include <QtGui/QX11Info>
#include <QtGui/QHBoxLayout>
#include <QtCore/QTimer>
#include <QtCore/QTimeLine>

#include <QtDebug>

#include "qpixmapcache.h"
#undef signals // Collides with GTK stymbols
#include "qgtkpainter_p.h"

#include <private/qt_x11_p.h>

#if defined(Q_WS_MAEMO_5)

QT_BEGIN_NAMESPACE

Ptr_gtk_widget_show QMaemo5StylePrivate::gtk_widget_show = 0;
Ptr_gtk_widget_set_name QMaemo5StylePrivate::gtk_widget_set_name = 0;
Ptr_gtk_text_view_new QMaemo5StylePrivate::gtk_text_view_new = 0;
Ptr_gtk_widget_get_size_request QMaemo5StylePrivate::gtk_widget_get_size_request = 0;
Ptr_gtk_button_set_label QMaemo5StylePrivate::gtk_button_set_label = 0;
Ptr_gtk_style_lookup_color QMaemo5StylePrivate::gtk_style_lookup_color = 0;

Ptr_hildon_init QMaemo5StylePrivate::hildon_init = 0;
Ptr_hildon_entry_new QMaemo5StylePrivate::hildon_entry_new = 0;
Ptr_hildon_number_editor_new QMaemo5StylePrivate::hildon_number_editor_new = 0;
Ptr_hildon_app_menu_new QMaemo5StylePrivate::hildon_app_menu_new = 0;
Ptr_hildon_app_menu_add_filter QMaemo5StylePrivate::hildon_app_menu_add_filter = 0;
Ptr_hildon_edit_toolbar_new_with_text QMaemo5StylePrivate::hildon_edit_toolbar_new_with_text = 0;
Ptr_hildon_button_new QMaemo5StylePrivate::hildon_button_new = 0;
Ptr_hildon_check_button_new QMaemo5StylePrivate::hildon_check_button_new = 0;
Ptr_hildon_pannable_area_new QMaemo5StylePrivate::hildon_pannable_area_new = 0;
Ptr_hildon_gtk_widget_set_theme_size QMaemo5StylePrivate::hildon_gtk_widget_set_theme_size = 0;
Ptr_hildon_dialog_new_with_buttons QMaemo5StylePrivate::hildon_dialog_new_with_buttons = 0;
Ptr_hildon_note_new_information QMaemo5StylePrivate::hildon_note_new_information = 0;

Ptr_hildon_file_chooser_dialog_new QMaemo5StylePrivate::hildon_file_chooser_dialog_new = 0;
Ptr_hildon_file_chooser_dialog_set_extension QMaemo5StylePrivate::hildon_file_chooser_dialog_set_extension = 0;
Ptr_hildon_file_chooser_dialog_add_extensions_combo QMaemo5StylePrivate::hildon_file_chooser_dialog_add_extensions_combo = 0;

QMap<QAbstractScrollArea *, ScrollBarFader *> QMaemo5StylePrivate::scrollBarFaders;
int QMaemo5StylePrivate::scrollBarFadeDelay = 3000;         // [ms] hardcoded in hildon pannable area
int QMaemo5StylePrivate::scrollBarFadeDuration = 400;       // [ms] hardcoded in hildon pannable area
int QMaemo5StylePrivate::scrollBarFadeUpdateInterval = 100; // [ms] hardcoded in hildon pannable area
GtkWidget *QMaemo5StylePrivate::radioButtonLeft = 0;
GtkWidget *QMaemo5StylePrivate::radioButtonMiddle = 0;
GtkWidget *QMaemo5StylePrivate::radioButtonRight = 0;


/*!
    \internal
    \reimp
*/
void QMaemo5StylePrivate::resolveGtk() const
{
    QGtkStylePrivate::resolveGtk();

    QLibrary libgtk(QLS("gtk-x11-2.0"), 0, 0);
    QLibrary libhildon(QLS("hildon-1"), 0, 0);
    QLibrary libhildonfm(QLS("hildonfm"), 2, 0);

    gtk_widget_show = (Ptr_gtk_widget_show)libgtk.resolve("gtk_widget_show");
    gtk_widget_set_name = (Ptr_gtk_widget_set_name)libgtk.resolve("gtk_widget_set_name");
    gtk_text_view_new = (Ptr_gtk_text_view_new)libgtk.resolve("gtk_text_view_new");
    gtk_widget_get_size_request = (Ptr_gtk_widget_get_size_request)libgtk.resolve("gtk_widget_get_size_request");
    gtk_button_set_label = (Ptr_gtk_button_set_label)libgtk.resolve("gtk_button_set_label");
    gtk_style_lookup_color = (Ptr_gtk_style_lookup_color)libgtk.resolve("gtk_style_lookup_color");

    hildon_init = (Ptr_hildon_init)libhildon.resolve("hildon_init");
    hildon_entry_new = (Ptr_hildon_entry_new)libhildon.resolve("hildon_entry_new");
    hildon_number_editor_new = (Ptr_hildon_number_editor_new)libhildon.resolve("hildon_number_editor_new");
    hildon_app_menu_new = (Ptr_hildon_app_menu_new)libhildon.resolve("hildon_app_menu_new");
    hildon_app_menu_add_filter = (Ptr_hildon_app_menu_add_filter)libhildon.resolve("hildon_app_menu_add_filter");
    hildon_edit_toolbar_new_with_text = (Ptr_hildon_edit_toolbar_new_with_text)libhildon.resolve("hildon_edit_toolbar_new_with_text");
    hildon_button_new = (Ptr_hildon_button_new)libhildon.resolve("hildon_button_new");
    hildon_check_button_new = (Ptr_hildon_check_button_new)libhildon.resolve("hildon_check_button_new");
    hildon_pannable_area_new = (Ptr_hildon_pannable_area_new)libhildon.resolve("hildon_pannable_area_new");
    hildon_gtk_widget_set_theme_size = (Ptr_hildon_gtk_widget_set_theme_size)libhildon.resolve("hildon_gtk_widget_set_theme_size");
    hildon_dialog_new_with_buttons = (Ptr_hildon_dialog_new_with_buttons)libhildon.resolve("hildon_dialog_new_with_buttons");
    hildon_note_new_information = (Ptr_hildon_note_new_information)libhildon.resolve("hildon_note_new_information");

    hildon_file_chooser_dialog_new = (Ptr_hildon_file_chooser_dialog_new)libhildonfm.resolve("hildon_file_chooser_dialog_new");
    hildon_file_chooser_dialog_set_extension = (Ptr_hildon_file_chooser_dialog_set_extension)libhildonfm.resolve("hildon_file_chooser_dialog_set_extension");
    hildon_file_chooser_dialog_add_extensions_combo = (Ptr_hildon_file_chooser_dialog_add_extensions_combo)libhildonfm.resolve("hildon_file_chooser_dialog_add_extensions_combo");

}

/*!
  \internal
  NOTE: Reverse color in fremantle themes are set only for
  GtkTextView widgets. GtkEntry returns a dark base color.
*/
GtkWidget* QMaemo5StylePrivate::getTextColorWidget() const
{
    return  gtkWidget(QLS("GtkTextView"));
}

/*!
    \internal
    \reimp
*/
void QMaemo5StylePrivate::initGtkMenu() const
{
    QGtkStylePrivate::initGtkMenu();

    // --- create widgets for the context sensitive menu
    GtkWidget *gtkMenu = gtk_menu_new();
    gtk_widget_set_name(gtkMenu, "hildon-context-sensitive-menu");
    gtk_widget_realize(gtkMenu);

    GtkWidget *gtkMenuItem = QGtkStylePrivate::gtk_menu_item_new();
    gtk_menu_shell_append((GtkMenuShell*)gtkMenu, gtkMenuItem);
    gtk_widget_realize(gtkMenuItem);

    GtkWidget *gtkCheckMenuItem = QGtkStylePrivate::gtk_check_menu_item_new();
    gtk_menu_shell_append((GtkMenuShell*)gtkMenu, gtkCheckMenuItem);
    gtk_widget_realize(gtkCheckMenuItem);

    GtkWidget *gtkMenuSeparator = QGtkStylePrivate::gtk_separator_menu_item_new();
    gtk_menu_shell_append((GtkMenuShell*)gtkMenu, gtkMenuSeparator);

    addAllSubWidgets(gtkMenu);

    // --- and now for the "normal" app menu
    GtkWidget *gtkMenu2 = hildon_app_menu_new();

    // -- three menu buttons. Sapwood will use different background images depending on their position.
    radioButtonLeft = gtk_radio_button_new(NULL);
    gtk_widget_show(radioButtonLeft); // only a visible button is really added.
    hildon_app_menu_add_filter( (HildonAppMenu*)gtkMenu2, (GtkButton*)radioButtonLeft);

    radioButtonMiddle = gtk_radio_button_new(NULL);
    gtk_widget_show(radioButtonMiddle); // only a visible button is really added.
    hildon_app_menu_add_filter( (HildonAppMenu*)gtkMenu2, (GtkButton*)radioButtonMiddle);

    radioButtonRight = gtk_radio_button_new(NULL);
    gtk_widget_show(radioButtonRight); // only a visible button is really added.
    hildon_app_menu_add_filter( (HildonAppMenu*)gtkMenu2, (GtkButton*)radioButtonRight);

    addAllSubWidgets(gtkMenu2);

    // and now a hack for the sapwood engine
    radioButtonLeft->allocation.x = 0;
    radioButtonLeft->allocation.y = 0;
    GTK_WIDGET_FLAGS(radioButtonLeft) |= GTK_MAPPED;
    radioButtonMiddle->allocation.x = 1;
    radioButtonMiddle->allocation.y = 0;
    GTK_WIDGET_FLAGS(radioButtonMiddle) |= GTK_MAPPED;
    radioButtonRight->allocation.x = 2;
    radioButtonRight->allocation.y = 0;
    GTK_WIDGET_FLAGS(radioButtonRight) |= GTK_MAPPED;
}

void QMaemo5StylePrivate::initGtkWidgets() const
{
    QGtkStylePrivate::initGtkWidgets();

    if (gtk_init && hildon_init) {
        hildon_init();

        if (!gtkWidgetMap()->contains(QLS("GtkWindow")))
            return; // without window all other widgets won't be created cleanly

        // ComboBoxes buttons
        addWidget(hildon_number_editor_new(0,1));
        addWidget(hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT));
        addWidget(gtk_text_view_new());
        addWidget(hildon_dialog_new_with_buttons("", NULL, GTK_DIALOG_MODAL, "", 0, NULL));
        addWidget(hildon_pannable_area_new());
        GtkWidget *note = hildon_note_new_information(NULL, "");
        gtk_widget_set_name(note, "HildonNote-InformationTheme");
        addWidget(note);

        // I need a tree view inside a pannable area
        GtkWidget *hildonPan = gtkWidget(QLatin1String("HildonPannableArea"));
        GtkWidget *gtkTreeView = gtk_tree_view_new();
        QGtkStylePrivate::gtk_container_add((GtkContainer*)hildonPan, gtkTreeView);
        addWidget(gtkTreeView);

        // Gtk bug - a GtkButton without a label won't even create it's child widgets
        GtkWidget *button = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
        gtk_button_set_label(reinterpret_cast<GtkButton *>(button), "");
        addWidget(button);
        button = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
        gtk_button_set_label(reinterpret_cast<GtkButton*>(button), "");
        addWidget(button);

        addWidget(hildon_edit_toolbar_new_with_text("", ""));
    }
}

extern QStringList qt_make_filter_list(const QString &filter);

void QMaemo5StylePrivate::setupGtkFileChooser(GtkWidget* gtkFileChooser, QWidget *parent,
                                const QString &dir, const QString &filter, QString *,
                                QFileDialog::Options options, bool isSaveDialog,
                                QMap<GtkFileFilter *, QString> *)
{
    g_object_set(gtkFileChooser, "do-overwrite-confirmation", gboolean(!(options & QFileDialog::DontConfirmOverwrite)), NULL);
    g_object_set(gtkFileChooser, "local_only", gboolean(true), NULL);
    if (!filter.isEmpty()) {
        QStringList filters = qt_make_filter_list(filter);

        if (isSaveDialog) {
            const int Nfilters = filters.count();
            char *ext_names[Nfilters];
            char *extensions[Nfilters];

            for (int i = 0; i < Nfilters; i++) {
                const QString &rawfilter = filters[i];
                QString name = rawfilter.left(rawfilter.indexOf(QLatin1Char('(')));
                QString extension = extract_filter(rawfilter)[0].remove(QLatin1String("*."));
                //Doesn't make sense adding '*' extension in a save dialog;
                if (extension.compare(QLatin1String("*")) == 0)
                    qWarning("'*' is not a valid extension for a save dialog");
                ext_names[i]= (char*) malloc(name.count()+1);
                memcpy(ext_names[i], qPrintable(name), name.count());
                extensions[i]= (char*) malloc(extension.count()+1);
                memcpy(extensions[i], qPrintable(extension), extension.count());
            }
            ext_names[Nfilters] = NULL;
            extensions[Nfilters] = NULL;

            GtkWidget* gtkExtensionCombo;
            gtkExtensionCombo = hildon_file_chooser_dialog_add_extensions_combo((HildonFileChooserDialog*)(gtkFileChooser),
                    extensions,ext_names);
            //gtk_signal_connect (GTK_OBJECT(gtkExtensionCombo), "changed",
            //                    GTK_SIGNAL_FUNC (qt_update_file_filter), NULL);
        } else {
            //Set "*.ext" filters in the open file dialogs. There we don't have a combobox to select a filter. :(
            //TODO the combobox in the HildonFileChooserDialog doesn't change the filter automatically. We have to se it
            //in Qt..
            GtkFileFilter *gtkFilter = gtk_file_filter_new ();
            foreach (const QString &rawfilter, filters) {
                QStringList extensions = extract_filter(rawfilter);
                foreach (const QString &fileExtension, extensions) {
                    gtk_file_filter_add_pattern (gtkFilter, qPrintable(fileExtension));
                }
            }
            g_object_set(gtkFileChooser, "filter", gtkFilter, NULL);
        }
    }

    // Using the currently active window is not entirely correct, however
    // it gives more sensible behavior for applications that do not provide a
    // parent
    QWidget *modalFor = parent ? parent->window() : qApp->activeWindow();
    if (modalFor) {
        gtk_widget_realize(gtkFileChooser); // Creates X window
        XSetTransientForHint(gdk_x11_drawable_get_xdisplay(gtkFileChooser->window),
                             gdk_x11_drawable_get_xid(gtkFileChooser->window),
                             modalFor->winId());
        gdk_x11_window_set_user_time (gtkFileChooser->window, QX11Info::appUserTime());
    }

    QFileInfo fileinfo(dir);
    if (dir.isEmpty())
        fileinfo.setFile(QDir::currentPath());
    fileinfo.makeAbsolute();
    if (fileinfo.isDir()) {
        gtk_file_chooser_set_current_folder((GtkFileChooser*)gtkFileChooser, qPrintable(dir));
    } else if (isSaveDialog) {
        gtk_file_chooser_set_current_folder((GtkFileChooser*)gtkFileChooser, qPrintable(fileinfo.absolutePath()));
        gtk_file_chooser_set_current_name((GtkFileChooser*)gtkFileChooser, qPrintable(fileinfo.fileName()));
    } else {
        gtk_file_chooser_set_filename((GtkFileChooser*)gtkFileChooser, qPrintable(dir));
    }
}

QString QMaemo5StylePrivate::openFilename(QWidget *parent, const QString &, const QString &dir, const QString &filter,
        QString *selectedFilter, QFileDialog::Options options)
{
    QMap<GtkFileFilter *, QString> filterMap;
    GtkWidget *gtkFileChooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_OPEN);

    setupGtkFileChooser(gtkFileChooser, parent, dir, filter, selectedFilter, options, false, &filterMap);

    QWidget modal_widget;
    modal_widget.setAttribute(Qt::WA_NoChildEventsForParent, true);
    modal_widget.setParent(parent, Qt::Window);
    QApplicationPrivate::enterModal(&modal_widget);

    QString filename;
    if (gtk_dialog_run ((GtkDialog*)gtkFileChooser) == GTK_RESPONSE_OK) {
        char *gtk_filename = gtk_file_chooser_get_filename ((GtkFileChooser*)gtkFileChooser);
        filename = QString::fromUtf8(gtk_filename);
        g_free (gtk_filename);
        if (selectedFilter) {
            GtkFileFilter *gtkFilter = gtk_file_chooser_get_filter ((GtkFileChooser*)gtkFileChooser);
            *selectedFilter = filterMap.value(gtkFilter);
        }
    }

    QApplicationPrivate::leaveModal(&modal_widget);
    gtk_widget_destroy (gtkFileChooser);
    return filename;
}


QString QMaemo5StylePrivate::openDirectory(QWidget *parent, const QString &, const QString &dir, QFileDialog::Options options)
{
    QMap<GtkFileFilter *, QString> filterMap;
    GtkWidget *gtkFileChooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

    setupGtkFileChooser(gtkFileChooser, parent, dir, QString(), 0, options);
    QWidget modal_widget;
    modal_widget.setAttribute(Qt::WA_NoChildEventsForParent, true);
    modal_widget.setParent(parent, Qt::Window);
    QApplicationPrivate::enterModal(&modal_widget);

    QString filename;
    if (gtk_dialog_run ((GtkDialog*)gtkFileChooser) == GTK_RESPONSE_OK) {
        char *gtk_filename = gtk_file_chooser_get_filename ((GtkFileChooser*)gtkFileChooser);
        filename = QString::fromUtf8(gtk_filename);
        g_free (gtk_filename);
    }

    QApplicationPrivate::leaveModal(&modal_widget);
    gtk_widget_destroy (gtkFileChooser);
    return filename;
}

QStringList QMaemo5StylePrivate::openFilenames(QWidget *parent, const QString &, const QString &dir, const QString &filter,
                                 QString *selectedFilter, QFileDialog::Options options)
{
    QStringList filenames;
    QMap<GtkFileFilter *, QString> filterMap;
    GtkWidget *gtkFileChooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_OPEN);

    setupGtkFileChooser(gtkFileChooser, parent, dir, filter, selectedFilter, options, false, &filterMap);
    g_object_set(gtkFileChooser, "select-multiple", gboolean(true), NULL);

    QWidget modal_widget;
    modal_widget.setAttribute(Qt::WA_NoChildEventsForParent, true);
    modal_widget.setParent(parent, Qt::Window);
    QApplicationPrivate::enterModal(&modal_widget);

    if (gtk_dialog_run ((GtkDialog*)gtkFileChooser) == GTK_RESPONSE_OK) {
        GSList *gtk_file_names = gtk_file_chooser_get_filenames((GtkFileChooser*)gtkFileChooser);
        for (GSList *iterator  = gtk_file_names ; iterator; iterator = iterator->next)
            filenames << QString::fromUtf8((const char*)iterator->data);
        g_slist_free(gtk_file_names);
        if (selectedFilter) {
            GtkFileFilter *gtkFilter = gtk_file_chooser_get_filter ((GtkFileChooser*)gtkFileChooser);
            *selectedFilter = filterMap.value(gtkFilter);
        }
    }

    QApplicationPrivate::leaveModal(&modal_widget);
    gtk_widget_destroy (gtkFileChooser);
    return filenames;
}

QString QMaemo5StylePrivate::saveFilename(QWidget *parent, const QString &, const QString &dir, const QString &filter,
                           QString *selectedFilter, QFileDialog::Options options)
{
    QMap<GtkFileFilter *, QString> filterMap;
    GtkWidget *gtkFileChooser = hildon_file_chooser_dialog_new (NULL, GTK_FILE_CHOOSER_ACTION_SAVE);
    setupGtkFileChooser(gtkFileChooser, parent, dir, filter, selectedFilter, options, true, &filterMap);

    QWidget modal_widget;
    modal_widget.setAttribute(Qt::WA_NoChildEventsForParent, true);
    modal_widget.setParent(parent, Qt::Window);
    QApplicationPrivate::enterModal(&modal_widget);

    QString filename;
    if (gtk_dialog_run ((GtkDialog*)gtkFileChooser) == GTK_RESPONSE_OK) {
        char *gtk_filename = gtk_file_chooser_get_filename ((GtkFileChooser*)gtkFileChooser);
        filename = QString::fromUtf8(gtk_filename);
        g_free (gtk_filename);
        if (selectedFilter) {
            GtkFileFilter *gtkFilter = gtk_file_chooser_get_filter ((GtkFileChooser*)gtkFileChooser);
            *selectedFilter = filterMap.value(gtkFilter);
        }
    }

    QApplicationPrivate::leaveModal(&modal_widget);
    gtk_widget_destroy (gtkFileChooser);
    return filename;
}


/*! \internal
 *  Returns one int value from the HildonAppMenu widget style.
 */
int QMaemo5StylePrivate::getAppMenuMetric( const char *metricName, int defaultValue ) const
{
    guint value = defaultValue;
    if (GtkWidget *hildonAppMenu = gtkWidget(QLS("HildonAppMenu")))
        gtk_widget_style_get(hildonAppMenu, metricName, &value, NULL);
    return value;
}


/*!
    \class QMaemo5Style
    \brief The QMaemo5Style class provides a widget style rendered by GTK+

    The QMaemo5Style style provides a look and feel that integrates well
    into Hildon-based desktop environments.

    It is largely based on QGtkStyle, but does some special magic for "hildonise" the look and feel.

    Note: The style requires GTK+ version 2.10 or later and the special patched GTK+ Hildon include files.
*/

/*!
    Constructs a QMaemo5Style object.
*/
QMaemo5Style::QMaemo5Style()
    : QGtkStyle(*new QMaemo5StylePrivate)
{
}

/*!
    Destroys the QMaemo5Style object.
*/
QMaemo5Style::~QMaemo5Style()
{
}

/*!
    \reimp
    Set the maemos style file dialog hooks.
*/
void QMaemo5Style::polish(QApplication *app)
{
    Q_D(QMaemo5Style);

    QGtkStyle::polish(app);

    if (d->hildon_init
            && app->desktopSettingsAware() && d->isThemeAvailable()) {
        // I want to use the maemo style dialogs
        qt_filedialog_open_filename_hook = &QMaemo5StylePrivate::openFilename;
        qt_filedialog_save_filename_hook = &QMaemo5StylePrivate::saveFilename;
        qt_filedialog_open_filenames_hook = &QMaemo5StylePrivate::openFilenames;
        qt_filedialog_existing_directory_hook = &QMaemo5StylePrivate::openDirectory;
    }
}


ScrollBarFader::ScrollBarFader(QAbstractScrollArea *area, int delay, int duration, int update_interval)
    : QObject(area), m_area(area), m_current_alpha(0.0)
{
    m_fade_timeline = new QTimeLine(duration, this);
    m_fade_timeline->setUpdateInterval(update_interval);
    connect(m_fade_timeline, SIGNAL(valueChanged(qreal)), this, SLOT(fade(qreal)));
    m_delay_timer = new QTimer(this);
    m_delay_timer->setInterval(delay);
    connect(m_delay_timer, SIGNAL(timeout()), this, SLOT(delayTimeout()));

    area->viewport()->installEventFilter(this);
}

ScrollBarFader::~ScrollBarFader()
{
}

bool ScrollBarFader::eventFilter(QObject *o, QEvent *e)
{
    if (o == m_area->viewport()) {
        switch (e->type()) {
        case QEvent::Show:
            m_fade_timeline->setDirection(QTimeLine::Forward);
            m_fade_timeline->start();
            m_delay_timer->start();
            break;

        case QEvent::MouseButtonPress:
        case QEvent::MouseMove:
        case QEvent::MouseButtonRelease:
        case QEvent::KeyPress:
        case QEvent::KeyRelease:
            if (m_fade_timeline->direction() == QTimeLine::Backward) {
                m_fade_timeline->setDirection(QTimeLine::Forward);
                m_fade_timeline->resume();
            }
            m_delay_timer->start();
            break;

        default:
            break;
        }
    }
    return QObject::eventFilter(o, e);
}

void ScrollBarFader::delayTimeout()
{
    m_fade_timeline->setDirection(QTimeLine::Backward);
    m_fade_timeline->resume();
    m_delay_timer->stop();
}

void ScrollBarFader::fade(qreal value)
{
    m_current_alpha = value;

    if (QScrollBar *hsb = m_area->horizontalScrollBar()) {
        if (hsb->isVisible()) {
            QStyleOptionSlider option;
            option.initFrom(hsb);
            option.subControls = QStyle::SC_All;
            hsb->update(); // ### hsb->style()->subControlRect(QStyle::CC_ScrollBar, &option, QStyle::SC_ScrollBarSlider, hsb));
        }
    }
    if (QScrollBar *vsb = m_area->verticalScrollBar()) {
        if (vsb->isVisible()) {
            QStyleOptionSlider option;
            option.initFrom(vsb);
            option.subControls = QStyle::SC_All;
            vsb->update(); // ### vsb->style()->subControlRect(QStyle::CC_ScrollBar, &option, QStyle::SC_ScrollBarSlider, vsb));
        }
    }
}

/*!
    \reimp
*/
void QMaemo5Style::polish(QWidget *widget)
{
    Q_D(QMaemo5Style);

    QGtkStyle::polish(widget);

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(widget)) {
        area->setFrameStyle(QFrame::NoFrame);

        d->scrollBarFaders.insert(area, new ScrollBarFader(area, d->scrollBarFadeDelay,
                                                                 d->scrollBarFadeDuration,
                                                                 d->scrollBarFadeUpdateInterval));

        if (!qobject_cast<QTextEdit *>(widget)) {
            QPalette palette = widget->palette();

            if (GtkWidget *treeView = d->gtkWidget(QLS("HildonPannableArea.GtkTreeView"))) {
                GdkColor gdkBg, gdkBase, gdkText, gdkForeground, gdkSbg, gdkSfg;
                QColor bg, base, text, fg, highlight, highlightText;
                gdkBg = treeView->style->bg[GTK_STATE_NORMAL];
                gdkForeground = treeView->style->fg[GTK_STATE_NORMAL];

                // Our base and selected color is primarily used for text
                // so we assume a treeView will have the most correct value
                gdkBase = treeView->style->base[GTK_STATE_NORMAL];
                gdkText = treeView->style->text[GTK_STATE_NORMAL];
                gdkSbg = treeView->style->base[GTK_STATE_SELECTED];
                gdkSfg = treeView->style->text[GTK_STATE_SELECTED];
                bg = QColor(gdkBg.red>>8, gdkBg.green>>8, gdkBg.blue>>8);
                text = QColor(gdkText.red>>8, gdkText.green>>8, gdkText.blue>>8);
                fg = QColor(gdkForeground.red>>8, gdkForeground.green>>8, gdkForeground.blue>>8);
                base = QColor(gdkBase.red>>8, gdkBase.green>>8, gdkBase.blue>>8);
                highlight = QColor(gdkSbg.red>>8, gdkSbg.green>>8, gdkSbg.blue>>8);
                highlightText = QColor(gdkSfg.red>>8, gdkSfg.green>>8, gdkSfg.blue>>8);

                palette.setColor(QPalette::HighlightedText, highlightText);
                palette.setColor(QPalette::Light, bg.lighter(125));
                palette.setColor(QPalette::Shadow, bg.darker(130));
                palette.setColor(QPalette::Dark, bg.darker(120));
                palette.setColor(QPalette::Text, text);
                palette.setColor(QPalette::WindowText, fg);
                palette.setColor(QPalette::ButtonText, fg);
                palette.setColor(QPalette::Base, base);

                widget->setPalette(palette);
            }
        }
    }

    if (widget && widget->parentWidget() && widget->parentWidget()->inherits("QMaemo5InformationBox")) {
        QString name = QLS("HildonNote-information-theme.GtkAlignment.GtkHBox.GtkVBox.GtkEventBox.GtkVBox.HildonNoteLabel-information-theme");
        if (d->gtkWidget(name))
            widget->setPalette(d->gtkWidgetPalette(name));
    }
}

void QMaemo5Style::unpolish(QWidget *widget)
{
    Q_D(QMaemo5Style);

    QGtkStyle::unpolish(widget);

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea*>(widget)) {
        delete d->scrollBarFaders.take(area);
    }
}

/*!
    \reimp
*/
int QMaemo5Style::pixelMetric(QStyle::PixelMetric metric,
                           const QStyleOption *option,
                           const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    if (!d->isThemeAvailable())
        return QGtkStyle::pixelMetric(metric, option, widget);

    switch (metric) {
    case PM_DefaultLayoutSpacing:
        return 8;
    case PM_DefaultTopLevelMargin:
        return 8;
    case PM_DialogButtonsButtonHeight:
        return 70;
    case PM_DialogButtonsButtonWidth:
        return 180;
    case PM_MessageBoxIconSize:
        return 0;
    case PM_ToolBarIconSize:
        return 48;
    case PM_ButtonShiftHorizontal: {
        guint horizontal_shift = 1;
        if (GtkWidget *gtkButton = d->gtkWidget(QLatin1String("HildonButton-finger")))
            d->gtk_widget_style_get(gtkButton, "child-displacement-x", &horizontal_shift, NULL);
        return horizontal_shift;
    }
    case PM_ButtonShiftVertical: {
        guint vertical_shift = 1;
        if (GtkWidget *gtkButton = d->gtkWidget(QLatin1String("HildonButton-finger")))
            d->gtk_widget_style_get(gtkButton, "child-displacement-y", &vertical_shift, NULL);
        return vertical_shift;
    }
    case PM_ScrollBarSliderMin: // fall through
    case PM_ScrollBarExtent: {
        guint indicator_width = 6;
        if (GtkWidget *hildonPan = d->gtkWidget(QLatin1String("HildonPannableArea")))
            d->gtk_widget_style_get(hildonPan, "indicator-width", &indicator_width, NULL);
        return indicator_width;
    }
    case PM_ScrollView_ScrollBarSpacing:
        return 0;

    case PM_ExclusiveIndicatorWidth:
    case PM_ExclusiveIndicatorHeight:
    case PM_IndicatorWidth:
    case PM_IndicatorHeight: {
        gint size = 48, spacing = 2;
        if (GtkWidget *hildonCheckButton = d->gtkWidget(QLS("HildonCheckButton-finger")))
            d->gtk_widget_style_get(hildonCheckButton, "image-spacing", &spacing, "checkbox-size", &size, NULL);
        return size + 2 * spacing;
    }
    case PM_Maemo5AppMenuHorizontalSpacing:
        return d->getAppMenuMetric("horizontal-spacing", 16);
    case PM_Maemo5AppMenuVerticalSpacing:
        return d->getAppMenuMetric("vertical-spacing", 16);
    case PM_Maemo5AppMenuContentMargin:
        return d->getAppMenuMetric("inner-border", 16);
    case PM_Maemo5AppMenuLandscapeXOffset:
        return d->getAppMenuMetric("external-border", 50);
    case PM_Maemo5AppMenuFilterGroupWidth:
        return d->getAppMenuMetric("filter-group-width", 444);
    case PM_Maemo5AppMenuFilterGroupVerticalSpacing:
        return d->getAppMenuMetric("filter-vertical-spacing", 16);

    case PM_Maemo5EditBarBackButtonWidth: {
        gint width = 112;
        if (GtkWidget *hildonEditBar = d->gtkWidget(QLS("toolbar-edit-mode")))
            d->gtk_widget_style_get(hildonEditBar, "arrow-width", &width, NULL);
        return width;
    }
    case PM_Maemo5EditBarBackButtonHeight: {
        gint height = 56;
        if (GtkWidget *hildonEditBar = d->gtkWidget(QLS("toolbar-edit-mode")))
            d->gtk_widget_style_get(hildonEditBar, "arrow-height", &height, NULL);
        return height;
    }

    default:
        return  QGtkStyle::pixelMetric(metric, option, widget);
    }
}

int QMaemo5Style::styleHint(StyleHint hint, const QStyleOption *option, const QWidget *widget,
                         QStyleHintReturn *returnData = 0) const
{
    Q_D(const QMaemo5Style);

    if (!d->isThemeAvailable())
        return QGtkStyle::styleHint(hint, option, widget, returnData);

    switch (hint) {
    case SH_Menu_Scrollable:
        return int(true);
    case SH_DialogButtonBox_ButtonsHaveIcons:
    case SH_ScrollBar_ContextMenu:
        return int(false);
    case SH_UnderlineShortcut:
        return int(false);
    case SH_RequestSoftwareInputPanel:
        return int(RSIP_OnMouseClick);
    case SH_Slider_AbsoluteSetButtons:
        return int(Qt::LeftButton);
    case SH_Slider_PageSetButtons:
        return int(Qt::NoButton);
    default:
        return QGtkStyle::styleHint(hint, option, widget, returnData);
    }
}

/*!
    \reimp
*/
void QMaemo5Style::drawPrimitive(QStyle::PrimitiveElement element,
                              const QStyleOption *option,
                              QPainter *painter,
                              const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    if (!d->isThemeAvailable()) {
        QGtkStyle::drawPrimitive(element, option, painter, widget);
        return;
    }

    QGtkPainter gtkPainter(painter);

    switch (element) {
    case PE_PanelButtonTool: // fall through
    case PE_PanelButtonCommand: {
        bool isTool = (element == PE_PanelButtonTool);
        bool isInDialog = (widget && qobject_cast<QDialogButtonBox *>(widget->parentWidget()));

        GtkStateType state = gtkPainter.gtkState(option);
        GtkShadowType shadow = GTK_SHADOW_OUT;
        if (option->state & State_On || option->state & State_Sunken) {
            state = GTK_STATE_ACTIVE;
            shadow = GTK_SHADOW_IN;
        }
        if (GtkWidget *gtkButton = d->gtkWidget(QLS(isTool ? "GtkToolButton.GtkButton"
                                                : isInDialog ? "HildonDialog.GtkAlignment.GtkHBox.hildon-dialog-action-area.GtkButton-finger"
                                                : "HildonButton-finger"))) {
            gtkPainter.paintBox(gtkButton, "button", option->rect, state, shadow,
                                gtkButton->style, QString());
        }
        break;
    }

    case PE_FrameFocusRect:
        // never required with Hildon
        break;

    case PE_PanelLineEdit:
        if (qstyleoption_cast<const QStyleOptionFrame *>(option)) {
            if (GtkWidget *gtkEntry = d->gtkWidget(QLS("HildonEntry-finger"))) {
                //HildonEntry-finger could be ythickness = 0; We MUST adjust the height of the rect.
                QRect textRect = option->rect.adjusted(gtkEntry->style->xthickness,
                                                       gtkEntry->style->ythickness ? gtkEntry->style->ythickness : gtkEntry->style->xthickness,
                                                       -gtkEntry->style->xthickness,
                                                       -(gtkEntry->style->ythickness ? gtkEntry->style->ythickness : gtkEntry->style->xthickness) );


                const QLineEdit *le = qobject_cast<const QLineEdit*>(widget);

                if (le && le->hasFrame()) {
                    gtkPainter.paintFlatBox(gtkEntry, "entry_bg", option->rect,
                                            option->state & State_Enabled ? GTK_STATE_NORMAL : GTK_STATE_INSENSITIVE, GTK_SHADOW_NONE, gtkEntry->style);
                    drawPrimitive(PE_FrameLineEdit, option, painter, widget);
                } else {
                    //QLineEdit used as Delegate widgets (Eg: in QTableView) should not have lineWidth in Hildon.
                    //In this case we have to cover the widget on the back.
                    QCleanlooksStyle::drawPrimitive(element, option, painter, widget);
                }
            }
        }
        break;

    case PE_FrameLineEdit:
        if (qstyleoption_cast<const QStyleOptionFrame *>(option)) {
            if (GtkWidget *gtkEntry = d->gtkWidget(QLS("HildonEntry-finger"))) {
                gboolean interior_focus = false;
                gint focus_line_width = 2;
                QRect rect = option->rect;
                d->gtk_widget_style_get(gtkEntry,
                                        "interior-focus",   &interior_focus,
                                        "focus-line-width", &focus_line_width, NULL);

                //See https://bugzilla.mozilla.org/show_bug.cgi?id=405421 for info about this hack
                g_object_set_data(G_OBJECT(gtkEntry), "transparent-bg-hint", GINT_TO_POINTER(TRUE));

                if (!interior_focus && option->state & State_HasFocus)
                    rect.adjust(focus_line_width, focus_line_width, -focus_line_width, -focus_line_width);


                gtkPainter.paintShadow(gtkEntry, "entry", rect, option->state & State_Enabled ?
                                       (option->state & State_HasFocus ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL) : GTK_STATE_INSENSITIVE,
                                       GTK_SHADOW_IN, gtkEntry->style,
                                       option->state & State_HasFocus ? QLS("focus") : QString());
            }
        }
        break;

    case PE_IndicatorCheckBox: {
        GtkShadowType shadow = GTK_SHADOW_OUT;
        GtkStateType state = gtkPainter.gtkState(option);

        if (option->state & State_Sunken)
            state = GTK_STATE_ACTIVE;

        if (option->state & State_NoChange)
            shadow = GTK_SHADOW_ETCHED_IN;
        else if (option->state & State_On)
            shadow = GTK_SHADOW_IN;
        else
            shadow = GTK_SHADOW_OUT;

        int spacing;

        if (GtkWidget *gtkCheckButton = d->gtkWidget(QLS("HildonCheckButton-finger"))) {
            GtkWidget *gtkCellView = d->gtkWidget(QLS("HildonCheckButton-finger.GtkAlignment.GtkHBox.GtkCellView"));
            // Some styles such as aero-clone assume they can paint in the spacing area
            gtkPainter.setClipRect(option->rect);

            d->gtk_widget_style_get(gtkCheckButton, "image-spacing", &spacing, NULL);

            QRect checkRect = option->rect.adjusted(spacing, spacing, -spacing, -spacing);

            gtkPainter.paintCheckbox(gtkCellView, checkRect, state, shadow, gtkCellView->style,
                                     QLS("checkbutton"));
        }
        break;
    }
    case PE_PanelItemViewItem: {
        if (GtkWidget *gtkTreeView = d->gtkWidget(QLS("HildonPannableArea.GtkTreeView"))) {
            const char *detail = "cell_odd_ruled";
            const QStyleOptionViewItemV4 *vopt = qstyleoption_cast<const QStyleOptionViewItemV4 *>(option);
            if (vopt && vopt->features & QStyleOptionViewItemV2::Alternate)
                detail = "cell_even_ruled";

            gtkPainter.paintFlatBox(gtkTreeView, detail, option->rect,
                    option->state & State_Selected ? GTK_STATE_SELECTED :
                    option->state & State_Enabled ? GTK_STATE_NORMAL : GTK_STATE_INSENSITIVE,
                    GTK_SHADOW_NONE, gtkTreeView->style);
        }
        break;
    }
    case PE_Maemo5InformationBox: {
        if (GtkWidget *hildonInformation = d->gtkWidget(QLS("HildonNote-information-theme")))
            gtkPainter.paintFlatBox(hildonInformation, NULL, option->rect, GTK_STATE_NORMAL, GTK_SHADOW_NONE, hildonInformation->style, QString());
        break;
    }
    case PE_Maemo5AppMenu: {
        if (GtkWidget *hildonAppMenu = d->gtkWidget(QLS("HildonAppMenu")))
            gtkPainter.paintFlatBox(hildonAppMenu, NULL, option->rect, GTK_STATE_NORMAL, GTK_SHADOW_NONE, hildonAppMenu->style, QString());
        break;
    }
    case PE_Maemo5EditBar: {
        if (GtkWidget *hildonEditBar = d->gtkWidget(QLS("toolbar-edit-mode")))
            gtkPainter.paintFlatBox(hildonEditBar, NULL, option->rect, GTK_STATE_NORMAL, GTK_SHADOW_NONE, hildonEditBar->style, QString());
        break;
    }
    case PE_Maemo5EditBarBackButton: {
        if (GtkWidget *hildonEditBarBackButton = d->gtkWidget(QLS("toolbar-edit-mode.hildon-edit-toolbar-arrow"))) {
            GtkStateType state = gtkPainter.gtkState(option);
            GtkShadowType shadow = GTK_SHADOW_OUT;
            if (option->state & State_On || option->state & State_Sunken) {
                state = GTK_STATE_ACTIVE;
                shadow = GTK_SHADOW_IN;
            }
            gtkPainter.paintBox(hildonEditBarBackButton, "button", option->rect, state, shadow, hildonEditBarBackButton->style, QString());
        }
        break;
    }
    default:
        QGtkStyle::drawPrimitive(element, option, painter, widget);
    }
}

/*!
    \reimp
*/
void QMaemo5Style::drawComplexControl(ComplexControl control, const QStyleOptionComplex *option,
                                   QPainter *painter, const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    if (!d->isThemeAvailable()) {
        QGtkStyle::drawComplexControl(control, option, painter, widget);
        return;
    }

    QGtkPainter gtkPainter(painter);

    switch (control) {
    case CC_ScrollBar:
        if (const QStyleOptionSlider *scrollBar = qstyleoption_cast<const QStyleOptionSlider *>(option)) {
            painter->fillRect(option->rect, option->palette.background());
            QRect scrollBarSlider = proxy()->subControlRect(control, scrollBar, SC_ScrollBarSlider, widget);
            QColor color = option->palette.color(QPalette::Text);

            if (widget && widget->parentWidget()) {
                if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea*>(widget->parentWidget()->parentWidget())) {
                    if (ScrollBarFader *fader = d->scrollBarFaders.value(area)) {
                        color.setAlphaF(fader->currentAlpha());
                    }
                }
            }

            painter->fillRect(scrollBarSlider, color);
        }
        break;

    default:
        QGtkStyle::drawComplexControl(control, option, painter, widget);
    }
}

static QHBoxLayout *findHBoxLayoutContaining(const QWidget *widget, QLayout *lay)
{
    for (int i = 0; i < lay->count(); ++i) {
        if (QLayout *sublay = lay->itemAt(i)->layout()) {
            if (QHBoxLayout *box = findHBoxLayoutContaining(widget, sublay))
                return box;
        } else if (lay->itemAt(i)->widget() == widget) {
            return qobject_cast<QHBoxLayout *>(lay);
        }
    }
    return 0;
}

/*!
    \reimp
*/
void QMaemo5Style::drawControl(ControlElement element,
                               const QStyleOption *option,
                               QPainter *painter,
                               const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    if (!d->isThemeAvailable()) {
        QCleanlooksStyle::drawControl(element, option, painter, widget);
        return;
    }

    QGtkPainter gtkPainter(painter);

    switch (element) {
    case CE_RadioButton:
        if (const QStyleOptionButton *btn = qstyleoption_cast<const QStyleOptionButton *>(option)) {
            bool handled = false;

            if (qobject_cast<const QRadioButton *>(widget) && widget->parentWidget() && widget->parentWidget()->layout()) {
                if (QHBoxLayout *box = findHBoxLayoutContaining(widget, widget->parentWidget()->layout())) {
                    QList<const QRadioButton *> buttonList;
                    for (int i = 0; i < box->count(); ++i) {
                        if (QRadioButton *radio = qobject_cast<QRadioButton *>(box->itemAt(i)->widget()))
                            buttonList << radio;
                    }

                    GtkWidget *gtkButton = 0;
                    int pos = buttonList.indexOf(qobject_cast<const QRadioButton *>(widget));
                    if (pos == 0)
                        gtkButton = d->radioButtonLeft;
                    else if (pos == buttonList.count() - 1)
                        gtkButton = d->radioButtonRight;
                    else
                        gtkButton = d->radioButtonMiddle;

                    GtkStateType state = gtkPainter.gtkState(option);
                    GtkShadowType shadow = GTK_SHADOW_OUT;
                    if (option->state & State_On || option->state & State_Sunken) {
                        state = GTK_STATE_ACTIVE;
                        shadow = GTK_SHADOW_IN;
                    }
                    if (gtkButton) {
                        gtkPainter.paintBox(gtkButton, "button", option->rect, state, shadow,
                                            gtkButton->style, QString());
                    }
                    handled = true;
                }
            }
            if (!handled)
                proxy()->drawControl(CE_PushButtonBevel, btn, painter, widget);

            QStyleOptionButton subopt = *btn;
            subopt.rect = proxy()->subElementRect(SE_RadioButtonContents, btn, widget);
            proxy()->drawControl(CE_RadioButtonLabel, &subopt, painter, widget);
        }
        break;

    case CE_CheckBox:
        if (const QStyleOptionButton *btn = qstyleoption_cast<const QStyleOptionButton *>(option)) {
            QStyleOptionButton bevelopt = *btn;
            bevelopt.state &= ~State_On;
            proxy()->drawControl(CE_PushButtonBevel, &bevelopt, painter, widget);
            QStyleOptionButton subopt = *btn;
            subopt.rect = proxy()->subElementRect(SE_CheckBoxIndicator, btn, widget);
            proxy()->drawPrimitive(PE_IndicatorCheckBox, &subopt, painter, widget);
            subopt.rect = subElementRect(SE_CheckBoxContents, btn, widget);
            proxy()->drawControl(CE_CheckBoxLabel, &subopt, painter, widget);
        }
        break;

    case CE_PushButton:
        if (const QStyleOptionButton *btn = qstyleoption_cast<const QStyleOptionButton *>(option)) {
            proxy()->drawControl(CE_PushButtonBevel, btn, painter, widget);
            QStyleOptionButton subopt = *btn;
            subopt.rect = subElementRect(SE_PushButtonContents, btn, widget);
            proxy()->drawControl(CE_PushButtonLabel, &subopt, painter, widget);
        }
        break;

    case CE_RadioButtonLabel: // fall through
    case CE_PushButtonLabel:
        if (const QStyleOptionMaemo5ValueButton *button = qstyleoption_cast<const QStyleOptionMaemo5ValueButton *>(option)) {
            QRect ir = button->rect;
            uint tf = Qt::AlignVCenter | Qt::TextHideMnemonic | Qt::AlignLeft;
            int spacing = 8;
            QRect textRect, valueRect, iconRect;

            if (!button->text.isEmpty())
                textRect = ir;
            if (!button->value.isEmpty())
                valueRect = ir;
            if (!button->icon.isNull()) {
                iconRect = ir;
                iconRect.setTop(ir.top() + (ir.height() - button->iconSize.height()) / 2);
                iconRect.setWidth(button->iconSize.width());
                iconRect.setHeight(button->iconSize.height());
                textRect.setLeft(iconRect.right() + spacing);
                valueRect.setLeft(textRect.left());
            }

            QFont valuefont = painter->font();
            QFontMetrics valuefm = button->fontMetrics;

            if (button->styles & QStyleOptionMaemo5ValueButton::ValueUnderText) {
                valuefont.setPointSize(valuefont.pointSize() * 13 / 18);
                valuefm = QFontMetrics(valuefont);
            }

            QSize textSize = button->fontMetrics.size(Qt::TextSingleLine, button->text);
            QSize valueSize = valuefm.size(Qt::TextSingleLine, button->value);

            if (textRect.isValid())
                textRect.setWidth(textSize.width());
            if (valueRect.isValid())
                valueRect.setWidth(valueSize.width());

            if (button->styles & QStyleOptionMaemo5ValueButton::ValueUnderText) {
                if (valueRect.isValid() && textRect.isValid()) {
                    int delta = (ir.height() - (textSize.height() + 4 + valueSize.height())) / 2;
                    textRect.setTop(textRect.top() + delta);
                    textRect.setHeight(textSize.height());
                    valueRect.setTop(textRect.bottom() + 4);
                    valueRect.setHeight(valueSize.height());
                }
                if (button->styles & QStyleOptionMaemo5ValueButton::Centered) {
                    int deltaText = (textRect.width() - valueRect.width()) / 2;
                    int offsetX = ir.left() + (ir.width() - ((iconRect.isValid() ? iconRect.width() + spacing : 0) + qMax(textRect.width(), valueRect.width()))) / 2;
                    if (iconRect.isValid()) {
                        iconRect.moveLeft(offsetX);
                        offsetX += (iconRect.width() + spacing);
                    }
                    if (textRect.isValid())
                        textRect.moveLeft(offsetX + (deltaText > 0 ? 0 : -deltaText));
                    if (valueRect.isValid())
                        valueRect.moveLeft(offsetX + (deltaText < 0 ? 0 : deltaText));
                }
            } else {
                int middle = ir.center().x();
                if (textRect.isValid() && valueRect.isValid()) {
                    if ((textRect.right() + spacing) < middle)
                        valueRect.moveLeft(middle);
                    else
                        valueRect.moveLeft(textRect.right() + spacing);
                }
            }
            valueRect = visualRect(button->direction, ir, valueRect & ir);
            textRect = visualRect(button->direction, ir, textRect & ir);
            iconRect = visualRect(button->direction, ir, iconRect & ir);

            //qDebug() << "Icon, Text, Value: " << iconRect << textRect << valueRect;

            if (iconRect.isValid()) {
                QIcon::Mode mode = button->state & State_Enabled ? QIcon::Normal
                                                              : QIcon::Disabled;
                QIcon::State state = QIcon::Off;
                if (button->state & State_On)
                    state = QIcon::On;

                QPixmap pixmap = button->icon.pixmap(button->iconSize, mode, state);
                painter->drawPixmap(iconRect, pixmap);
            }
            if (textRect.isValid()) {
                proxy()->drawItemText(painter, textRect, tf, button->palette, (button->state & State_Enabled),
                                      button->text, QPalette::ButtonText);
            }
            if (valueRect.isValid()) {
                QPalette pal = button->palette;
                if (!(button->state & (State_On | State_Sunken))) {
                    GdkColor gdkVc;
                    bool ispick = button->styles & QStyleOptionMaemo5ValueButton::PickButton;
                    if (d->gtk_style_lookup_color(d->gtkStyle(QLS("HildonButton-finger")), ispick ? "ActiveTextColor" : "SecondaryTextColor", &gdkVc))
                        pal.setBrush(QPalette::ButtonText, QColor(gdkVc.red>>8, gdkVc.green>>8, gdkVc.blue>>8));
                }
                QFont oldfont = painter->font();
                painter->setFont(valuefont);

                proxy()->drawItemText(painter, valueRect, tf, pal, (button->state & State_Enabled),
                                      button->value, QPalette::ButtonText);
                painter->setFont(oldfont);
            }
        } else if (const QStyleOptionButton *button = qstyleoption_cast<const QStyleOptionButton *>(option)) {
            QRect ir = button->rect;
            uint tf = Qt::AlignVCenter | Qt::TextHideMnemonic;

            if (!button->icon.isNull()) {
                //Center both icon and text
                QPoint point;

                QIcon::Mode mode = button->state & State_Enabled ? QIcon::Normal
                                                              : QIcon::Disabled;
                QIcon::State state = QIcon::Off;
                if (button->state & State_On)
                    state = QIcon::On;

                QPixmap pixmap = button->icon.pixmap(button->iconSize, mode, state);
                int w = pixmap.width();
                int h = pixmap.height();

                if (!button->text.isEmpty())
                    w += button->fontMetrics.boundingRect(option->rect, tf, button->text).width() + 2;

                point = QPoint(ir.x() + ir.width() / 2 - w / 2,
                               ir.y() + ir.height() / 2 - h / 2);

                if (button->direction == Qt::RightToLeft)
                    point.rx() += pixmap.width();

                painter->drawPixmap(visualPos(button->direction, button->rect, point), pixmap);

                if (button->direction == Qt::RightToLeft)
                    ir.translate(-point.x() - 2, 0);
                else
                    ir.translate(point.x() + pixmap.width(), 0);

                // left-align text if there is
                if (!button->text.isEmpty())
                    tf |= Qt::AlignLeft;

            } else {
                tf |= Qt::AlignHCenter;
            }

            if (button->features & QStyleOptionButton::HasMenu)
                ir = ir.adjusted(0, 0, -pixelMetric(PM_MenuButtonIndicator, button, widget), 0);
            proxy()->drawItemText(painter, ir, tf, button->palette, (button->state & State_Enabled),
                         button->text, QPalette::ButtonText);
        }
        break;

    default:
        QGtkStyle::drawControl(element, option, painter, widget);
        break;
    }
}

/*!
    \reimp
*/
QRect QMaemo5Style::subControlRect(ComplexControl control, const QStyleOptionComplex *option,
                                   SubControl subControl, const QWidget *widget) const
{
    QRect r;
    switch (control) {
    case CC_ScrollBar:
        if (const QStyleOptionSlider *scrollbar = qstyleoption_cast<const QStyleOptionSlider *>(option)) {
            const QRect scrollBarRect = scrollbar->rect;
            int maxlen = ((scrollbar->orientation == Qt::Horizontal) ?
                          scrollBarRect.width() : scrollBarRect.height());
            int sliderlen;

            // calculate slider length
            if (scrollbar->maximum != scrollbar->minimum) {
                uint range = scrollbar->maximum - scrollbar->minimum;
                sliderlen = (qint64(scrollbar->pageStep) * maxlen) / (range + scrollbar->pageStep);

                int slidermin = proxy()->pixelMetric(PM_ScrollBarSliderMin, scrollbar, widget);
                if (sliderlen < slidermin || range > INT_MAX / 2)
                    sliderlen = slidermin;
                if (sliderlen > maxlen)
                    sliderlen = maxlen;
            } else {
                sliderlen = maxlen;
            }

            int sliderstart = sliderPositionFromValue(scrollbar->minimum,
                                                      scrollbar->maximum,
                                                      scrollbar->sliderPosition,
                                                      maxlen - sliderlen,
                                                      scrollbar->upsideDown);

            switch (subControl) {
            case SC_ScrollBarSubLine:            // top/left button
            case SC_ScrollBarAddLine:            // bottom/right button
                r = QRect();
                break;
            case SC_ScrollBarSubPage:            // between top/left button and slider
                if (scrollbar->orientation == Qt::Horizontal)
                    r.setRect(0, 0, sliderstart, scrollBarRect.height());
                else
                    r.setRect(0, 0, scrollBarRect.width(), sliderstart);
                break;
            case SC_ScrollBarAddPage:            // between bottom/right button and slider
                if (scrollbar->orientation == Qt::Horizontal)
                    r.setRect(sliderstart + sliderlen, 0,
                              maxlen - sliderstart - sliderlen, scrollBarRect.height());
                else
                    r.setRect(0, sliderstart + sliderlen, scrollBarRect.width(),
                              maxlen - sliderstart - sliderlen);
                break;
            case SC_ScrollBarGroove:
                r = scrollBarRect;
                break;
            case SC_ScrollBarSlider:
                if (scrollbar->orientation == Qt::Horizontal)
                    r.setRect(sliderstart, 0, sliderlen, scrollBarRect.height());
                else
                    r.setRect(0, sliderstart, scrollBarRect.width(), sliderlen);
                break;
            default:
                break;
            }
            r = visualRect(scrollbar->direction, scrollBarRect, r);
        }
        break;

    default:
        r = QGtkStyle::subControlRect(control, option, subControl, widget);
        break;
    }
    return r;
}

/*!
    \reimp
*/
QSize QMaemo5Style::sizeFromContents(ContentsType type, const QStyleOption *option,
                                  const QSize &size, const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    QSize newSize = QGtkStyle::sizeFromContents(type, option, size, widget);
    if (!d->isThemeAvailable())
        return newSize;

    GtkWidget *gtkWidget = 0;

    switch (type) {

    case CT_ToolButton:
        if (const QStyleOptionToolButton *toolbutton = qstyleoption_cast<const QStyleOptionToolButton *>(option)) {
            // tool buttons are always at least as high as the icon (even if it only has a text)
            // the 78/70 is a hack, since these values are also hardcoded in Hildon
            newSize = newSize.expandedTo(toolbutton->iconSize + QSize(12, 12)).expandedTo(QSize(78, 70));
        }
        break;

    case CT_LineEdit: {
        const QLineEdit *le = qobject_cast<const QLineEdit*>(widget);
        if (le && !le->hasFrame())
            break;
        gtkWidget = d->gtkWidget(QLS("HildonEntry-finger"));
        if (gtkWidget) {
            newSize = size + QSize(2*gtkWidget->style->xthickness,
                                   2*qMax(gtkWidget->style->ythickness, gtkWidget->style->xthickness));
        }
        break;
    }

    case CT_PushButton: {
        if (widget && qobject_cast<QDialogButtonBox *>(widget->parentWidget()))
            newSize.setWidth(qMax(newSize.width(), 174)); // hardcoded value in gtkdialog.c
        gtkWidget = d->gtkWidget(QLS("HildonButton-finger"));
        break;
    }

    case CT_RadioButton:
    case CT_CheckBox: {
        gtkWidget = d->gtkWidget(QLS("HildonCheckButton-finger"));
        break;
    }

    case CT_MenuItem: {
        if (const QStyleOptionMenuItem *menuItem = qstyleoption_cast<const QStyleOptionMenuItem *>(option)) {
            int textMargin = 8;

            if (menuItem->menuItemType == QStyleOptionMenuItem::Separator) {
                gboolean wide_separators = true;
                gint     separator_height = 8;
                if (GtkWidget *gtkMenuSeparator = d->gtkWidget(QLS("hildon-context-sensitive-menu.GtkSeparatorMenuItem"))) {
                    d->gtk_widget_style_get(gtkMenuSeparator,
                                           "wide-separators",    &wide_separators,
                                           "separator-height",   &separator_height,
                                           NULL);
                }
                newSize = QSize(size.width(), wide_separators ? separator_height - 1 : 7 );
                break;
            }

            if (GtkWidget *gtkMenuItem = d->gtkWidget(QLS("hildon-context-sensitive-menu.GtkMenuItem"))) {
                GtkStyle* style = gtkMenuItem->style;
                newSize += QSize(textMargin + style->xthickness - 2, style->ythickness - 4);

                // Cleanlooks assumes a check column of 20 pixels so we need to
                // expand it a bit
                gint checkSize;
                d->gtk_widget_style_get(d->gtkWidget(QLS("hildon-context-sensitive-menu.GtkCheckMenuItem")), "indicator-size", &checkSize, NULL);
                newSize.setHeight(qMax(newSize.height(), checkSize + 2));
                newSize.setWidth(newSize.width() + qMax(0, checkSize - 20));
            }
        }
        break;
    }

    case CT_ItemViewItem: {
         uint rowHeight = 70;
         if (GtkWidget *gtkTreeView = d->gtkWidget(QLS("HildonPannableArea.GtkTreeView")))
             d->gtk_widget_style_get(gtkTreeView, "row-height", &rowHeight, NULL);
         newSize = newSize.expandedTo(QSize(0, rowHeight));
         break;
    }

    case CT_ProgressBar: {
        gint minw = 70, minh = 70;
        if (GtkWidget *gtkProgressBar = d->gtkWidget(QLS("GtkProgressBar"))) {
            d->gtk_widget_style_get(gtkProgressBar,
                                   "min-vertical-bar-width",    &minw,
                                   "min-horizontal-bar-height", &minh,
                                   NULL);
        }
        newSize = newSize.expandedTo(QSize(minw, minh));
        break;
    }

    case CT_Slider: {
        if (const QStyleOptionSlider *slider = qstyleoption_cast<const QStyleOptionSlider *>(option)) {
            // I have no idea why the Gtk widget ends up being 70px high given the Hildon gtkrc and the gtkscale source code...
            QSize expandFix(0, 70);
            if (slider->orientation == Qt::Horizontal) {
                gtkWidget = d->gtkWidget(QLS("GtkHScale"));
            } else {
                gtkWidget = d->gtkWidget(QLS("GtkVScale"));
                expandFix.transpose();
            }
            newSize = newSize.expandedTo(expandFix);
        }
        break;
    }

    default:
        break;
    }

    if (gtkWidget) {
        gint w = -1, h = -1;
        d->gtk_widget_get_size_request(gtkWidget, &w, &h);
        newSize.setHeight(qMax(newSize.height(), h));
        newSize.setWidth(qMax(newSize.width(), w));
    }
    return newSize;
}

/*!
    \reimp
*/
QRect QMaemo5Style::subElementRect(SubElement element, const QStyleOption *option, const QWidget *widget) const
{
    Q_D(const QMaemo5Style);

    QRect r;
    switch (element) {
    case SE_LineEditContents: {
        const QLineEdit *le = qobject_cast<const QLineEdit*>(widget);
        if (le && !le->hasFrame())
            return QGtkStyle::subElementRect(element, option, widget);

        if (const QStyleOptionFrame *f = qstyleoption_cast<const QStyleOptionFrame *>(option)) {
            if (GtkWidget *gtkEntry = d->gtkWidget(QLS("HildonEntry-finger"))) {
                int x = qMax(gtkEntry->style->xthickness, f->lineWidth);
                int y = qMax(gtkEntry->style->ythickness, f->lineWidth);
                r = f->rect.adjusted(x,y, -x, -y);
                r = visualRect(option->direction, option->rect, r);
            }
        }
    }
    break;

    case SE_CheckBoxIndicator: {
        int h = proxy()->pixelMetric(PM_IndicatorHeight, option, widget);
        r.setRect(option->rect.x(), option->rect.y() + ((option->rect.height() - h) / 2),
                  proxy()->pixelMetric(PM_IndicatorWidth, option, widget), h);
        r = visualRect(option->direction, option->rect, r);
        break;
    }
    case SE_RadioButtonContents:
        r = proxy()->subElementRect(SE_PushButtonContents, option, widget);
        break;

    case SE_RadioButtonClickRect: // fall through
    case SE_CheckBoxClickRect:
        r = visualRect(option->direction, option->rect, option->rect);
        break;

    default:
        r = QGtkStyle::subElementRect(element, option, widget);
    }
    return r;
}

QStyle::SubControl QMaemo5Style::hitTestComplexControl(ComplexControl cc, const QStyleOptionComplex *opt,
                              const QPoint &pt, const QWidget *w) const
{
    switch (cc) {
    case CC_ScrollBar:
        return SC_None;
    default:
        return QGtkStyle::hitTestComplexControl(cc, opt, pt, w);
    }
}

/*!
    \enum QMaemo5Style::PrimitiveElement
    \since 4.6

    This enum contains additional Maemo 5 specific PrimitiveElement entries.

    \value PE_Maemo5InformationBox Represends an Maemo 5 information box overlay

    \value PE_Maemo5AppMenu        Represends an Maemo 5 application menu frame

    \value PE_Maemo5EditBar        Represends the special Maemo 5 edit bar popup background

    \value PE_Maemo5EditBarBackButton  Represends the back button for the edit bar

    \sa QStyle::PrimitiveElement
*/

/*!
    \enum QMaemo5Style::PixelMetric
    \since 4.6

    This enum contains additional Maemo 5 specific PixelMetric entries.

    \value PM_Maemo5AppMenuHorizontalSpacing The horizontal space between entries inside the Maemo 5 app menu

    \value PM_Maemo5AppMenuVerticalSpacing   The vertical space between entries inside the Maemo 5 app menu

    \value PM_Maemo5AppMenuContentMargin     The space around the Maemo 5 app menu content

    \value PM_Maemo5AppMenuLandscapeXOffset  The space from the edge of the screen to the menu border

    \value PM_Maemo5AppMenuFilterGroupWidth   The width of the Maemo5 menu filter group.

    \value PM_Maemo5AppMenuFilterGroupVerticalSpacing The space between the filter group and the rest of the Maemo5 menu.

    \value PM_Maemo5EditBarBackButtonWidth   The width of the Maemo5 edit bar back button

    \value PM_Maemo5EditBarBackButtonHeight  The height of the Maemo5 edit bar back button

    \sa QStyle::PixelMetric
*/

QT_END_NAMESPACE

#endif

