/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** No Commercial Usage
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**
**
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include <qmaemo5kineticscroller.h>
#include <private/qmaemo5kineticscroller_p.h>

#include <QGraphicsView>
#include <QScrollBar>
#include <QAbstractItemView>
#include <QCoreApplication>

#include <QtDebug>

QT_BEGIN_NAMESPACE

//#define KINETIC_SCROLLER_DEBUG

#ifdef KINETIC_SCROLLER_DEBUG
#  define qKSDebug  qDebug
#else
#  define qKSDebug  while (false) qDebug
#endif

QMaemo5KineticScroller::QMaemo5KineticScroller(QWidget *w)
    : QObject(w), d_ptr(new QMaemo5KineticScrollerPrivate())
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

QMaemo5KineticScroller::QMaemo5KineticScroller(QAbstractScrollArea *w)
    : QObject(w), d_ptr(new QMaemo5KineticScrollerPrivate())
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

QMaemo5KineticScroller::QMaemo5KineticScroller(QMaemo5KineticScrollerPrivate &dd, QWidget *w)
    : QObject(w), d_ptr(&dd)
{
    Q_D(QMaemo5KineticScroller);
    d->q_ptr = this;
    d->init(w);
}

QMaemo5KineticScroller::~QMaemo5KineticScroller()
{
}

bool QMaemo5KineticScroller::eventFilter(QObject *o, QEvent *e)
{
    Q_D(QMaemo5KineticScroller);

    bool res = false;

    if (e != d->ignoreEvent) {
        if (o == d->filterWidget) {
            switch (e->type()) {
            case QEvent::MouseButtonPress:
                res = d->handleMousePress(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::MouseMove:
                res = d->handleMouseMove(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::MouseButtonRelease:
                res = d->handleMouseRelease(static_cast<QMouseEvent *>(e));
                break;
            case QEvent::ChildAdded:
            case QEvent::ChildRemoved:
                d->handleChildEvent(static_cast<QChildEvent *>(e));
                break;
            default:
                break;
            }
        }
    }
    d->ignoreEvent = 0;
    return res ? true : QObject::eventFilter(o, e);
}

void QMaemo5KineticScroller::timerEvent(QTimerEvent *te)
{
    Q_D(QMaemo5KineticScroller);

    if (te->timerId() == d->idleTimerId)
        d->handleIdleTimer();
    else if (te->timerId() == d->scrollTimerId)
        d->handleScrollTimer();
}

QMaemo5KineticScrollerPrivate::QMaemo5KineticScrollerPrivate()
    : mode(QMaemo5KineticScroller::AutoMode),
    lastType(QEvent::User), buttonPressed(false), moved(false), lastIn(true),
    firstDrag(true), centerOnChildFocusPending(false), lowFrictionMode(false),
    scrollTo(-1, -1), bounceSteps(3), maxOverShoot(150, 150),
    minVelocity(10), maxVelocity(3500), fastVelocityFactor(0.01), deceleration(0.85),
    scrollsPerSecond(20), panningThreshold(25), directionErrorMargin(10), force(50),
    dragInertia(0.85), scrollTime(1000),
    childItem(0), childWidget(0),
    idleTimerId(0), scrollTimerId(0), widget(0), filterWidget(0), ignoreEvent(0)
{
}

void QMaemo5KineticScrollerPrivate::init(QWidget *w)
{
    Q_Q(QMaemo5KineticScroller);

    widget = w;

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(w)) {
        if (QAbstractItemView *itemview = qobject_cast<QAbstractItemView *>(area)) {
            itemview->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);
            itemview->setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
        }
        filterWidget = area->viewport();
    } else {
        filterWidget = w;
    }
    if (filterWidget)
        filterWidget->installEventFilter(q);
}

void QMaemo5KineticScrollerPrivate::sendEvent(QWidget *w, QEvent *e)
{
    ignoreEvent = e;
    QCoreApplication::sendEvent(w, e);
}

QRect QMaemo5KineticScroller::positionRange() const
{
    Q_D(const QMaemo5KineticScroller);

    QRect r;
    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        if (QAbstractSlider *s = area->horizontalScrollBar()) {
            r.setLeft(s->minimum());
            r.setRight(s->maximum());
        }
        if (QAbstractSlider *s = area->verticalScrollBar()) {
            r.setTop(s->minimum());
            r.setBottom(s->maximum());
        }
    }
    return r;
}

QPoint QMaemo5KineticScroller::position() const
{
    Q_D(const QMaemo5KineticScroller);

    QPoint p;
    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        if (QAbstractSlider *s = area->horizontalScrollBar())
            p.setX(s->value());
        if (QAbstractSlider *s = area->verticalScrollBar())
            p.setY(s->value());
    }
    return p;
}

void QMaemo5KineticScroller::setPosition(const QPoint &pos)
{
    Q_D(QMaemo5KineticScroller);

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(d->widget)) {
        if (QAbstractSlider *s = area->horizontalScrollBar())
            s->setValue(pos.x());
        if (QAbstractSlider *s = area->verticalScrollBar())
            s->setValue(pos.y());
    }
}

void QMaemo5KineticScrollerPrivate::handleChildEvent(QChildEvent *e)
{
    if (e->child()->isWidgetType())
        static_cast<QWidget *>(e->child())->setAttribute(Qt::WA_TransparentForMouseEvents, e->added());
}

void QMaemo5KineticScrollerPrivate::handleScrollTimer()
{
    Q_Q(QMaemo5KineticScroller);

    qKSDebug("handleScrollTimer: %d/%d", motion.x(), motion.y());

    if (!motion.isNull())
        q->setPosition(q->position() - motion);
    q->killTimer(scrollTimerId);
    scrollTimerId = 0;
}

void QMaemo5KineticScrollerPrivate::handleIdleTimer()
{
    Q_Q(QMaemo5KineticScroller);

    if (mode == QMaemo5KineticScroller::PushMode) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
        return;
    }
    qKSDebug() << "idle timer - velocity: " << velocity;

    q->setPosition(q->position() - velocity.toPoint());

    if (!buttonPressed) {
        // Decelerate gradually when pointer is raised
        if (overShootDist.isNull()) {
            // in case we move to a specific point do not decelerate when arriving
            if (scrollTo.x() != -1 || scrollTo.y() != -1) {
                if (qAbs(velocity.x()) >= qreal(1.5))
                    velocity.rx() *= deceleration;
                if (qAbs(velocity.y()) >= qreal(1.5))
                    velocity.ry() *= deceleration;
            } else {
                if (!lowFrictionMode || (qAbs(velocity.x()) < qreal(0.8) * maxVelocity))
                    velocity.rx() *= deceleration;
                if (!lowFrictionMode || (qAbs(velocity.y()) < qreal(0.8) * maxVelocity))
                    velocity.ry() *= deceleration;

                if ((qAbs(velocity.x()) < qreal(1.0)) && (qAbs(velocity.y()) < qreal(1.0))) {
                    velocity = QPointF(0, 0);
                    q->killTimer(idleTimerId);
                    idleTimerId = 0;
                }
            }
        }
    } else if (mode == QMaemo5KineticScroller::AutoMode) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
    }
}


static QWidget *mouseTransparentChildAtGlobalPos(QWidget *parent, const QPoint &gp)
{
    foreach (QWidget *w, parent->findChildren<QWidget *>()) {
        if (w && !w->isWindow() && !w->isHidden() && (w->geometry().contains(w->mapFromGlobal(gp)))) {
            if (QWidget *t = mouseTransparentChildAtGlobalPos(w, gp))
                return t;
            else
                return w;
        }
    }
    return 0;
}

bool QMaemo5KineticScrollerPrivate::handleMousePress(QMouseEvent *e)
{
    Q_Q(QMaemo5KineticScroller);
    qKSDebug("MP: start");
    if (e->button() != Qt::LeftButton)
        return false;

    lastTime.start();
    lastPressTime.start();
    lastType = e->type();

    scrollTo = QPoint(-1, -1);

    if (buttonPressed && (childItem || childWidget)) {
        // send leave event to child...
        if (childItem) {
            //TODO:
        } else if (childWidget) {
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);
        }
    }

    pos = e->pos();
    ipos = pos;

    childItem = 0;
    childWidget = 0;

    // don't allow a click if we're still moving fast
    if ((qAbs(velocity.x()) <= (maxVelocity * fastVelocityFactor)) &&
        (qAbs(velocity.y()) <= (maxVelocity * fastVelocityFactor))) {
        if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(widget))
            childWidget = area->viewport();
        else
            childWidget = widget;

        if (QWidget *w = mouseTransparentChildAtGlobalPos(widget, e->globalPos()))
            childWidget = w;

        if (QGraphicsView *gfxview = qobject_cast<QGraphicsView *>(widget))
            childItem = gfxview->itemAt(pos);
    }
    buttonPressed = true;

    // stop scrolling on mouse press (so you can flick, then hold to stop)
    oldVelocity = velocity;
    velocity = QPointF(0, 0);

    if (idleTimerId) {
        q->killTimer(idleTimerId);
        idleTimerId = 0;
    }

    if (childItem || childWidget) {
        lastIn = true;

        if (childItem) {
            //TODO:
        } else if (childWidget) {
            QEvent ee(QEvent::Enter);
            sendEvent(childWidget, &ee);

            QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                           e->globalPos(), e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);
        }
    }
    qKSDebug("MP: end");
    return true;
}

bool QMaemo5KineticScrollerPrivate::handleMouseMove(QMouseEvent *e)
{
    qKSDebug() << "MM: pos: " << e->pos() << " - time: " << QTime::currentTime().msec();
    if (!buttonPressed || !lastTime.elapsed())
        return false;
    if (lastType == QEvent::MouseButtonPress)
        firstDrag = true;

    QPoint delta = e->pos() - pos;
    qKSDebug() << "Delta: " << delta << " Moved: " << moved;
    if (!moved) {
        checkMove(e, delta);
        qKSDebug() << "After Check: " << delta << " Moved: " << moved;
    }
    if (moved) {
        handleMove(e, delta);
        qKSDebug() << "After Handle: " << delta << " Moved: " << moved;
    } else if (childItem || childWidget) {
        // send leave event to child according to lastIn and set lastIn
        if (childItem) {
            //TODO:
        } else if (childWidget) {
            bool in = childWidget->geometry().contains(childWidget->mapFromGlobal(e->globalPos()));

            if (in != lastIn) {
                QEvent ele(in ? QEvent::Enter : QEvent::Leave);
                sendEvent(childWidget, &ele);
                lastIn = in;
            }
            // hack to remove the current selection as soon as we start to scroll
            // GtkTreeView apparently reacts on the Leave event, but Qt does not
            if (QAbstractItemView *view = qobject_cast<QAbstractItemView *>(childWidget->parentWidget())) {
                qKSDebug() << "Clearing selection in item view";
                view->selectionModel()->clear();
                view->selectionModel()->setCurrentIndex(QModelIndex(), QItemSelectionModel::Current);
            }
        }
    }
    lastTime.restart();
    lastType = e->type();

    if (childItem || childWidget) {
        // send mouse move event to child
        if (childItem) {
            //TODO:
        } else if (childWidget) {
            QMouseEvent me(e->type(), childWidget->mapFromGlobal(e->globalPos()),
                           e->globalPos(), e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);
        }
        // (see above)
    }
    qKSDebug("MM: end");
    return true;
}

bool QMaemo5KineticScrollerPrivate::handleMouseRelease(QMouseEvent *e)
{
    qKSDebug() << "MR: pos: " << e->pos() << " - time: " << QTime::currentTime().msec();
    Q_Q(QMaemo5KineticScroller);

    if (!buttonPressed || e->button() != Qt::LeftButton)
        return false;

    // if last event was a motion-notify we have to check the
    // movement and launch the animation
    if (lastType == QEvent::MouseMove) {
//xx        QPoint delta = e->pos() - pos;
//xx        if (!moved)
//xx            checkMove(e, delta);

        if (moved) {
//xx            int delta_time = lastTime.elapsed();
//xx
//xx            handleMove(e, delta);

            // move all the way to the last position now
            if (scrollTimerId) {
                q->killTimer(scrollTimerId);
                scrollTimerId = 0;
                q->setPosition(q->position() - motion);
                motion = QPoint(0, 0);
            }

//xx            if (delta_time >= CursorStoppedTimeout) {
//xx                if (qAbs(delta.x()) < 4)
//xx                    velocity.setX(0);
//xx                if (qAbs(delta.y()) < 4)
//xx                    velocity.setY(0);
//xx            }
        }
    }
    // If overshoot has been initiated with a finger down,
    // on release set max speed
    if (overShootDist.x()) {
        overShooting.setX(bounceSteps); // Hack to stop a bounce in the finger down case
        velocity.setX(overShootDist.x() * qreal(0.9));

    }
    if (overShootDist.y()) {
        overShooting.setY(bounceSteps); // Hack to stop a bounce in the finger down case
        velocity.setY(overShootDist.y() * qreal(0.9));
    }
    buttonPressed = false;

    bool forceFast = true;

    // if widget was moving fast in the panning, increase speed even more
    if ((lastPressTime.elapsed() < FastClick) &&
        ((qAbs(oldVelocity.x()) > minVelocity) ||
         (qAbs(oldVelocity.y()) > minVelocity)) &&
        ((qAbs(oldVelocity.x()) > MinimumAccelerationThreshold) ||
         (qAbs(oldVelocity.y()) > MinimumAccelerationThreshold))) {

        qKSDebug() << "FAST CLICK - using oldVelocity " << oldVelocity;
        int signX = 0, signY = 0;
        if (velocity.x())
            signX = (velocity.x() > 0) == (oldVelocity.x() > 0) ? 1 : -1;
        if (velocity.y())
            signY = (velocity.y() > 0) == (oldVelocity.y() > 0) ? 1 : -1;

        velocity.setX(signX * (oldVelocity.x() + (oldVelocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x())));
        velocity.setY(signY * (oldVelocity.y() + (oldVelocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y())));
        forceFast = false;
    }

    if ((qAbs(velocity.x()) >= minVelocity) ||
        (qAbs(velocity.y()) >= minVelocity)) {

        qKSDebug() << "over min velocity: " << velocity;
        // we have to move because we are in overshooting position
        if (!moved) {
            // (opt) emit panningStarted()
        }
        // (must) enable scrollbars

        if (forceFast) {
            if ((qAbs(velocity.x()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.x() > MaximumVelocityThreshold)) {
                velocity.setX(velocity.x() > 0 ? accelerationVelocity.x() : -accelerationVelocity.x());
            }
            if ((qAbs(velocity.y()) > MaximumVelocityThreshold) &&
                (accelerationVelocity.y() > MaximumVelocityThreshold)) {
                velocity.setY(velocity.y() > 0 ? accelerationVelocity.y() : -accelerationVelocity.y());
            }
            qKSDebug() << "Force fast is on - velocity: " << velocity;

        }
        if (!idleTimerId)
            idleTimerId = q->startTimer(1000 / scrollsPerSecond);
    } else {
        if (centerOnChildFocusPending) {
            centerOnChildFocus();
        }

        if (moved) {
            // (opt) emit panningFinished()
        }
    }
    centerOnChildFocusPending = false;
    lastTime.restart();
    lastType = e->type();

    if (!childItem && !childWidget) {
        moved = false;
        return false;
    }

    QGraphicsItem *newChildItem = 0;
    QWidget *newChildWidget = 0;

    if (QAbstractScrollArea *area = qobject_cast<QAbstractScrollArea *>(widget))
        newChildWidget = area->viewport();
    else
        newChildWidget = widget;

    if (QWidget *w = mouseTransparentChildAtGlobalPos(widget, e->globalPos()))
        newChildWidget = w;

    if (QGraphicsView *gfxview = qobject_cast<QGraphicsView *>(widget))
        newChildItem = gfxview->itemAt(pos);

    if (moved || (newChildWidget != childWidget) || (newChildItem != childItem)) {
        // send buttonRelease event with pos() NOT on child to child
        if (childItem) {
            //TODO:
        } else if (childWidget) {
            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);

            QMouseEvent me(e->type(), QPoint(-16384, -16384),
                           e->button(), e->buttons(), e->modifiers());
            sendEvent(childWidget, &me);
        }
    } else {
        // send buttonRelease event to child
        if (newChildItem) {
            //TODO:
        } else if (newChildWidget) {
            QMouseEvent me(e->type(), newChildWidget->mapFromGlobal(e->globalPos()),
                           e->globalPos(), e->button(), e->buttons(), e->modifiers());
            sendEvent(newChildWidget, &me);

            QEvent le(QEvent::Leave);
            sendEvent(childWidget, &le);
        }
    }
    childItem = 0;
    childWidget = 0;
    moved = false;
    return true;
}

void QMaemo5KineticScrollerPrivate::checkMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    if (firstDrag && !moved && ((qAbs(delta.x()) > panningThreshold) || (qAbs(delta.y()) > panningThreshold))) {
        moved = true;
        delta = QPoint(0, 0);

        if (firstDrag) {
            int deltaXtoY = qAbs(ipos.x() - me->pos().x()) - qAbs(ipos.y() - me->pos().y());

            qKSDebug() << "First Drag with delta " << delta << ", greater than " << panningThreshold << " -- deltaXtoY: " << deltaXtoY;

            QRect posRange = q->positionRange();
            bool canScrollX = (posRange.width() > 0);
            bool canScrollY = (posRange.height() > 0);

            if (deltaXtoY < 0) {
                if (!canScrollY && (!canScrollX || (-deltaXtoY >= directionErrorMargin)))
                    moved = false;
            } else {
                if (!canScrollX && (!canScrollY || (deltaXtoY >= directionErrorMargin)))
                    moved = false;
            }

            if (moved && (childItem || childWidget)) {
                // send leave event
                if (childItem) {
                    //TODO:
                } else if (childWidget) {
                    qKSDebug() << "Leaving " << childWidget << " since we are moving";
                    QEvent le(QEvent::Leave);
                    sendEvent(childWidget, &le);

                    // hack to remove the current selection as soon as we start to scroll
                    // GtkTreeView apparently reacts on the Leave event, but Qt does not
                    if (QAbstractItemView *view = qobject_cast<QAbstractItemView *>(childWidget->parentWidget())) {
                        qKSDebug() << "Clearing selection in item view";
                        view->selectionModel()->clear();
                        view->selectionModel()->setCurrentIndex(QModelIndex(), QItemSelectionModel::Current);
                    }
                }
            }
        }
        firstDrag = false;

        if (mode == QMaemo5KineticScroller::AccelerationMode) {
            if (!idleTimerId)
                idleTimerId = q->startTimer(1000 / scrollsPerSecond);
        }
    }
}

void QMaemo5KineticScrollerPrivate::handleMove(QMouseEvent *me, QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    switch (mode) {
    case QMaemo5KineticScroller::PushMode:
        // Scroll by the amount of pixels the cursor has moved
        // since the last motion event.
        scrollUpdate(delta);
        pos = me->pos();
        break;

    case QMaemo5KineticScroller::AccelerationMode:
        // Set acceleration relative to the initial click
        // epos = me->pos(); //TODO: what the heck is epos?
        velocity.setX(qreal(delta.x() < 0 ? -1 : 1) * ((qreal(qAbs(delta.x())) / qreal(widget->width()) * (maxVelocity - minVelocity)) + minVelocity));
        velocity.setY(qreal(delta.y() < 0 ? -1 : 1) * ((qreal(qAbs(delta.y())) / qreal(widget->height()) * (maxVelocity - minVelocity)) + minVelocity));
        break;

    case QMaemo5KineticScroller::AutoMode:
        QPointF newVelocity = calculateVelocity(me->pos() - pos, lastTime.elapsed());
        QRect posRange = q->positionRange();

        if (!posRange.width()) {
            delta.setX(0);
            newVelocity.setX(0);
        }
        if (!posRange.height()) {
            delta.setY(0);
            newVelocity.setY(0);
        }
        velocity = newVelocity;

        scrollUpdate(delta);

        if (posRange.width())
            pos.setX(me->pos().x());
        if (posRange.height())
            pos.setY(me->pos().y());
        break;
    }
}

QPointF QMaemo5KineticScrollerPrivate::calculateVelocity(const QPointF &dPixel, int dTime)
{
    qKSDebug() << "calculateVelocity(dP = " << dPixel << ", dT = " << dTime << ") -- velocity: " << velocity;

    QPointF newv = velocity;

    // fast than 15 pix / ms seems bogus (that's a screen height in 1/30 second)
    if ((dPixel / qreal(dTime)).manhattanLength() < 25) {
        QPointF rawv = dPixel / qreal(dTime) * qreal(force);
        newv = newv * (qreal(1) - dragInertia) + rawv * dragInertia;
    }

    qKSDebug() << " --> " << newv << " (before clamping)";

    newv.setX(dPixel.x() ? qBound(-maxVelocity, newv.x(), maxVelocity) : velocity.x());
    newv.setY(dPixel.y() ? qBound(-maxVelocity, newv.y(), maxVelocity) : velocity.y());
    return newv;
}

void QMaemo5KineticScrollerPrivate::scrollUpdate(const QPoint &delta)
{
    Q_Q(QMaemo5KineticScroller);

    if (scrollTimerId) {
        motion += delta;
    } else {
        // we do not delay the first event but the next ones
        q->setPosition(q->position() - delta);
        motion = QPoint(0, 0);
        scrollTimerId = q->startTimer(1000 / MotionEventsPerSecond);
    }
}


void QMaemo5KineticScrollerPrivate::centerOnChildFocus()
{
    //TODO:
}

QMaemo5KineticScroller::Mode QMaemo5KineticScroller::mode() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->mode;
}

void QMaemo5KineticScroller::setMode(Mode mode)
{
    Q_D(QMaemo5KineticScroller);
    d->mode = mode;
}


bool QMaemo5KineticScroller::isLowFrictionEnabled() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->lowFrictionMode;
}

void QMaemo5KineticScroller::setLowFrictionEnabled(bool b)
{
    Q_D(QMaemo5KineticScroller);
    d->lowFrictionMode = b;
}


qreal QMaemo5KineticScroller::dragInertia() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->dragInertia;
}

void QMaemo5KineticScroller::setDragInertia(qreal inertia)
{
    Q_D(QMaemo5KineticScroller);
    d->dragInertia = inertia;
}


int QMaemo5KineticScroller::directionErrorMargin() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->directionErrorMargin;
}

void QMaemo5KineticScroller::setDirectionErrorMargin(int errorMargin)
{
    Q_D(QMaemo5KineticScroller);
    d->directionErrorMargin = errorMargin;
}


int QMaemo5KineticScroller::panningThreshold() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->panningThreshold;
}

void QMaemo5KineticScroller::setPanningThreshold(int threshold)
{
    Q_D(QMaemo5KineticScroller);
    d->panningThreshold = threshold;
}


qreal QMaemo5KineticScroller::decelerationFactor() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->deceleration;
}

void QMaemo5KineticScroller::setDecelerationFactor(qreal f)
{
    Q_D(QMaemo5KineticScroller);
    d->deceleration = f;
}


qreal QMaemo5KineticScroller::fastVelocityFactor() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->fastVelocityFactor;
}

void QMaemo5KineticScroller::setFastVelocityFactor(qreal f)
{
    Q_D(QMaemo5KineticScroller);
    d->fastVelocityFactor = f;
}


qreal QMaemo5KineticScroller::minimumVelocity() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->minVelocity;
}

void QMaemo5KineticScroller::setMinimumVelocity(qreal v)
{
    Q_D(QMaemo5KineticScroller);
    d->minVelocity = v;
}


qreal QMaemo5KineticScroller::maximumVelocity() const
{
    Q_D(const QMaemo5KineticScroller);
    return d->maxVelocity;
}

void QMaemo5KineticScroller::setMaximumVelocity(qreal v)
{
    Q_D(QMaemo5KineticScroller);
    d->maxVelocity = v;
}


void QMaemo5KineticScroller::scrollTo(const QPoint &pos)
{
    Q_D(QMaemo5KineticScroller);
    //TODO:
}


QT_END_NAMESPACE
