/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "text_p.h"
#include "text_p_p.h"
#include <QTextEdit>
#include <QVBoxLayout>
#include <QActionGroup>
#include <QMoveEvent>
#include <QResizeEvent>
#include <QGraphicsOpacityEffect>

Text::Text(QWidget *parent) :
    Item(*new TextPrivate(this), parent)
{
    Q_D(Text);

    d->textEdit = new QTextEdit(this);
    d->textEdit->setStyleSheet("background: transparent");
    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->addWidget(d->textEdit);
}

Text::Text(TextPrivate &dd, QWidget *parent) :
    Item(dd, parent)
{
    Q_D(Text);

    if (!d->textEdit) {
        d->textEdit = new QTextEdit(this);
    }

    d->textEdit->setStyleSheet("background: transparent");
    QVBoxLayout *vbox = new QVBoxLayout(this);
    vbox->addWidget(d->textEdit);
}

Text::~Text() {}

QColor Text::color() const {
    Q_D(const Text);

    return d->textEdit.textColor();
}

void Text::setColor(const QColor &color) {
    if (color != this->color()) {
        Q_D(Text);
        d->textEdit->setTextColor(color);
        emit colorChanged();
    }
}

QColor Text::styleColor() const {
    Q_D(const Text);

    return d->textEdit->textBackgroundColor();
}

void Text::setStyleColor(const QColor &color) {
    if (color != this->styleColor()) {
        Q_D(Text);
        d->textEdit->setTextBackgroundColor(color);
        emit styleColorChanged();
    }
}

Text::ElideMode Text::elide() const {
    Q_D(const Text);

    for (int i = Text::ElideLeft; i <= Text::ElideNone; i++) {
        if (d->textEdit->alignment() & i) {
            return static_cast<Text::ElideMode>(i);
        }
    }

    return Text::ElideNone;
}

void Text::setElide(ElideMode elide) {
    if (elide != this->elide()) {
        Q_D(Text);
        d->textEdit->setAlignment(static_cast<Qt::AlignmentFlag>(this->horizontalAlignment()),
                                  static_cast<Qt::AlignmentFlag>(this->verticalAlignment()),
                                  static_cast<Qt::TextElideMode>(elide));
        emit alignmentChanged();
    }
}

Text::HAlignment Text::horizontalAlignment() const {
    Q_D(const Text);

    for (int i = Text::AlignLeft; i <= Text::AlignJustify; i++) {
        if (d->textEdit->alignment() & i) {
            return static_cast<Text::HAlignment>(i);
        }
    }

    return Text::AlignJustify;
}

void Text::setHorizontalAlignment(HAlignment alignment) {
    if (alignment != this->horizontalAlignment()) {
        Q_D(Text);
        d->textEdit->setAlignment(static_cast<Qt::AlignmentFlag>(alignment),
                                  static_cast<Qt::AlignmentFlag>(this->verticalAlignment()),
                                  static_cast<Qt::TextElideMode>(this->elide()));
        emit alignmentChanged();
    }
}

Text::VAlignment Text::verticalAlignment() const {
    Q_D(const Text);

    for (int i = Text::AlignTop; i <= Text::AlignVCenter; i++) {
        if (d->textEdit->alignment() & i) {
            return static_cast<Text::VAlignment>(i);
        }
    }

    return Text::AlignJustify;
}

void Text::setVerticalAlignment(VAlignment alignment) {
    if (alignment != this->verticalAlignment()) {
        Q_D(Text);
        d->textEdit->setAlignment(static_cast<Qt::AlignmentFlag>(this->horizontalAlignment()),
                                  static_cast<Qt::AlignmentFlag>(alignment),
                                  static_cast<Qt::TextElideMode>(this->elide()));
        emit alignmentChanged();
    }
}

int Text::lineCount() const {
    Q_D(const Text);

    return d->textEdit->document()->lineCount();
}

int Text::maximumLineCount() const {
    return 0;
}

void Text::setMaximumLineCount(int count) {
    Q_UNUSED(count);
}

qreal Text::lineHeight() const {
    Q_D(const Text);

    return 0.0;
}

Text::LineHeightMode Text::lineHeightMode() const {
    return Text::ProportionalHeight;
}

void Text::setLineHeightMode(LineHeightMode mode) {
    Q_UNUSED(mode);
}

bool Text::smooth() const {
    return true;
}

void Text::setSmooth(bool smooth) {
    Q_UNUSED(smooth);
}

Text::TextStyle Text::textStyle() const {
    return Text::Normal;
}

void Text::setTextStyle(TextStyle style) {
    Q_UNUSED(style);
}

Text::TextFormat Text::textFormat() const {
    return Text::AutoText;
}

void Text::setTextFormat(TextFormat format) {
    Q_UNUSED(format);
}

QString Text::text() const {
    Q_D(const Text);

    return d->textEdit->toHtml();
}

void Text::setText(const QString &text) {
    if (text != this->text()) {
        Q_D(Text);
        d->textEdit->setText(text);
        emit textChanged();
    }
}

bool Text::truncated() const {
    Q_D(const Text);

    return false;
}

Text::WrapMode Text::wrapMode() const {
    Q_D(const Text);

    return static_cast<Text::WrapMode>(d->textEdit->wordWrapMode());
}

void Text::setWrapMode(WrapMode mode) {
    if (mode != this->wrapMode()) {
        Q_D(Text);
        d->textEdit->setWordWrapMode(static_cast<QTextOption::WrapMode>(mode));
        emit wrapModeChanged();
    }
}

#include "moc_text_p.cpp"
