/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef WEBVIEW_P_H
#define WEBVIEW_P_H

class WebView : public Flickable
{

    Q_PROPERTY(string title READ title)
    Q_PROPERTY(url url READ url WRITE setUrl)
    Q_PROPERTY(string icon READ icon)
    Q_PROPERTY(string html READ toHtml WRITE setHtm)
    Q_PROPERTY(string text READ toPlainText WRITE setText)
    Q_PROPERTY(bool textSelectionEnabled READ textSelectionEnabled WRITE setTextSelectionEnabled)
    Q_PROPERTY(bool hasSelection READ hasSelection)
    Q_PROPERTY(bool modified READ isModified)
    Q_PROPERTY(string selectedText READ selectedText)
    Q_PROPERTY(real zoomFactor READ zoomFactor WRITE setZoomFactor)
    Q_PROPERTY(enumeration linkDelegationPolicy READ linkDelegationPolicy WRITE setLinkDelegationPolicy)
    Q_PROPERTY(bool forwardUnsupportedContent READ forwardUnsupportedContent WRITE setForwardUnsupportedContent)
    Q_PROPERTY(int progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(string statusText READ statusText)
    Q_PROPERTY(string userAgent READ userAgent WRITE setUserAgent)
    Q_PROPERTY(Component newWindowComponent READ newWindowComponent WRITE setNewWindowComponent)
    Q_PROPERTY(Item newWindowParent READ newWindowParent WRITE setNewWindowParent)
    Q_PROPERTY(WebPage page READ page WRITE setPage)
    Q_PROPERTY(WebHistory history READ history CONSTANT)
    Q_PROPERTY(WebSettings settings READ settings CONSTANT)
    Q_PROPERTY(list<Object> javaScriptWindowObjects READ javaScriptWindowObjects)

public:
    jsobject hitTestContent(int x, int y);

    bool findText(const string &text);
    bool findAllText(const string &text);
    
    void stop();
    void back();
    void forward();
    void reload();

    void copy();
    void cut();
    void paste();

signals:
    void linkClicked(url link);
    void downloadRequested(jsobject request);
    void unsupportedContent(jsobject content);

/*!
    \class WebView
    \brief The WebView component provides an item that is used to view and edit
    web documents.

    \ingroup org-hildon-webkit

    WebView is the main component of the org.hildon.webkit web browsing module.
    It can be used in various applications to display web content live from the
    Internet.
*/

/*!
    \property url WebView::url
    \brief The source url of the web view
*/

/*!
    \property string WebView::icon
    \brief The source of the web view icon
*/

/*!
    \property string WebView::title
    \brief The title of the web view
*/

/*!
    \property string WebView::html
    \brief The HTML of the web view
*/

/*!
    \property string WebView::text
    \brief The HTML of the web view as plain text
*/

/*!
    \property real WebView::zoomFactor
    \brief The zoom factor of the web view
*/

/*!
    \property int WebView::progress
    \brief The loading progress of the web view
*/

/*!
    \property enumeration WebView::status
    \brief The current status of the web view
    
    Possible values are:
    
    \li \c Null
    \li \c Loading
    \li \c Ready
    \li \c Error
    
    The default value is \c Null.
*/

/*!
    \property string WebView::statusText
    \brief The status text of the web view
*/

/*!
    \property string WebView::selectedText
    \brief The selected text of the web view
*/

/*!
    \property bool WebView::hasSelection
    \brief Whether any text is selected in the web view
*/

/*!
    \property bool WebView::textSelectionEnabled
    \brief Whether gestures are used to scroll the web view 
    or select text
*/

/*!
    \property bool WebView::modified
    \brief Whether the web page has been modified
*/

/*!
    \property bool WebView::forwardUnsupportedContent
    \brief Whether to forward content that is not supported by WebPage
    
    If this property is set to true, the unsupportedContent() signal 
    will be emitted when unsupported content is found.
*/

/*!
    \property enumeration WebView::linkDelegationPolicy
    \brief The policy used to determine how to handle activated links
    
    The possible values are:
    
    \li \c DontDelegateLinks - No links are delegated. Instead, 
    WebPage tries to handle them all.
    \li \c DelegateExternalLinks - When activating links that point to 
    documents not stored on the local filesystem or an equivalent - 
    such as the Qt resource system - then linkClicked() is emitted.
    \li \c DelegateAllLinks - Whenever a link is activated the linkClicked() 
    signal is emitted.
    
    The default value is \c DontDelegateLinks.
*/

/*!
    \property string WebView::userAgent
    \brief The user-agent string used when loading a web page
*/

/*!
    \property Component WebView::newWindowComponent
    \brief The component used when a new window is requested
*/

/*!
    \property WebHistory WebView::history
    \brief The history of the web view
*/

/*!
    \property WebSettings WebView::settings
    \brief The settings used by the web view
*/

/*!
    \property WebPage WebView::page
    \brief The web page used for the web view
*/

/*!
    \property list<Object> WebView::javaScriptWindowObjects
    \brief The objects exposed to scripting in the webpage
*/

/*!
    \fn jsobject WebView::hitTestContent(int x, int y)
    
    Performs a hit test at \a x,\a y, and returns the result.
*/

/*!
    \fn bool WebView::findText(string text)
    
    Searches for the next occurrance of \a text in 
    the web view. Returns true if successful.
*/

/*!
    \fn bool WebView::findAllText(string text)
    
    Searches for all occurrances of \a text in the 
    web page. Returns true if successful.
*/

/*!
    \fn void WebView::reload()
    
    Reloads the current web page.
*/

/*!
    \fn void WebView::stop()
    
    Aborts the loading of a web page.
*/

/*!
    \fn void WebView::back()
    
    Navigates to the previous item in the history, 
    if possible.
    
    \sa WebPage::back()
*/

/*!
    \fn void WebView::forward()
    
    Navigates to the next item in the history, 
    if possible.
    
    \sa WebPage::forward()
*/

/*!
    \fn void WebView::copy()
    
    Copies the selected HTML to the clipboard.
    
    \sa WebPage::copy()
*/

/*!
    \fn void WebView::cut()
    
    Copies the selected HTML to the clipboard 
    and removes it from the web view.
    
    \sa WebPage::cut()
*/

/*!
    \fn void WebView::paste()
    
    Inserts the the clipboard's contents into 
    the web view.
    
    \sa WebPage::paste()
*/

/*!
    \fn void WebView::downloadRequested(jsobject request)

    This signal is emitted when the user decides to download a link. The url of
    the link as well as additional meta-information is contained in \a request.

    \sa unsupportedContent()
*/

/*!
    \fn void WebView::unsupportedContent(jsobject content)

    This signal is emitted when WebKit cannot handle a link the user navigated to or a
    web server's response includes a "Content-Disposition" header with the 'attachment' 
    directive. If "Content-Disposition" is present in \a content, the web server is indicating
    that the client should prompt the user to save the content regardless of content-type. 
    See RFC 2616 sections 19.5.1 for details about Content-Disposition.

    \note This signal is only emitted if the \link forwardUnsupportedContent\endlink 
    property is set to true.

    \sa downloadRequested()
*/

/*!
    \fn void WebView::linkClicked(url url)

    This signal is emitted whenever the user clicks on a link and the page's 
    \link linkDelegationPolicy\endlink property is set to delegate the link handling 
    for the specified \a url.

    By default no links are delegated and are handled by WebPage instead.

    \note This signal possibly won't be emitted for clicked links which use
    JavaScript to trigger navigation.
*/

};

#endif // WEBVIEW_P_H
