/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */


#ifndef TREEVIEW_P_H
#define TREEVIEW_P_H

class TreeView : public Flickable
{

    // QAbstractItemView
    Q_PROPERTY(flags editTriggers READ editTriggers WRITE setEditTriggers)
    Q_PROPERTY(enumeration selectionMode READ selectionMode WRITE setSelectionMode)
    Q_PROPERTY(enumeration selectionBehavior READ selectionBehavior WRITE setSelectionBehavior)
    Q_PROPERTY(size iconSize READ iconSize WRITE setIconSize)
    Q_PROPERTY(enumeration textElideMode READ textElideMode WRITE setTextElideMode)
    Q_PROPERTY(enumeration verticalScrollMode READ verticalScrollMode WRITE setVerticalScrollMode)
    Q_PROPERTY(enumeration horizontalScrollMode READ horizontalScrollMode WRITE setHorizontalScrollMode)
    
    // QTreeView
    Q_PROPERTY(int autoExpandDelay READ autoExpandDelay WRITE setAutoExpandDelay)
    Q_PROPERTY(int indentation READ indentation WRITE setIndentation)
    Q_PROPERTY(bool rootIsDecorated READ rootIsDecorated WRITE setRootIsDecorated)
    Q_PROPERTY(bool uniformRowHeights READ uniformRowHeights WRITE setUniformRowHeights)
    Q_PROPERTY(bool itemsExpandable READ itemsExpandable WRITE setItemsExpandable)
    Q_PROPERTY(bool sortingEnabled READ isSortingEnabled WRITE setSortingEnabled)
    Q_PROPERTY(bool animated READ isAnimated WRITE setAnimated)
    Q_PROPERTY(bool allColumnsShowFocus READ allColumnsShowFocus WRITE setAllColumnsShowFocus)
    Q_PROPERTY(bool wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(bool headerHidden READ isHeaderHidden WRITE setHeaderHidden)
    Q_PROPERTY(bool expandsOnDoubleClick READ expandsOnDoubleClick WRITE setExpandsOnDoubleClick)
    
    // TreeView
    Q_PROPERTY(variant model READ model WRITE setModel)
    Q_PROPERTY(QModelIndex rootIndex READ rootIndex WRITE setRootIndex)
    Q_PROPERTY(QModelIndex currentIndex READ currentIndex WRITE setCurrentIndex)
    Q_PROPERTY(list<QModelIndex> selectedIndexes READ selectedIndexes WRITE setSelectedIndexes)
    Q_PROPERTY(ListItem delegate READ delegate WRITE setDelegate)
    Q_PROPERTY(bool showRowNumbers READ rowNumbersVisible WRITE setRowNumbersVisible)
    Q_PROPERTY(HeaderView header READ horizontalHeader WRITE setHeader)
    
public:
    // QAbstractItemView
    void reset();
    void selectAll();
    void edit(QModelIndex index);
    void clearSelection();
    void update(QModelIndex index);
    
    // QTreeView
    void hideColumn(int column);
    void showColumn(int column);
    void expand(QModelIndex index);
    void collapse(QModelIndex index);
    void resizeColumnToContents(int column);
    void expandAll();
    void collapseAll();
    void expandToDepth(int depth);
    
    //TreeView
    void positionViewAtBeginning();
    void positionViewAtEnd();
    void positionViewAtIndex(QModelIndex index, enumeration hint);
    
signals:
    // QAbstractItemView
    void pressed();
    void clicked();
    void doubleClicked();
    void activated();
    
/*!
    \class TreeView
    \brief The TreeView component provides an item view used for displaying model data
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-views
    
*/

/*!
    \property variant TreeView::model
    \brief The \a model for the view to present

    The view <em>does not</em> take ownership of the model unless it is the model's
    parent object because the model may be shared between many different views.
    
    The model can be any of the following types:
    
    \li \c int
    \li \c string
    \li \c list
    \li \c stringlist
    \li \c ListModel
    \li \c QAbstractItemModel

    \sa ListModel
*/

/*!
    \property ListItem TreeView::delegate
    \brief The item delegate for this view and its model
    
    This is useful if you want complete control over the editing and
    display of items.

    Any existing delegate will be removed, but not deleted. The view
    does not take ownership of the delegate.

    \warning You should not share the same instance of a delegate between views.
    Doing so can cause incorrect or unintuitive editing behavior since each
    view connected to a given delegate may receive the ListItem::closeEditor()
    signal, and attempt to access, modify or close an editor that has already been closed.
    
    \sa ListItem
*/

/*!
    \property HeaderView TreeView::header
    \brief The view's header
*/

/*!
    \property bool TreeView::showRowNumbers
    \brief Whether the table's row numbers are visible
    
    The default value is \a true.
*/

/*!
    \property enumeration TreeView::selectionMode
    \brief Which selection mode the view operates in

    This property controls whether the user can select one or many items
    and, in many-item selections, whether the selection must be a
    continuous range of items.
    
    Possible values are:
    
    \li \c SingleSelection - When the user selects an item, 
    any already-selected item becomes unselected, 
    and the user cannot unselect the selected 
    item by clicking on it.
    
    \li \c ContiguousSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Shift key while 
    clicking on an item, all items between the current item and 
    the clicked item are selected or unselected, depending on 
    the state of the clicked item.
    
    \li \c ExtendedSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Ctrl key when 
    clicking on an item, the clicked item gets toggled and 
    all other items are left untouched. If the user presses 
    the Shift key while clicking on an item, all items between 
    the current item and the clicked item are selected or 
    unselected, depending on the state of the clicked item. 
    Multiple items can be selected by dragging the mouse 
    over them.
    
    \li \c MultiSelection - When the user selects an item 
    in the usual way, the selection status of that item is 
    toggled and the other items are left alone. Multiple 
    items can be toggled by dragging the mouse over them.
    
    \li \c NoSelection - Items cannot be selected.
    
    The default value is \c SingleSelection.

    \sa selectionBehavior
*/

/*!
    \property enumeration TreeView::selectionBehavior
    \brief Which selection behavior the view uses

    This property holds whether selections are done
    in terms of single items, rows or columns.
    
    Possible values are:
    
    \li \c SelectItems - Selecting single items.
    \li \c SelectRows - Select only rows.
    \li \c SelectColumns - Select only columns.
    
    The default value is \c SelectItems.

    \sa selectionMode
*/

/*!
    \property QModelIndex TreeView::currentIndex
    \brief The model index of the current item

    \sa selectedIndexes, selectionBehavior, selectionMode
*/

/*!
    \property QModelIndex TreeView::rootIndex
    \brief The model index of the model's root item. 
    
    The root item is the parent item to the view's toplevel items. 
    The root can be invalid.
*/

/*!
    \property list<QModelIndex> TreeView::selectedIndexes
    \brief The model indexes of the selected items
    
    \sa currentIndex, selectionBehavior, selectionMode
*/

/*!
    \property flags TreeView::editTriggers
    \brief Which actions will initiate item editing

    This property is a selection of flags combined using the OR
    operator. The view will only initiate the editing of an item if the
    action performed is set in this property.
    
    Available flags are:
    
    \li \c NoEditTriggers - No editing possible.
    \li \c CurrentChanged - Editing start whenever current item changes.
    \li \c DoubleClicked - Editing starts when an item is double clicked.
    \li \c SelectedClicked - Editing starts when clicking on an already selected item.
    \li \c EditKeyPressed - Editing starts when the platform edit key has been pressed over an item.
    \li \c AnyKeyPressed - Editing starts when any key is pressed over an item.
    \li \c AllEditTriggers - Editing starts for all above actions.
*/

/*!
    \property enumeration TreeView::verticalScrollMode
    \brief How the view scrolls its contents in the vertical direction

    This property controls how the view scroll its contents vertically.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa horizontalScrollMode
*/

/*!
    \property enumeration TreeView::horizontalScrollMode
    \brief How the view scrolls its contents in the horizontal direction

    This property controls how the view scroll its contents horizontally.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa verticalScrollMode
*/

/*!
    \property size TreeView::iconSize
    \brief The size of items' icons

    Setting this property when the view is visible will cause the
    items to be laid out again.
*/

/*!
    \property enumeration TreeView::textElideMode

    \brief The position of the "..." in elided text.
    
    Possible values are:
    
    \li \c Qt.ElideLeft - The ellipsis should appear at the beginning of the text.
    \li \c Qt.ElideRight - The ellipsis should appear at the end of the text.
    \li \c Qt.ElideMiddle - The ellipsis should appear in the middle of the text.
    \li \c Qt.ElideNone - Ellipsis should NOT appear in the text.

    The default value is Qt.ElideRight.
*/

/*!
    \property int TreeView::autoExpandDelay
    \brief The delay time before items in a tree are opened during a drag and drop operation.

    This property holds the amount of time in milliseconds that the user must wait over
    a node before that node will automatically open or close.  If the time is
    set to less then 0 then it will not be activated.

    By default, this property has a value of -1, meaning that auto-expansion is disabled.
*/

/*!
    \property int TreeView::indentation
    \brief Indentation of the items in the tree view.

    This property holds the indentation measured in pixels of the items for each
    level in the tree view. For top-level items, the indentation specifies the
    horizontal distance from the viewport edge to the items in the first column;
    for child items, it specifies their indentation from their parent items.

    By default, this property has a value of 20.
*/

/*!
    \property bool TreeView::rootIsDecorated
    \brief Whether to show controls for expanding and collapsing top-level items

    Items with children are typically shown with controls to expand and collapse
    them, allowing their children to be shown or hidden. If this property is
    false, these controls are not shown for top-level items. This can be used to
    make a single level tree structure appear like a simple list of items.

    By default, this property is true.
*/

/*!
    \property TreeView::uniformRowHeights
    \brief Whether all items in the treeview have the same height

    This property should only be set to true if it is guaranteed that all items
    in the view has the same height. This enables the view to do some
    optimizations.

    The height is obtained from the first item in the view.  It is updated
    when the data changes on that item.

    By default, this property is false.
*/

/*!
    \property bool TreeView::itemsExpandable
    \brief Whether the items are expandable by the user.

    This property holds whether the user can expand and collapse items
    interactively.

    By default, this property is true.
*/

/*!
    \property bool TreeView::expandsOnDoubleClick
    \brief Whether the items can be expanded by double-clicking.

    This property holds whether the user can expand and collapse items
    by double-clicking. The default value is true.

    \sa itemsExpandable
*/

/*!
    \property bool TreeView::headerHidden
    \brief Whether the header is shown or not.

    If this property is true, the header is not shown otherwise it is.
    The default value is false.

    \sa header
*/

/*!
    \property bool TreeView::sortingEnabled
    \brief Whether sorting is enabled

    If this property is true, sorting is enabled for the tree; if the property
    is false, sorting is not enabled. The default value is false.

    \note In order to avoid performance issues, it is recommended that
    sorting is enabled \e after inserting the items into the tree.
    Alternatively, you could also insert the items into a list before inserting
    the items into the tree.

    \sa sortByColumn()
*/

/*!
    \property bool TreeView::animated
    \brief Whether animations are enabled

    If this property is true the treeview will animate expandsion
    and collasping of branches. If this property is false, the treeview
    will expand or collapse branches immediately without showing
    the animation.

    By default, this property is false.
*/

/*!
    \property bool TreeView::allColumnsShowFocus
    \brief Whether items should show keyboard focus using all columns

    If this property is true all columns will show focus, otherwise only
    one column will show focus.

    The default is false.
*/

/*!
    \property bool TreeView::wordWrap
    \brief The item text word-wrapping policy

    If this property is true then the item text is wrapped where
    necessary at word-breaks; otherwise it is not wrapped at all.
    This property is false by default.

    Note that even if wrapping is enabled, the cell will not be
    expanded to fit all text. Ellipsis will be inserted according to
    the current textElideMode.
*/

/*!
    \fn void TreeView::reset()
    
    Resets the internal state of the view.

    \warning This function will reset open editors, scroll bar positions,
    selections, etc. Existing changes will not be committed.
*/

/*!
    \fn void TreeView::edit(QModelIndex index)
    
    Starts editing the item corresponding to the given \a index if it is
    editable.

    Note that this function does not change the current index. Since the current
    index defines the next and previous items to edit, users may find that
    keyboard navigation does not work as expected. To provide consistent navigation
    behavior, set currentIndex before this function with the same model
    index.
    
    \sa editTriggers
*/

/*!
    \fn void TreeView::update(QModelIndex index)

    Updates the area occupied by the given \a index.
*/

/*!
    \fn void TreeView::hideColumn(int column)
    
    Hides the \a column given.

    \note This function should only be called after the model has been
    initialized, as the view needs to know the number of columns in order to
    hide \a column.

    \sa showColumn()
*/

/*!
    \fn void TreeView::expandAll()
    
    Expands all expandable items.

    Warning: if the model contains a large number of items,
    this function will take some time to execute.

    \sa collapseAll(), expand(), collapse()
*/

/*!
    \fn void TreeView::collapseAll()

    Collapses all expanded items.

    \sa expandAll(), expand(), collapse()
*/

/*!
    \fn void TreeView::showColumn(int column)
    
    Shows the given \a column in the tree view.

    \sa hideColumn()
*/

/*!
    \fn void TreeView::expand(QModelIndex index)

    Expands the model item specified by the \a index.
*/

/*!
    \fn void TreeView::expandToDepth(int depth)
    
    Expands all expandable items to the given \a depth.

    \sa expandAll() ,collapseAll(), expand(), collapse()
*/

/*!
    \fn void TreeView::collapse(QModelIndex index)

    Collapses the model item specified by the \a index.
*/

/*!
    \fn void TreeView::resizeColumnToContents(int column)
    
    Resizes the \a column given to the size of its contents.
*/

/*!
    \fn void TreeView::selectAll()
    
    Selects all items in the view.
    This function will use the selection behavior
    set on the view when selecting.

    \sa selectedIndexes, clearSelection()
*/

/*!
    \fn void TreeView::clearSelection()
    
    Deselects all selected items. The current index will not be changed.

    \sa selectedIndexes, selectAll()
*/

/*!
    \fn void TreeView::positionViewAtBeginning()
    
    Positions the view at the beginning.
*/

/*!
    \fn void TreeView::positionViewAtEnd()
    
    Positions the view at the end.
*/

/*!
    \fn void TreeView::positionViewAtIndex(QModelIndex index, enumeration hint)
    
    Positions the view at \a index. The exact position depends upon \a hint.
    
    Possible values for \a hint are:
    
    \li \c EnsureVisible - Scroll to ensure that the item is visible.
    \li \c PositionAtTop - Scroll to position the item at the top of the view.
    \li \c PositionAtBottom - Scroll to position the item at the bottom of the view.
    \li \c PositionAtCenter - Scroll to position the item at the center of the view.
*/

/*!
    \fn void TreeView::activated()

    This signal is emitted when an item is activated by the user 
    by single- or double-clicking the item, or by
    pressing the Return or Enter key when the item is current.

    \sa clicked(), doubleClicked(), pressed()
*/

/*!
    \fn void TreeView::clicked()

    This signal is emitted when an item is activated by the user 
    clicking the item.

    \sa activated(), doubleClicked(), pressed()
*/

/*!
    \fn void TreeView::doubleClicked()

    This signal is emitted when an item is activated by the user 
    by double-clicking the item.

    \sa activated(), clicked(), pressed()
*/

/*!
    \fn void TreeView::pressed()

    This signal is emitted when an item is pressed by the user.

    \sa activated(), clicked(), doubleClicked()
*/

};

#endif // TREEVIEW_P_H
