/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TEXTFIELD_P_H
#define TEXTFIELD_P_H

class TextField : public Item
{

    Q_PROPERTY(string inputMask READ inputMask WRITE setInputMask)
    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(int maxLength READ maxLength WRITE setMaxLength)
    Q_PROPERTY(bool frame READ hasFrame WRITE setFrame)
    Q_PROPERTY(enumeration echoMode READ echoMode WRITE setEchoMode)
    Q_PROPERTY(string displayText READ displayText)
    Q_PROPERTY(int cursorPosition READ cursorPosition WRITE setCursorPosition)
    Q_PROPERTY(flags alignment READ alignment WRITE setAlignment)
    Q_PROPERTY(bool modified READ isModified WRITE setModified)
    Q_PROPERTY(bool hasSelectedText READ hasSelectedText)
    Q_PROPERTY(string selectedText READ selectedText)
    Q_PROPERTY(bool dragEnabled READ dragEnabled WRITE setDragEnabled)
    Q_PROPERTY(bool readOnly READ isReadOnly WRITE setReadOnly)
    Q_PROPERTY(bool undoAvailable READ isUndoAvailable)
    Q_PROPERTY(bool redoAvailable READ isRedoAvailable)
    Q_PROPERTY(bool acceptableInput READ hasAcceptableInput)
    Q_PROPERTY(string placeholderText READ placeholderText WRITE setPlaceholderText)
    Q_PROPERTY(QValidator validator READ validator WRITE setValidator)

public:
    void clear();
    void selectAll();
    void undo();
    void redo();
    void cut();
    void copy();
    void paste();
    
signals:
    void textEdited(string text);
    void returnPressed();
    void editingFinished();
    void selectionChanged();
    
/*!
    \class TextField
    \brief The TextField component is a one-line text editor.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-text

    A text field allows the user to enter and edit a single line of
    plain text with a useful collection of editing functions,
    including undo and redo, cut and paste, and drag and drop.

    By changing the \c echoMode of a text field, it can also be used as
    a "write-only" field, for inputs such as passwords.

    The length of the text can be constrained to \c maxLength. The text
    can be arbitrarily constrained using a \c validator or an
    \c inputMask, or both. When switching between a validator and an input mask 
    on the same text field, it is best to clear the validator or input mask to 
    prevent undefined behavior.
    
    \sa TextEdit, Text, Label
*/

/*!
    \property string TextField::text
    \brief The text field's text

    Setting this property clears the selection, clears the undo/redo
    history, moves the cursor to the end of the line and resets the
    \c modified property to false. The text is not validated when
    set programmatically.

    The text is truncated to \c maxLength length.

    By default, this property contains an empty string.

    \sa clear()
*/

/*!
    \property string TextField::placeholderText
    \brief The text field's placeholder text

    Setting this property makes the text field display a grayed-out
    placeholder text as long as the \c text is empty and the item doesn't
    have focus.

    By default, this property contains an empty string.

    \sa text
*/

/*!
    \property string TextField::displayText
    \brief The displayed text

    If \c echoMode is \c Normal this returns the same as \c text; if
    \c EchoMode is \c Password or \c PasswordEchoOnEdit it returns a string of asterisks
    \c text.length characters long, e.g. "******"; if \c EchoMode is
    \c NoEcho returns an empty string, "".

    By default, this property contains an empty string.

    \sa echoMode, text
*/

/*!
    \property int TextField::maxLength
    \brief The maximum permitted length of the text

    If the text is too long, it is truncated at the limit.

    If truncation occurs any selected text will be unselected, the
    cursor position is set to 0 and the first part of the string is
    shown.

    If the text field has an input mask, the mask defines the maximum
    string length.

    By default, this property contains a value of 32767.

    \sa inputMask
*/

/*!
    \property bool TextField::frame
    \brief Whether the text field draws itself with a frame

    If enabled (the default) the text field draws itself inside a
    frame, otherwise the text field draws itself without any frame.
*/

/*!
    \property enumeration TextField::echoMode
    \brief The text field's echo mode

    The echo mode determines how the text entered in the text field is
    displayed (or echoed) to the user.

    The most common setting is \c Normal, in which the text entered by the
    user is displayed verbatim, but TextField also supports modes that allow
    the entered text to be suppressed or obscured: these include \c NoEcho,
    \c Password and \c PasswordEchoOnEdit.

    The item's display and the ability to copy or drag the text is
    affected by this setting.

    By default, this property is set to \c Normal.

    \sa displayText
*/

/*!
    \property int TextField::cursorPosition
    \brief The current cursor position for this text field

    Setting the cursor position causes a repaint when appropriate.

    By default, this property contains a value of 0.
*/

/*!
    \property flags TextField::alignment
    \brief The alignment of the text field

    Both horizontal and vertical alignment is allowed here, Qt.AlignJustify
    will map to Qt.AlignLeft.

    By default, this property contains a combination of Qt.AlignLeft and Qt.AlignVCenter.
*/

/*!
    \property bool TextField::modified
    \brief Whether the text field's contents has been modified by the user

    The modified flag is never read by TextField; it has a default value
    of false and is changed to true whenever the user changes the line
    edit's contents.

    This is useful for things that need to provide a default value but
    do not start out knowing what the default should be (perhaps it
    depends on other fields on the form). Start the text field without
    the best default, and when the default is known, if \c modified
    is false (the user hasn't entered any text), insert the
    default value.

    Setting \c text programmatically resets the modified flag to false.
*/

/*!
    \property bool TextField::hasSelectedText
    \brief Whether there is any text selected

    \c hasSelectedText returns true if some or all of the text has been
    selected by the user; otherwise returns false.

    By default, this property is false.

    \sa selectedText
*/

/*!
    \property string TextField::selectedText
    \brief The selected text

    If there is no selected text this property's value is
    an empty string.

    By default, this property contains an empty string.

    \sa hasSelectedText
*/

/*!
    \property bool TextField::undoAvailable
    \brief Whether undo is available

    Undo becomes available once the user has modified the text in the text field.

    By default, this property is false.
*/

/*!
    \property bool TextField::redoAvailable
    \brief Whether redo is available

    Redo becomes available once the user has performed one or more undo operations
    on text in the text field.

    By default, this property is false.
*/

/*!
    \property bool TextField::dragEnabled
    \brief Whether the lineedit starts a drag if the user presses and
    moves the mouse on some selected text

    Dragging is disabled by default.
*/

/*!
    \property TextField::acceptableInput
    \brief whether the input satisfies the inputMask and the
    validator.

    By default, this property is true.

    \sa inputMask, validator
*/

/*!
    \property enumeration TextField::inputMask
    \brief The validation input mask

    If no mask is set, \c inputMask is an empty string.

    Sets the TextField's validation mask. Validators can be used
    instead of, or in conjunction with masks; see \link validator\endlink.

    Unset the mask and return to normal TextField operation by passing
    an empty string ("").

    The mask consists of a string of mask characters and separators,
    optionally followed by a semicolon and the character used for
    blanks. The blank characters are always removed from the text
    after editing.

    To get range control (e.g., for an IP address) use masks together
    with \link validator\endlink.

    \sa maxLength
*/

/*!
    \property QValidator TextField::validator
    \brief The validator used to test if the input is acceptable
    
    \sa acceptableInput
*/

/*!
    \property bool TextField::readOnly
    \brief Whether the text field is read only.

    In read-only mode, the user can still copy the text to the
    clipboard, or drag and drop the text (if \c echoMode is \c Normal),
    but cannot edit it.

    TextField does not show a cursor in read-only mode.

    By default, this property is false.

    \sa enabled
*/

/*!
    \fn void TextField::cut()
    
    Copies the selected text to the clipboard and deletes it, if there
    is any, and if \c echoMode is \c Normal.

    If the current validator disallows deleting the selected text,
    cut() will copy without deleting.

    \sa copy(), paste(), validator
*/

/*!
    \fn void TextField::copy()
    
    Copies the selected text to the clipboard, if there is any, and if
    \c echoMode is \c Normal.

    \sa cut(), paste()
*/

/*!
    \fn void TextField::paste()
    
    Inserts the clipboard's text at the cursor position, deleting any
    selected text, providing the text field is not \c readOnly.

    If the end result would not be acceptable to the current
    \link validator\endlink, nothing happens.

    \sa copy(), cut()
*/

/*!
    \fn void TextField::selectAll()
    
    Selects all the text (i.e. highlights it) and moves the cursor to
    the end. This is useful when a default value has been inserted
    because if the user types before clicking on the item, the
    selected text will be deleted
*/

/*!
    \fn void TextField::clear()
    
    Clears the contents of the text field.

    \sa text
*/

/*!
    \fn void TextField::undo()
    
    Undoes the last operation if \link undoAvailable\endlink is true. 
    Deselects any current selection, and updates the selection start 
    to the current cursor position.
*/

/*!
    \fn void TextField::redo()
    
    Redoes the last operation if \link redoAvailable\endlink is true.
*/

/*!
    \fn void TextField::textEdited(string text)

    This signal is emitted whenever the text is edited. The \a text
    argument is the new text.

    This signal is not emitted when the text is
    changed programmatically.
*/

/*!
    \fn void TextField::selectionChanged()

    This signal is emitted whenever the selection changes.

    \sa hasSelectedText, selectedText
*/

/*!
    \fn void TextField::returnPressed()

    This signal is emitted when the Return or Enter key is pressed.
    Note that if there is a \c validator or \c inputMask set on the line
    edit, the returnPressed() signal will only be emitted if the input
    follows the \c inputMask and the \c validator returns
    QValidator::Acceptable.
*/

/*!
    \fn void TextField::editingFinished()

    This signal is emitted when the Return or Enter key is pressed or
    the text field loses focus. Note that if there is a \c validator or
    \c inputMask set on the text field and enter/return is pressed, the
    editingFinished() signal will only be emitted if the input follows
    the \c inputMask and the \c validator returns QValidator::Acceptable.
*/

};

#endif // TEXTFIELD_P_H
