/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TEXTEDIT_P_H
#define TEXTEDIT_P_H

class TextEdit : public Flickable
{

    Q_PROPERTY(flags alignment READ alignment WRITE setAlignment)
    Q_PROPERTY(enumeration autoFormatting READ autoFormatting WRITE setAutoFormatting)
    Q_PROPERTY(bool tabChangesFocus READ tabChangesFocus WRITE setTabChangesFocus)
    Q_PROPERTY(string documentTitle READ documentTitle WRITE setDocumentTitle)
    Q_PROPERTY(bool undoRedoEnabled READ isUndoRedoEnabled WRITE setUndoRedoEnabled)
    Q_PROPERTY(enumeration lineWrapMode READ lineWrapMode WRITE setLineWrapMode)
    Q_PROPERTY(int lineWrapColumnOrWidth READ lineWrapColumnOrWidth WRITE setLineWrapColumnOrWidth)
    Q_PROPERTY(enumeration wordWrapMode READ wordWrapMode WRITE setWordWrapMode)
    Q_PROPERTY(bool readOnly READ isReadOnly WRITE setReadOnly)
    Q_PROPERTY(string html READ html WRITE setHtml)
    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(bool overwriteMode READ overwriteMode WRITE setOverwriteMode)
    Q_PROPERTY(int tabStopWidth READ tabStopWidth WRITE setTabStopWidth)
    Q_PROPERTY(bool acceptRichText READ acceptRichText WRITE setAcceptRichText)
    Q_PROPERTY(int cursorWidth READ cursorWidth WRITE setCursorWidth)
    Q_PROPERTY(flags textInteractionFlags READ textInteractionFlags WRITE setTextInteractionFlags)
    Q_PROPERTY(color textColor READ textColor WRITE setTextColor)
    Q_PROPERTY(color textBackgroundColor READ textBackgroundColor WRITE setTextBackgroundColor)
    Q_PROPERTY(bool undoAvailable READ isUndoAvailable)
    Q_PROPERTY(bool redoAvailable READ isRedoAvailable)
    Q_PROPERTY(bool copyAvailable READ isCopyAvailable)

public:
    void clear();
    void copy();
    void cut();
    void paste();
    void redo();
    void undo();
    void selectAll();
    void append(string text);
    void scrollToAnchor(string name);
    void zoomIn(int range = 1);
    void zoomOut(int range = 1);
    
signals:
    void selectionChanged();

/*!
    \class TextEdit
    \brief The TextEdit component provides a widget that is used to edit and display
    both plain and rich text.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-text

    TextEdit is an advanced WYSIWYG viewer/editor supporting rich
    text formatting using HTML-style tags. It is optimized to handle
    large documents and to respond quickly to user input.

    TextEdit works on paragraphs and characters. A paragraph is a
    formatted string which is word-wrapped to fit into the width of
    the widget. By default when reading plain text, one newline
    signifies a paragraph. A document consists of zero or more
    paragraphs. The words in the paragraph are aligned in accordance
    with the paragraph's alignment. Paragraphs are separated by hard
    line breaks. Each character within a paragraph has its own
    attributes, for example, font and color.

    TextEdit can display images, lists and tables. If the text is
    too large to view within the text edit's viewport, scroll bars will
    appear. The text edit can load both plain text and HTML files (a
    subset of HTML 3.2 and 4).

    If you just need to display a small piece of rich text use Label.
*/

/*!
    \property TextEdit::alignment
    \brief The alignment of the current paragraph
    
    Possible values are:
    
    \li \c Qt.AlignLeft
    \li \c Qt.AlignRight
    \li \c Qt.AlignJustify
    \li \c Qt.AlignCenter
*/

/*!
    \property TextEdit::undoRedoEnabled
    \brief Whether undo and redo are enabled

    Users are only able to undo or redo actions if this property is
    true, and if there is an action that can be undone (or redone).
*/

/*!
    \property string TextEdit::text

    This property gets and sets the text editor's contents as plain
    text. Previous contents are removed and undo/redo history is reset
    when the property is set.

    By default, for an editor with no contents, this property contains
    an empty string.

    \sa html
*/

/*!
    \property string TextEdit::html

    This property provides an HTML interface to the text of the text edit.

    When settings this property, any previous text is removed and 
    the undo/redo history is cleared. The input text is
    interpreted as rich text in html format.

    \note It is the responsibility of the caller to make sure that the
    text is correctly decoded.

    By default, for a newly-created, empty document, this property contains
    text to describe an HTML 4.0 document with no body text.

    \sa text
*/

/*!
    \property string TextEdit::documentTitle
    \brief The title of the document parsed from the text.

    By default, for a newly-created, empty document, this property contains
    an empty string.
*/

/*!
    \property bool TextEdit::overwriteMode
    \brief Whether text entered by the user will overwrite existing text

    As with many text editors, the text editor widget can be configured
    to insert or overwrite existing text with new text entered by the user.

    If this property is true, existing text is overwritten, character-for-character
    by new text; otherwise, text is inserted at the cursor position, displacing
    existing text.

    By default, this property is false (new text does not overwrite existing text).
*/

/*!
    \property int TextEdit::tabStopWidth
    \brief The tab stop width in pixels

    By default, this property contains a value of 80 pixels.
*/

/*!
    \property int TextEdit::cursorWidth
    \brief The width of the cursor in pixels.
    
    The default value is 1.
*/

/*!
    \property bool TextEdit::acceptRichText
    \brief Whether the text edit accepts rich text insertions by the user

    When this propery is set to false text edit will accept only
    plain text input from the user. For example through clipboard or drag and drop.

    This property's default is true.
*/

/*!
    \property bool TextEdit::readOnly
    \brief Whether the text edit is read-only

    In a read-only text edit the user can only navigate through the
    text and select text; modifying the text is not possible.

    This property's default is false.
*/

/*!
    \property flags TextEdit::textInteractionFlags
    \brief How the text edit should interact with user input.
    
    Available flags are:
    
    \li \c Qt.NoTextInteraction - No interaction with the text is possible.
    \li \c Qt.TextSelectableByMouse - Text can be selected with the mouse 
    and copied to the clipboard using a context menu or standard keyboard shortcuts.
    
    \li \c Qt.TextSelectableByKeyboard - Text can be selected with the cursor 
    keys on the keyboard. A text cursor is shown.
    
    \li \c Qt.LinksAccessibleByMouse - Links can be highlighted and activated 
    with the mouse.
    
    \li \c Qt.LinksAccessibleByKeyboard - Links can be focused using tab and 
    activated with enter.
    
    \li \c Qt.TextEditable - The text is fully editable.

    The default value depends on whether the TextEdit is read-only
    or editable.
*/

/*!
    \property flags TextEdit::autoFormatting
    \brief The enabled set of auto formatting features

    The available flags are:
    
    \li \c AutoNone - Don't do any automatic formatting.
    \li \c AutoBulletList - Automatically create bullet lists.
    
    The default value is AutoNone.
*/

/*! 
    \property bool TextEdit::tabChangesFocus
    \brief Whether Tab changes focus or is accepted as input

    In some occasions text edits should not allow the user to input
    tabulators or change indentation using the Tab key, as this breaks
    the focus chain. The default is false.
*/

/*!
    \property enumeration TextEdit::lineWrapMode
    \brief The line wrap mode

    The possible values are:
    
    \li \c NoWrap
    \li \c WidgetWidth
    \li \c FixedPixelWidth
    \li \c FixedColumnWidth
    
    The default value is \c WidgetWidth.

    \sa lineWrapColumnOrWidth
*/

/*!
    \property int TextEdit::lineWrapColumnOrWidth
    \brief The position (in pixels or columns depending on the wrap mode) where text will be wrapped

    If the wrap mode is \c FixedPixelWidth, the value is the number of
    pixels from the left edge of the text edit at which text should be
    wrapped. If the wrap mode is \c FixedColumnWidth, the value is the
    column number (in character columns) from the left edge of the
    text edit at which text should be wrapped.

    By default, this property contains a value of 0.

    \sa lineWrapMode
*/

/*!
    \property enumeration TextEdit::wordWrapMode
    \brief The mode TextEdit will use when wrapping text by words

    Possible values are:
    
    \li \c NoWrap,
    \li \c WordWrap
    \li \c ManualWrap
    \li \c WrapAnywhere
    \li \c WrapAtWordBoundaryOrAnywhere
    
    The default value is \c WrapAtWordBoundaryOrAnywhere.
*/

/*!
    \property color TextEdit::textColor
    \brief The color of the current format
*/

/*!
    \property color TextEdit::textBackgroundColor
    \brief The background color of the current format
*/

/*!
    \property bool TextEdit::undoAvailable
    \brief Whether it is possible to undo()
*/

/*!
    \property bool TextEdit::redoAvailable
    \brief Whether it is possible to redo()
*/

/*!
    \property bool TextEdit::copyAvailable
    \brief Whether it is possible to copy()
*/

/*!
    \fn void TextEdit::cut()
    
    Copies the selected text to the clipboard and deletes it.

    If the current validator disallows deleting the selected text,
    cut() will copy without deleting.

    \sa copy(), paste()
*/

/*!
    \fn void TextEdit::copy()
    
    Copies the selected text to the clipboard, if there is any.

    \sa cut(), paste()
*/

/*!
    \fn void TextEdit::paste()
    
    Inserts the clipboard's text at the cursor position, deleting any
    selected text.

    \sa copy(), cut()
*/

/*!
    \fn void TextEdit::selectAll()
    
    Selects all the text (i.e. highlights it) and moves the cursor to
    the end. This is useful when a default value has been inserted
    because if the user types before clicking on the item, the
    selected text will be deleted
*/

/*!
    \fn void TextEdit::clear()
    
    Clears the contents of the text edit.

    \sa text
*/

/*!
    \fn void TextEdit::undo()
    
    Undoes the last operation if \link undoAvailable\endlink is true. 
    Deselects any current selection, and updates the selection start 
    to the current cursor position.
*/

/*!
    \fn void TextEdit::redo()
    
    Redoes the last operation if \link redoAvailable\endlink is true.
*/

/*!
    \fn void TextEdit::zoomIn(int range)

    Zooms in on the text by making the base font size \a range
    points larger and recalculating all font sizes to be the new size.
    This does not change the size of any images.

    \sa zoomOut()
*/

/*!
    \fn void TextEdit::zoomOut(int range)

    Zooms out on the text by making the base font size \a range points
    smaller and recalculating all font sizes to be the new size. This
    does not change the size of any images.

    \sa zoomIn()
*/

/*!
    \fn void TextEdit::append(string text)
    
    Appends a new paragraph with \a text to the end of the text edit.

    \note The new paragraph appended will have the same character format and
    block format as the current paragraph, determined by the position of the cursor.
*/

/*!
    \fn void TextEdit::scrollToAnchor(string name)
    
    Scrolls the text edit so that the anchor with the given \a name is
    visible; does nothing if the \a name is empty, or is already
    visible, or isn't found.
*/

/*!
    \fn void TextEdit::selectionChanged()

    This signal is emitted whenever the selection changes.
*/

};

#endif // TEXTEDIT_P_H
