/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef TEXTBROWSER_P_H
#define TEXTBROWSER_P_H

class TextBrowser : public TextEdit
{

    Q_PROPERTY(url source READ source WRITE setSource)
    Q_PROPERTY(stringlist searchPaths READ searchPaths WRITE setSearchPaths)
    Q_PROPERTY(bool backwardAvailable READ isBackwardAvailable)
    Q_PROPERTY(bool forwardAvailable READ isForwardAvailable)
    Q_PROPERTY(int backwardHistoryCount READ backwardHistoryCount)
    Q_PROPERTY(int forwardHistoryCount READ forwardHistoryCount)
    Q_PROPERTY(bool openLinks READ openLinks WRITE setOpenLinks)
    Q_PROPERTY(bool openExternalLinks READ openExternalLinks WRITE setOpenExternalLinks)

public:
    string historyTitle(int i);
    url historyUrl(int i);
    
    void backward();
    void forward();
    void home();
    void reload();
    void clearHistory();
    
signals:
    void highlighted(url link);
    void anchorClicked(url link);
    
/*!
    \class TextBrowser
    \brief The TextBrowser component provides a rich text browser with hypertext navigation.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-text

    This component extends TextEdit (in read-only mode), adding some navigation
    functionality so that users can follow links in hypertext documents.

    If you want to provide your users with an editable rich text editor,
    use TextEdit. If you want a text browser without hypertext navigation
    use TextEdit, and use TextEdit::readOnly to disable
    editing. If you just need to display a small piece of rich text
    use Label.
*/

/*!
    \property url TextBrowser::source
    \brief the name of the displayed document.

    This is a an invalid url if no document is displayed or if the
    source is unknown.

    When setting this property TextBrowser tries to find a document
    with the specified name in the paths of the searchPaths property
    and directory of the current source, unless the value is an absolute
    file path. It also checks for optional anchors and scrolls the document
    accordingly

    If the first tag in the document is \c \<qt type=detail\>, the
    document is displayed as a popup rather than as new document in
    the browser window itself. Otherwise, the document is displayed
    normally in the text browser with the text set to the contents of
    the named document with \link html\endlink.

    By default, this property contains an empty URL.
*/

/*!
    \property stringlist TextBrowser::searchPaths
    \brief the search paths used by the text browser to find supporting
    content

    TextBrowser uses this list to locate images and documents.

    By default, this property contains an empty string list.
*/

/*!
    \property bool TextBrowser::backwardAvailable

    This property is false when the user is at home();
    otherwise it is true.
*/

/*!
    \property bool TextBrowser::forwardAvailable

    This property is true after the user navigates backward() and false
    when the user navigates or goes forward().
*/

/*!
    \property int TextBrowser::forwardHistoryCount
    \brief The number of locations forward in the history.
*/

/*!
    \property int TextBrowser::backwardHistoryCount
    \brief The number of locations backward in the history.
*/

/*!
    \property bool TextBrowser::openExternalLinks

    Specifies whether TextBrowser should automatically open links to external
    sources using Qt.openUrlExternally() instead of emitting the
    anchorClicked() signal. Links are considered external if their scheme is
    neither file or qrc.

    The default value is false.
*/

/*!
   \property bool TextBrowser::openLinks

   This property specifies whether TextBrowser should automatically open links the user tries to
   activate by mouse or keyboard.

   Regardless of the value of this property the anchorClicked() signal is always emitted.

   The default value is true.
*/

/*!
    \fn string TextBrowser::historyTitle(int i)
    
    Returns the documentTitle() of the HistoryItem.
*/

/*!
    \fn url TextBrowser::historyUrl(int i)
    
    Returns the url of the HistoryItem.
*/

/*!
    \fn void TextBrowser::backward()
    
    Changes the document displayed to the previous document in the
    list of documents built by navigating links. Does nothing if there
    is no previous document.

    \sa forward(), backwardAvailable
*/

/*!
    \fn void TextBrowser::forward()
    
    Changes the document displayed to the next document in the list of
    documents built by navigating links. Does nothing if there is no
    next document.

    \sa backward(), forwardAvailable
*/

/*!
    \fn void TextBrowser::home()
    
    Changes the document displayed to be the first document from
    the history.
*/

/*!
    \fn void TextBrowser::reload()
    
    Reloads the current set source.
*/

/*!
    \fn void TextBrowser::clearHistory()

    Clears the history of visited documents and disables the forward and
    backward navigation.

    \sa backward(), forward()
*/

/*!
    \fn void TextBrowser::anchorClicked(url link)

    This signal is emitted when the user clicks an anchor. The
    URL referred to by the anchor is passed in \a link.

    Note that the browser will automatically handle navigation to the
    location specified by \a link unless the openLinks property
    is set to false.
    
    This mechanism is used to override the default navigation 
    features of the browser.
*/

/*!  \fn void TextBrowser::highlighted(url link)

    This signal is emitted when the user has selected but not
    activated an anchor in the document. The URL referred to by the
    anchor is passed in \a link.
*/
    
};

#endif // TEXTBROWSER_P_H
