/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MOUSEAREA_P_H
#define MOUSEAREA_P_H

class MouseEvent
{

    Q_PROPERTY(int x READ x)
    Q_PROPERTY(int y READ y)
    Q_PROPERTY(int button READ button)
    Q_PROPERTY(int buttons READ buttons)
    Q_PROPERTY(int modifiers READ modifiers)
    Q_PROPERTY(bool wasHeld READ wasHeld)
    Q_PROPERTY(bool isClick READ isClick)
    Q_PROPERTY(bool accepted READ isAccepted WRITE setAccepted)
    
/*!
    \class MouseEvent
    \brief The MouseEvent object provides information about a mouse event.
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-events

    \sa MouseArea
*/

/*!
    \property real MouseEvent::x
    
    The x coordinate of the mouse event.
*/

/*!
    \property real MouseEvent::y
    
    The y coordinate of the mouse event.
*/


/*!
    \property bool MouseEvent::accepted

    Setting \a accepted to true prevents the mouse event from being
    propagated to items below this item.

    Generally, if the item acts on the mouse event then it should be accepted
    so that items lower in the stacking order do not also respond to the same event.
*/

/*!
    \property enumeration MouseEvent::button

    This property holds the button that caused the event.  It can be one of:

    \li \c Qt.LeftButton
    \li \c Qt.RightButton
    \li \c Qt.MiddleButton
*/

/*!
    \property bool MouseEvent::isClick
    
    This property is true if the event is a mouse click event.
*/

/*!
    \property bool MouseEvent::wasHeld

    This property is true if the mouse button has been held pressed longer the
    threshold (800ms).
*/

/*!
    \property int MouseEvent::buttons

    This property holds the mouse buttons pressed when the event was generated.
    For mouse move events, this is all buttons that are pressed down. For mouse
    press and double click events this includes the button that caused the event.
    For mouse release events this excludes the button that caused the event.

    It contains a bitwise combination of:

    \li \c Qt.LeftButton
    \li \c Qt.RightButton
    \li \c Qt.MiddleButton
*/

/*!
    \property int MouseEvent::modifiers

    This property holds the keyboard modifier flags that existed immediately
    before the event occurred.

    It contains a bitwise combination of:

    \li \c Qt.NoModifier - No modifier key is pressed.
    \li \c Qt.ShiftModifier - A Shift key on the keyboard is pressed.
    \li \c Qt.ControlModifier - A Ctrl key on the keyboard is pressed.
    \li \c Qt.AltModifier - An Alt key on the keyboard is pressed.
    \li \c Qt.MetaModifier - A Meta key on the keyboard is pressed.
    \li \c Qt.KeypadModifier - A keypad button is pressed.

    For example, to react to a Shift key + Left mouse button click:
    \code
    MouseArea {
        onClicked: {
            if ((mouse.button == Qt.LeftButton) && (mouse.modifiers & Qt.ShiftModifier))
                doSomething();
        }
    }
    \endcode
*/

};

class MouseArea : public Item
{
    Q_OBJECT

    Q_PROPERTY(bool containsMouse READ containsMouse NOTIFY containsMouseChanged)
    Q_PROPERTY(int mouseX READ mouseX NOTIFY positionChanged)
    Q_PROPERTY(int mouseY READ mouseY NOTIFY positionChanged)
    Q_PROPERTY(bool pressed READ isPressed NOTIFY pressedChanged)
    Q_PROPERTY(bool preventStealing READ preventStealing WRITE setPreventStealing NOTIFY preventStealingChanged)
    Q_PROPERTY(MouseDrag drag READ drag CONSTANT FINAL)

signals:
    void canceled();
    void entered();
    void exited();
    void positionChanged(MouseEvent mouse);
    void pressed(MouseEvent mouse);
    void pressAndHold(MouseEvent mouse);
    void released(MouseEvent mouse);
    void clicked(MouseEvent mouse);
    void doubleClicked(MouseEvent mouse);
    
/*!
    \class MouseArea
    \brief The MouseArea component enables simple mouse handling.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-interaction

    A MouseArea is an invisible item that is typically used in conjunction with
    a visible item in order to provide mouse handling for that item.
    By effectively acting as a proxy, the logic for mouse handling can be
    contained within a MouseArea item.

    For basic key handling, see the \link Keys\endlink attached property.

    The \link enabled\endlink property is used to enable and disable mouse handling for
    the proxied item. When disabled, the mouse area becomes transparent to
    mouse events.

    The \link pressed\endlink read-only property indicates whether or not the user is
    holding down a mouse button over the mouse area. This property is often
    used in bindings between properties in a user interface. The containsMouse
    read-only property indicates the presence of the mouse cursor over the
    mouse area but, by default, only when a mouse button is held down; see below
    for further details.

    Information about the mouse position and button clicks are provided via
    signals for which event handler properties are defined. The most commonly
    used involved handling mouse presses and clicks: clicked(), doubleClicked(),
    pressed(), released() and pressAndHold().

    By default, MouseArea items only report mouse clicks and not changes to the
    position of the mouse cursor. Setting the hoverEnabled property ensures that
    handlers defined for onPositionChanged, onEntered and onExited are used and
    that the containsMouse property is updated even when no mouse buttons are
    pressed.
*/

/*!
    \property real MouseArea::mouseX
    \brief The x coordinate of the mouse cursor.
*/

/*!
    \property real MouseArea::mouseY
    \brief The y coordinate of the mouse cursor.
*/

/*!
    \property bool MouseArea::preventStealing
    \brief Whether the mouse events may be stolen from this
    MouseArea.

    If a MouseArea is placed within an item that filters child mouse
    events, such as Flickable, the mouse
    events may be stolen from the MouseArea if a gesture is recognized
    by the parent element, e.g. a flick gesture.  If preventStealing is
    set to true, no element will steal the mouse events.

    Note that setting preventStealing to true once an element has started
    stealing events will have no effect until the next press event.

    By default this property is false.
*/

/*!
    \property bool MouseArea::containsMouse
    \brief Whether the mouse is currently inside the mouse area.

    \warning This property is not updated if the area moves under the mouse: 
    \e containsMouse will not change.
*/

/*!
    \property bool MouseArea::pressed
    \brief Whether the mouse area is currently pressed.
*/

/*!
    \property MouseDrag MouseArea::drag

    \c drag provides a convenient way to make an item draggable.

    \li \c drag.target specifies the id of the item to drag.
    \li \c drag.active specifies if the target item is currently being dragged.
    \li \c drag.axis specifies whether dragging can be done horizontally (\c Drag.XAxis), vertically (\c Drag.YAxis), or both (\c Drag.XandYAxis)
    \li \c drag.minimum and \c drag.maximum limit how far the target can be dragged along the corresponding axes.
*/

/*!
    \fn void MouseArea::entered()

    This signal is emitted when the mouse enters the mouse area.
*/

/*!
    \fn void MouseArea::exited()

    This signal is emitted when the mouse exits the mouse area.
*/

/*!
    \fn void MouseArea::positionChanged(MouseEvent mouse)

    This signal is emitted when the mouse position changes.

    The \e mouse parameter provides information about the mouse, including the x and y
    position, and any buttons currently pressed.

    The \e accepted property of the MouseEvent parameter is ignored in this handler.
*/

/*!
    \fn void MouseArea::clicked(MouseEvent mouse)

    This signal is emitted when there is a click. A click is defined as a press followed by a release,
    both inside the MouseArea (pressing, moving outside the MouseArea, and then moving back inside and
    releasing is also considered a click).

    The \e mouse parameter provides information about the click, including the x and y
    position of the release of the click, and whether the click was held.

    The \e accepted property of the MouseEvent parameter is ignored in this handler.
*/

/*!
    \fn void MouseArea::pressed(MouseEvent mouse)

    This signal is emitted when there is a press.
    The \e mouse parameter provides information about the press, including the x and y
    position and which button was pressed.

    The \e accepted property of the MouseEvent parameter determines whether this MouseArea
    will handle the press and all future mouse events until release.  The default is to accept
    the event and not allow other MouseArea beneath this one to handle the event.  If \e accepted
    is set to false, no further events will be sent to this MouseArea until the button is next
    pressed.
*/

/*!
    \fn void MouseArea::released(MouseEvent mouse)

    This signal is emitted when there is a release.
    The \e mouse parameter provides information about the click, including the x and y
    position of the release of the click, and whether the click was held.

    The \e accepted property of the MouseEvent parameter is ignored in this handler.

    \sa canceled()
*/

/*!
    \fn void MouseArea::pressAndHold(MouseEvent mouse)

    This signal is emitted when there is a long press (currently 800ms).
    The \e mouse parameter provides information about the press, including the x and y
    position of the press, and which button is pressed.

    The \e accepted property of the MouseEvent parameter is ignored in this handler.
*/

/*!
    \fn void MouseArea::doubleClicked(MouseEvent mouse)

    This signal is emitted when there is a double-click (a press followed by a release followed by a press).
    The \e mouse parameter provides information about the click, including the x and y
    position of the release of the click, and whether the click was held.

    If the \e accepted property of the \e mouse parameter is set to false
    in the handler, the pressed()/released()/clicked() signals will be emitted for the second
    click; otherwise they are suppressed.  The accepted property defaults to true.
*/

/*!
    \fn void MouseArea::canceled()

    This signal is emitted when mouse events have been canceled, 
    either because an event was not accepted, or
    because another item stole the mouse event handling.
*/

};

#endif // MOUSEAREA_P_H
