/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef LISTVIEW_P_H
#define LISTVIEW_P_H

class ListView : public Flickable
{

    // QAbstractItemView
    Q_PROPERTY(flags editTriggers READ editTriggers WRITE setEditTriggers)
    Q_PROPERTY(enumeration selectionMode READ selectionMode WRITE setSelectionMode)
    Q_PROPERTY(enumeration selectionBehavior READ selectionBehavior WRITE setSelectionBehavior)
    Q_PROPERTY(size iconSize READ iconSize WRITE setIconSize)
    Q_PROPERTY(enumeration textElideMode READ textElideMode WRITE setTextElideMode)
    Q_PROPERTY(enumeration verticalScrollMode READ verticalScrollMode WRITE setVerticalScrollMode)
    Q_PROPERTY(enumeration horizontalScrollMode READ horizontalScrollMode WRITE setHorizontalScrollMode)
    
    // QListView
    Q_PROPERTY(enumeration movement READ movement WRITE setMovement)
    Q_PROPERTY(enumeration flow READ flow WRITE setFlow)
    Q_PROPERTY(bool isWrapping READ isWrapping WRITE setWrapping)
    Q_PROPERTY(enumeration resizeMode READ resizeMode WRITE setResizeMode)
    Q_PROPERTY(enumeration layoutMode READ layoutMode WRITE setLayoutMode)
    Q_PROPERTY(int spacing READ spacing WRITE setSpacing)
    Q_PROPERTY(size gridSize READ gridSize WRITE setGridSize)
    Q_PROPERTY(enumeration viewMode READ viewMode WRITE setViewMode)
    Q_PROPERTY(int modelColumn READ modelColumn WRITE setModelColumn)
    Q_PROPERTY(bool uniformItemSizes READ uniformItemSizes WRITE setUniformItemSizes)
    Q_PROPERTY(int batchSize READ batchSize WRITE setBatchSize)
    Q_PROPERTY(bool wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(bool selectionRectVisible READ isSelectionRectVisible WRITE setSelectionRectVisible)
    
    // ListView
    Q_PROPERTY(variant model READ model WRITE setModel)
    Q_PROPERTY(QModelIndex rootIndex READ rootIndex WRITE setRootIndex)
    Q_PROPERTY(QModelIndex currentIndex READ currentIndex WRITE setCurrentIndex)
    Q_PROPERTY(list<QModelIndex> selectedIndexes READ selectedIndexes WRITE setSelectedIndexes)
    Q_PROPERTY(ListItem delegate READ delegate WRITE setDelegate)

public:
    // QAbstractItemView
    void reset();
    void selectAll();
    void edit(QModelIndex index);
    void clearSelection();
    void update(QModelIndex index);
    
    //ListView
    void positionViewAtBeginning();
    void positionViewAtEnd();
    void positionViewAtIndex(QModelIndex index, enumeration hint);
    
signals:
    // QAbstractItemView
    void pressed();
    void clicked();
    void doubleClicked();
    void activated();

/*!
    \class ListView
    \brief The ListView component provides an item view for displaying model data
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-views
    
*/

/*!
    \property variant ListView::model
    \brief The \a model for the view to present

    The view <em>does not</em> take ownership of the model unless it is the model's
    parent object because the model may be shared between many different views.
    
    The model can be any of the following types:
    
    \li \c int
    \li \c string
    \li \c list
    \li \c stringlist
    \li \c ListModel
    \li \c QAbstractItemModel

    \sa ListModel
*/

/*!
    \property ListItem ListView::delegate
    \brief The item delegate for this view and its model
    
    This is useful if you want complete control over the editing and
    display of items.

    Any existing delegate will be removed, but not deleted. The view
    does not take ownership of the delegate.

    \warning You should not share the same instance of a delegate between views.
    Doing so can cause incorrect or unintuitive editing behavior since each
    view connected to a given delegate may receive the ListItem::closeEditor()
    signal, and attempt to access, modify or close an editor that has already been closed.
    
    \sa ListItem
*/

/*!
    \property enumeration ListView::selectionMode
    \brief Which selection mode the view operates in

    This property controls whether the user can select one or many items
    and, in many-item selections, whether the selection must be a
    continuous range of items.
    
    Possible values are:
    
    \li \c SingleSelection - When the user selects an item, 
    any already-selected item becomes unselected, 
    and the user cannot unselect the selected 
    item by clicking on it.
    
    \li \c ContiguousSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Shift key while 
    clicking on an item, all items between the current item and 
    the clicked item are selected or unselected, depending on 
    the state of the clicked item.
    
    \li \c ExtendedSelection - When the user selects an item 
    in the usual way, the selection is cleared and the new item 
    selected. However, if the user presses the Ctrl key when 
    clicking on an item, the clicked item gets toggled and 
    all other items are left untouched. If the user presses 
    the Shift key while clicking on an item, all items between 
    the current item and the clicked item are selected or 
    unselected, depending on the state of the clicked item. 
    Multiple items can be selected by dragging the mouse 
    over them.
    
    \li \c MultiSelection - When the user selects an item 
    in the usual way, the selection status of that item is 
    toggled and the other items are left alone. Multiple 
    items can be toggled by dragging the mouse over them.
    
    \li \c NoSelection - Items cannot be selected.
    
    The default value is \c SingleSelection.

    \sa selectionBehavior
*/

/*!
    \property enumeration ListView::selectionBehavior
    \brief Which selection behavior the view uses

    This property holds whether selections are done
    in terms of single items, rows or columns.
    
    Possible values are:
    
    \li \c SelectItems - Selecting single items.
    \li \c SelectRows - Select only rows.
    \li \c SelectColumns - Select only columns.
    
    The default value is \c SelectItems.

    \sa selectionMode
*/

/*!
    \property QModelIndex ListView::currentIndex
    \brief The model index of the current item

    \sa selectedIndexes, selectionBehavior, selectionMode
*/

/*!
    \property QModelIndex ListView::rootIndex
    \brief The model index of the model's root item. 
    
    The root item is the parent item to the view's toplevel items. 
    The root can be invalid.
*/

/*!
    \property list<QModelIndex> ListView::selectedIndexes
    \brief The model indexes of the selected items
    
    \sa currentIndex, selectionBehavior, selectionMode
*/

/*!
    \property flags ListView::editTriggers
    \brief Which actions will initiate item editing

    This property is a selection of flags combined using the OR
    operator. The view will only initiate the editing of an item if the
    action performed is set in this property.
    
    Available flags are:
    
    \li \c NoEditTriggers - No editing possible.
    \li \c CurrentChanged - Editing start whenever current item changes.
    \li \c DoubleClicked - Editing starts when an item is double clicked.
    \li \c SelectedClicked - Editing starts when clicking on an already selected item.
    \li \c EditKeyPressed - Editing starts when the platform edit key has been pressed over an item.
    \li \c AnyKeyPressed - Editing starts when any key is pressed over an item.
    \li \c AllEditTriggers - Editing starts for all above actions.
*/

/*!
    \property enumeration ListView::verticalScrollMode
    \brief How the view scrolls its contents in the vertical direction

    This property controls how the view scroll its contents vertically.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa horizontalScrollMode
*/

/*!
    \property enumeration ListView::horizontalScrollMode
    \brief How the view scrolls its contents in the horizontal direction

    This property controls how the view scroll its contents horizontally.
    
    Possible values are:
    
    \li \c ScrollPerItem - The view will scroll the contents one item at a time.
    \li \c ScrollPerPixel - The view will scroll the contents one pixel at a time.
    
    The default value is ScrollPerPixel.
    
    \sa verticalScrollMode
*/

/*!
    \property size ListView::iconSize
    \brief The size of items' icons

    Setting this property when the view is visible will cause the
    items to be laid out again.
*/

/*!
    \property enumeration ListView::textElideMode

    \brief The position of the "..." in elided text.
    
    Possible values are:
    
    \li \c Qt.ElideLeft - The ellipsis should appear at the beginning of the text.
    \li \c Qt.ElideRight - The ellipsis should appear at the end of the text.
    \li \c Qt.ElideMiddle - The ellipsis should appear in the middle of the text.
    \li \c Qt.ElideNone - Ellipsis should NOT appear in the text.

    The default value is Qt.ElideRight.
*/

/*!
    \property enumeration ListView::movement
    \brief Whether the items can be moved freely, are snapped to a
    grid, or cannot be moved at all.

    Possible values are:
    
    \li \c Static - The items cannot be moved by the user.
    \li \c Free - The items can be moved freely by the user.
    \li \c Snap - The items snap to the specified grid when moved; \see gridSize.

    Setting this property when the view is visible will cause the
    items to be laid out again.

    By default, this property is set to \c Static.

    \sa gridSize, resizeMode, viewMode
*/

/*!
    \property enumeration ListView::flow
    \brief Which direction the items layout should flow.

    Possible values are:
    
    \li \c LeftToRight - The items are laid out in the view from the left to the right.
    \li \c TopToBottom - The items are laid out in the view from the top to the bottom.

    Setting this property when the view is visible will cause the
    items to be laid out again.

    By default, this property is set to \c TopToBottom.

    \sa viewMode
*/

/*!
    \property bool ListView::isWrapping
    \brief Whether the items layout should wrap.

    This property holds whether the layout should wrap when there is
    no more space in the visible area. The point at which the layout wraps
    depends on the flow property.

    Setting this property when the view is visible will cause the
    items to be laid out again.

    By default, this property is false.

    \sa viewMode
*/

/*!
    \property enumeration ListView::resizeMode
    \brief Whether the items are laid out again when the view is resized.

    Possible values are:
    
    \li \c Fixed - The items will only be laid out the first time the view is shown.
    \li \c Adjust - The items will be laid out every time the view is resized.

    By default, this property is set to \c Fixed.

    \sa movement, gridSize, viewMode
*/

/*!
    \property enumeration ListView::layoutMode
    \brief Determines whether the layout of items should happen immediately or be delayed.

    Possible values are:
    
    \li \c SinglePass - The items are laid out all at once.
    \li \c Batched - The items are laid out in batches of \a batchSize items.

    \sa batchSize, viewMode
*/

/*!
    \property int ListView::spacing
    \brief the space around the items in the layout

    This property is the size of the empty space that is padded around
    an item in the layout.

    Setting this property when the view is visible will cause the
    items to be laid out again.

    By default, this property contains a value of 0.

    \sa viewMode
*/

/*!
    \property int ListView::batchSize
    \brief the number of items laid out in each batch if layoutMode is
    set to \c Batched

    The default value is 100.
*/

/*!
    \property size ListView::gridSize
    \brief The size of the layout grid

    This property is the size of the grid in which the items are laid
    out. The default is an empty size which means that there is no
    grid and the layout is not done in a grid. Setting this property
    to a non-empty size switches on the grid layout. (When a grid
    layout is in force the spacing property is ignored.)

    Setting this property when the view is visible will cause the
    items to be laid out again.

    \sa viewMode
*/

/*!
    \property enumeration ListView::viewMode
    \brief The view mode of the ListView.

    The possible values are:
    
    \li \c ListMode - The items are laid out using \c TopToBottom flow, 
    with Small size and Static movement.
    
    \li \c IconMode - The items are laid out using \c LeftToRight flow, 
    with \c Large size and \c Free movement.
    
    The default value is \c ListMode.

    \sa isWrapping, spacing, gridSize, flow, movement, resizeMode
*/

/*!
    \property int ListView::modelColumn
    \brief The column in the model that is visible

    By default, this property contains 0, indicating that the first
    column in the model will be shown.
*/

/*!
    \property bool ListView::uniformItemSizes
    \brief Whether all items in the listview have the same size

    This property should only be set to true if it is guaranteed that all items
    in the view have the same size. This enables the view to do some
    optimizations for performance purposes.

    By default, this property is false.
*/

/*!
    \property bool ListView::wordWrap
    \brief The item text word-wrapping policy

    If this property is true then the item text is wrapped where
    necessary at word-breaks; otherwise it is not wrapped at all.
    This property is false by default.

    Please note that even if wrapping is enabled, the cell will not be
    expanded to make room for the text. It will print ellipsis for
    text that cannot be shown, according to the view's
    textElideMode.
*/

/*!
    \property bool ListView::selectionRectVisible
    \brief If the selection rectangle should be visible

    If this property is true then the selection rectangle is visible;
    otherwise it will be hidden.

    \note The selection rectangle will only be visible if the selection mode
    is in a mode where more than one item can be selected; i.e., it will not
    draw a selection rectangle if the selection mode is
    \c SingleSelection.

    By default, this property is false.
*/

/*!
    \fn void ListView::reset()
    
    Resets the internal state of the view.

    \warning This function will reset open editors, scroll bar positions,
    selections, etc. Existing changes will not be committed.
*/

/*!
    \fn void ListView::edit(QModelIndex index)
    
    Starts editing the item corresponding to the given \a index if it is
    editable.

    Note that this function does not change the current index. Since the current
    index defines the next and previous items to edit, users may find that
    keyboard navigation does not work as expected. To provide consistent navigation
    behavior, set currentIndex before this function with the same model
    index.
    
    \sa editTriggers
*/

/*!
    \fn void ListView::update(QModelIndex index)

    Updates the area occupied by the given \a index.
*/

/*!
    \fn void ListView::selectAll()
    
    Selects all items in the view.
    This function will use the selection behavior
    set on the view when selecting.

    \sa selectedIndexes, clearSelection()
*/

/*!
    \fn void ListView::clearSelection()
    
    Deselects all selected items. The current index will not be changed.

    \sa selectedIndexes, selectAll()
*/

/*!
    \fn void ListView::positionViewAtBeginning()
    
    Positions the view at the beginning.
*/

/*!
    \fn void ListView::positionViewAtEnd()
    
    Positions the view at the end.
*/

/*!
    \fn void ListView::positionViewAtIndex(QModelIndex index, enumeration hint)
    
    Positions the view at \a index. The exact position depends upon \a hint.
    
    Possible values for \a hint are:
    
    \li \c EnsureVisible - Scroll to ensure that the item is visible.
    \li \c PositionAtTop - Scroll to position the item at the top of the view.
    \li \c PositionAtBottom - Scroll to position the item at the bottom of the view.
    \li \c PositionAtCenter - Scroll to position the item at the center of the view.
*/

/*!
    \fn void ListView::activated()

    This signal is emitted when an item is activated by the user 
    by single- or double-clicking the item, or by
    pressing the Return or Enter key when the item is current.

    \sa clicked(), doubleClicked(), pressed()
*/

/*!
    \fn void ListView::clicked()

    This signal is emitted when an item is activated by the user 
    clicking the item.

    \sa activated(), doubleClicked(), pressed()
*/

/*!
    \fn void ListView::doubleClicked()

    This signal is emitted when an item is activated by the user 
    by double-clicking the item.

    \sa activated(), clicked(), pressed()
*/

/*!
    \fn void ListView::pressed()

    This signal is emitted when an item is pressed by the user.

    \sa activated(), clicked(), doubleClicked()
*/

};

#endif // LISTVIEW_P_H
