/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef LISTITEMIMAGE_P_H
#define LISTITEMIMAGE_P_H

class ListItemImage : public ListItemContent
{

    Q_PROPERTY(url source READ source WRITE setSource)
    Q_PROPERTY(size sourceSize READ sourceSize WRITE setSourceSize)
    Q_PROPERTY(bool asynchronous READ asynchronous WRITE setAsynchronous)
    Q_PROPERTY(bool cache READ cache WRITE setCache)
    Q_PROPERTY(bool mirror READ mirror WRITE setMirror)
    Q_PROPERTY(bool smooth READ smooth WRITE setSmooth)
    Q_PROPERTY(real progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(enumeration fillMode READ fillMode WRITE setFillMode)
    Q_PROPERTY(int paintedWidth READ paintedWidth)
    Q_PROPERTY(int paintedHeight READ paintedHeight)

/*!
    \class ListItemImage
    \brief The ListItemImage component displays an image in ListItem
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-graphics
    \ingroup org-hildon-components-views
    
    \sa ListItem, Image
*/

/*!
    \property enumeration ListItemImage::fillMode

    Set this property to define what happens when the source image has a different size
    than the item.

    \li \c ListItemImage.Stretch - the image is scaled to fit
    \li \c ListItemImage.PreserveAspectFit - the image is scaled uniformly to fit without cropping
    \li \c ListItemImage.PreserveAspectCrop - the image is scaled uniformly to fill, cropping if necessary
    \li \c ListItemImage.Tile - the image is duplicated horizontally and vertically
    \li \c ListItemImage.TileVertically - the image is stretched horizontally and tiled vertically
    \li \c ListItemImage.TileHorizontally - the image is stretched vertically and tiled horizontally


    \image html declarative-qtlogo-stretch.png
    \c Stretch (default)
    
    \code
    ListItemImage {
        width: 130; height: 100
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-preserveaspectfit.png
    \c PreserveAspectFit
    
    \code
    ListItemImage {
        width: 130; height: 100
        fillMode: ListItemImage.PreserveAspectFit
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-preserveaspectcrop.png
    \c PreserveAspectCrop
    
    \code
    ListItemImage {
        width: 130; height: 100
        fillMode: ListItemImage.PreserveAspectCrop
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tile.png
    \c Tile
    
    \code
    ListItemImage {
        width: 120; height: 120
        fillMode: ListItemImage.Tile
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tilevertically.png
    \c TileVertically
    
    \code
    ListItemImage {
        width: 120; height: 120
        fillMode: ListItemImage.TileVertically
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tilehorizontally.png
    \c TileHorizontally
    
    \code
    ListItemImage {
        width: 120; height: 120
        fillMode: ListItemImage.TileHorizontally
        smooth: true
        source: "qtlogo.png"
    }
    \endcode
*/

/*!
    \property real ListItemImage::paintedWidth
    \property real ListItemImage::paintedHeight

    These properties hold the size of the image that is actually painted.
    In most cases it is the same as \c width and \c height, but when using a 
    \c fillMode \c PreserveAspectFit or \c fillMode \c PreserveAspectCrop 
    \c paintedWidth or \c paintedHeight can be smaller or larger than 
    \c width and \c height of the ListItemImage component.
*/

/*!
    \property enumeration ListItemImage::status

    This property holds the status of image loading. 
    It can be one of:

    \li \c ListItemImage.Null - no image has been set
    \li \c ListItemImage.Ready - the image has been loaded
    \li \c ListItemImage.Loading - the image is currently being loaded
    \li \c ListItemImage.Error - an error occurred while loading the image

    Use this status to provide an update or respond to the status change in some way.
    For example, you could:

    \li \c Implement an \c onStatusChanged signal handler:
    
    \code
        ListItemImage {
            id: image
            onStatusChanged: if (image.status == ListItemImage.Ready) console.log('Loaded')
        }
    \endcode

    \li \c Bind to the status value:
    
    \code
        Text { text: image.status == ListItemImage.Ready ? 'Loaded' : 'Not loaded' }
    \endcode

    \sa progress
*/

/*!
    \property real ListItemImage::progress

    This property holds the progress of image loading, from 0.0 (nothing loaded)
    to 1.0 (finished).

    \sa status
*/

/*!
    \property bool ListItemImage::smooth

    Set this property if you want the image to be smoothly filtered when scaled or
    transformed.  Smooth filtering gives better visual quality, but is slower.  If
    the image is displayed at its natural size, this property has no visual or
    performance effect.

    \note Generally scaling artifacts are only visible if the image is stationary on
    the screen.  A common pattern when animating an image is to disable smooth
    filtering at the beginning of the animation and reenable it at the conclusion.
*/

/*!
    \property size ListItemImage::sourceSize

    This property holds the actual width and height of the loaded image.

    Unlike the {Item::}{width} and {Item::}{height} properties, which scale
    the painting of the image, this property sets the actual number of pixels
    stored for the loaded image so that large images do not use more
    memory than necessary. For example, this ensures the image in memory is no
    larger than 1024x1024 pixels, regardless of the ListItemImage's width and 
    height values:

    \code
    Rectangle {
        width: ...
        height: ...

        ListItemImage {
           anchors.fill: parent
           source: "reallyBigImage.jpg"
           sourceSize.width: 1024
           sourceSize.height: 1024
        }
    }
    \endcode

    If the image's actual size is larger than the sourceSize, the image is scaled down.
    If only one dimension of the size is set to greater than 0, the
    other dimension is set in proportion to preserve the source image's aspect ratio.
    (The fillMode is independent of this.)

    If the source is an instrinsically scalable image (eg. SVG), this property
    determines the size of the loaded image regardless of intrinsic size.
    Avoid changing this property dynamically; rendering an SVG is \e slow compared
    to an image.

    If the source is a non-scalable image (eg. JPEG), the loaded image will
    be no greater than this property specifies. For some formats (currently only JPEG),
    the whole image will never actually be loaded into memory.

    The sourceSize can be cleared to the natural size of the image
    by setting sourceSize to \c undefined.
 
    \note \e Changing this property dynamically causes the image source to be reloaded,
    potentially even from the network, if it is not in the disk cache.
*/

/*!
    \property url ListItemImage::source

    ListItemImage can handle any image format supported by Qt, loaded from any URL scheme supported by Qt.

    The URL may be absolute, or relative to the URL of the component.
*/

/*!
    \property bool ListItemImage::asynchronous

    Specifies that images on the local filesystem should be loaded
    asynchronously in a separate thread.  The default value is
    false, causing the user interface thread to block while the
    image is loaded.  Setting \a asynchronous to true is useful where
    maintaining a responsive user interface is more desirable
    than having images immediately visible.

    Note that this property is only valid for images read from the
    local filesystem.  Images loaded via a network resource (e.g. HTTP)
    are always loaded asynchonously.
*/

/*!
    \property bool ListItemImage::cache

    Specifies whether the image should be cached. The default value is
    true. Setting \a cache to false is useful when dealing with large images,
    to make sure that they aren't cached at the expense of small 'ui component' images.
*/

/*!
    \property bool ListItemImage::mirror

    This property holds whether the image should be horizontally inverted
    (effectively displaying a mirrored image).

    The default value is false.
*/

};

#endif // LISTITEMIMAGE_P_H
