/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef LABEL_P_H
#define LABEL_P_H

class Label : public Item
{
    
    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(enumeration textFormat READ textFormat WRITE setTextFormat)
    Q_PROPERTY(flags alignment READ alignment WRITE setAlignment)
    Q_PROPERTY(bool wordWrap READ wordWrap WRITE setWordWrap)
    Q_PROPERTY(int margin READ margin WRITE setMargin)
    Q_PROPERTY(int indent READ indent WRITE setIndent)
    Q_PROPERTY(bool openExternalLinks READ openExternalLinks WRITE setOpenExternalLinks)
    Q_PROPERTY(flags textInteractionFlags READ textInteractionFlags WRITE setTextInteractionFlags)
    Q_PROPERTY(bool hasSelectedText READ hasSelectedText)
    Q_PROPERTY(string selectedText READ selectedText)
    
public:
    void clear();
    
signals:
    void linkActivated(string link);

/*!
    \class Label
    \brief The Label component provides a text display.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-text

    Label is used for displaying text. No user
    interaction functionality is provided. The visual appearance of
    the label can be configured in various ways, and it can be used
    for specifying a focus mnemonic key for another item.
*/

/*!
    \property string Label::text
    \brief The label's text

    Setting the text clears any previous content.

    The text will be interpreted either as plain text or as rich
    text, depending on the text format setting; see \link textFormat\endlink.
    The default setting is Qt.AutoText; i.e. Label will try to
    auto-detect the format of the text set.

    Note that Label is well-suited to display small rich text
    documents, such as small documents that get their document
    specific settings (font, text color, link color) from the label's
    palette and font properties. For large documents, use TextEdit
    in read-only mode instead. TextEdit can also provide a scroll bar
    when necessary.

    \sa textFormat, alignment
*/

/*!
    \property flags Label::alignment
    \brief The alignment of the label's contents

    By default, the contents of the label are left-aligned and vertically-centered.

    \sa text
*/

/*!
    \property bool Label::wordWrap
    \brief The label's word-wrapping policy

    If this property is true then label text is wrapped where
    necessary at word-breaks; otherwise it is not wrapped at all.

    By default, word wrap is disabled.

    \sa text
*/

/*!
    \property int Label::indent
    \brief The label's text indent in pixels

    If a label displays text, the indent applies to the left edge if
    \link alignment\endlink is Qt.AlignLeft, to the right edge if 
    \link alignment\endlink is Qt.AlignRight, to the top edge if 
    \link alignment\endlink is Qt.AlignTop, and to the bottom edge 
    if \link alignment\endlink is Qt.AlignBottom.

    By default, the indent is -1, meaning that an effective indent is
    calculating in the manner described above.

    \sa alignment, margin
*/

/*!
    \property int Label::margin
    \brief The width of the margin

    The margin is the distance between the innermost pixel of the
    frame and the outermost pixel of contents.

    The default margin is 0.

    \sa indent
*/

/*!
    \property bool Label::openExternalLinks

    Specifies whether Label should automatically open links using
    Qt.openUrlExternally() instead of emitting the
    linkActivated() signal.

    \note The \link textInteractionFlags\endlink set on the label need to include
    either \c LinksAccessibleByMouse or \c LinksAccessibleByKeyboard.

    The default value is false.

    \sa textInteractionFlags
*/

/*!
    \property flags Label::textInteractionFlags

    Specifies how the label should interact with user input if it displays text.

    Available flags are:
    
    \li \c Qt.NoTextInteraction - No interaction with the text is possible.
    \li \c Qt.TextSelectableByMouse - Text can be selected with the mouse 
    and copied to the clipboard using a context menu or standard keyboard shortcuts.
    
    \li \c Qt.TextSelectableByKeyboard - Text can be selected with the cursor 
    keys on the keyboard. A text cursor is shown.
    
    \li \c Qt.LinksAccessibleByMouse - Links can be highlighted and activated 
    with the mouse.
    
    \li \c Qt.LinksAccessibleByKeyboard - Links can be focused using tab and 
    activated with enter.
    
    \li \c Qt.TextEditable - The text is fully editable.

    The default value is Qt.LinksAccessibleByMouse.
*/

/*!
    \property bool Label::hasSelectedText
    \brief Whether there is any text selected

    This property is true if some or all of the text has been
    selected by the user.

    By default, this property is false.

    \note The \link textInteractionFlags\endlink set on the label need to include
    either \c TextSelectableByMouse or \c TextSelectableByKeyboard.

    \sa selectedText
*/

/*!
    \property string Label::selectedText
    \brief The selected text

    If there is no selected text this property's value is
    an empty string.

    By default, this property contains an empty string.
    
    \note The \link textInteractionFlags\endlink set on the label need to include
    either \c TextSelectableByMouse or \c TextSelectableByKeyboard.

    \sa hasSelectedText
*/

/*!
    \property enumeration Label::textFormat
    \brief The label's text format

    Possible values are:
    
    \li \c Qt.PlainText - The text string is interpreted as a plain text string.
    \li \c Qt.RichText - The text string is interpreted as a rich text string.
    \li \c Qt.AutoText - The text string is interpreted as for Qt.RichText if 
    Qt.mightBeRichText() returns true, otherwise as Qt.PlainText.

    The default format is Qt.AutoText.

    \sa text
*/

/*!
    \fn void Label::clear()
    
    Clears any label contents.
*/

/*!
    \fn void Label::linkActivated(string link)

    This signal is emitted when the user clicks a link. The URL
    referred to by the anchor is passed in \a link.
*/

};

#endif // LABEL_P_H
