/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef IMAGE_P_H
#define IMAGE_P_H

class Image : public Item
{

    Q_PROPERTY(url source READ source WRITE setSource)
    Q_PROPERTY(size sourceSize READ sourceSize WRITE setSourceSize RESET resetSourceSize)
    Q_PROPERTY(bool asynchronous READ asynchronous WRITE setAsynchronous)
    Q_PROPERTY(bool cache READ cache WRITE setCache)
    Q_PROPERTY(bool mirror READ mirror WRITE setMirror)
    Q_PROPERTY(bool smooth READ smooth WRITE setSmooth)
    Q_PROPERTY(real progress READ progress)
    Q_PROPERTY(enumeration status READ status)
    Q_PROPERTY(enumeration fillMode READ fillMode WRITE setFillMode)
    Q_PROPERTY(int paintedWidth READ paintedWidth)
    Q_PROPERTY(int paintedHeight READ paintedHeight)

/*!
    \class Image
    \brief The Image component displays an image in a declarative user interface

    \ingroup org-hildon-components
    \ingroup org-hildon-components-graphics

    The Image component is used to display images in a declarative user interface.

    The source of the image is specified as a URL using the source property.
    Images can be supplied in any of the standard image formats supported by Qt,
    including bitmap formats such as PNG and JPEG, and vector graphics formats
    such as SVG. If you need to display animated images, use the AnimatedImage
    component.

    If the width and height properties are not specified, 
    the Image component automatically uses the size of the loaded image.
    By default, specifying the width and height of the component causes the image
    to be scaled to that size. This behavior can be changed by setting the
    fillMode property, allowing the image to be stretched and tiled instead.

    \section usage Example Usage

    The following example shows the simplest usage of the Image component.
    
    \code
    import org.hildon.components 1.0
    
    Image {
        id: image
        
        width: 130; height: 130
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo.png

    \section performance Performance

    By default, locally available images are loaded immediately, and the user interface
    is blocked until loading is complete. If a large image is to be loaded, it may be
    preferable to load the image in a low priority thread, by enabling the asynchronous
    property.

    If the image is obtained from a network rather than a local resource, it is
    automatically loaded asynchronously, and the progress and status properties
    are updated as appropriate.

    Images are cached and shared internally, so if several Image components have the same source,
    only one copy of the image will be loaded.

    \note Images are often the greatest user of memory in QML user interfaces.  It is recommended
    that images which do not form part of the user interface have their
    size bounded via the sourceSize property. This is especially important for content
    that is loaded from external sources or provided by the user.
*/

/*!
    \property enumeration Image::fillMode

    Set this property to define what happens when the source image has a different size
    than the item.

    \li \c Image.Stretch - the image is scaled to fit
    \li \c Image.PreserveAspectFit - the image is scaled uniformly to fit without cropping
    \li \c Image.PreserveAspectCrop - the image is scaled uniformly to fill, cropping if necessary
    \li \c Image.Tile - the image is duplicated horizontally and vertically
    \li \c Image.TileVertically - the image is stretched horizontally and tiled vertically
    \li \c Image.TileHorizontally - the image is stretched vertically and tiled horizontally


    \image html declarative-qtlogo-stretch.png
    \c Stretch (default)
    
    \code
    Image {
        width: 130; height: 100
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-preserveaspectfit.png
    \c PreserveAspectFit
    
    \code
    Image {
        width: 130; height: 100
        fillMode: Image.PreserveAspectFit
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-preserveaspectcrop.png
    \c PreserveAspectCrop
    
    \code
    Image {
        width: 130; height: 100
        fillMode: Image.PreserveAspectCrop
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tile.png
    \c Tile
    
    \code
    Image {
        width: 120; height: 120
        fillMode: Image.Tile
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tilevertically.png
    \c TileVertically
    
    \code
    Image {
        width: 120; height: 120
        fillMode: Image.TileVertically
        smooth: true
        source: "qtlogo.png"
    }
    \endcode

    \image html declarative-qtlogo-tilehorizontally.png
    \c TileHorizontally
    
    \code
    Image {
        width: 120; height: 120
        fillMode: Image.TileHorizontally
        smooth: true
        source: "qtlogo.png"
    }
    \endcode
*/

/*!
    \property real Image::paintedWidth
    \property real Image::paintedHeight

    These properties hold the size of the image that is actually painted.
    In most cases it is the same as \c width and \c height, but when using a 
    \c fillMode \c PreserveAspectFit or \c fillMode \c PreserveAspectCrop 
    \c paintedWidth or \c paintedHeight can be smaller or larger than 
    \c width and \c height of the Image component.
*/

/*!
    \property enumeration Image::status

    This property holds the status of image loading. 
    It can be one of:

    \li \c Image.Null - no image has been set
    \li \c Image.Ready - the image has been loaded
    \li \c Image.Loading - the image is currently being loaded
    \li \c Image.Error - an error occurred while loading the image

    Use this status to provide an update or respond to the status change in some way.
    For example, you could:

    \li \c Implement an \c onStatusChanged signal handler:
    
    \code
        Image {
            id: image
            onStatusChanged: if (image.status == Image.Ready) console.log('Loaded')
        }
    \endcode

    \li \c Bind to the status value:
    
    \code
        Text { text: image.status == Image.Ready ? 'Loaded' : 'Not loaded' }
    \endcode

    \sa progress
*/

/*!
    \property real Image::progress

    This property holds the progress of image loading, from 0.0 (nothing loaded)
    to 1.0 (finished).

    \sa status
*/

/*!
    \property bool Image::smooth

    Set this property if you want the image to be smoothly filtered when scaled or
    transformed.  Smooth filtering gives better visual quality, but is slower.  If
    the image is displayed at its natural size, this property has no visual or
    performance effect.

    \note Generally scaling artifacts are only visible if the image is stationary on
    the screen.  A common pattern when animating an image is to disable smooth
    filtering at the beginning of the animation and reenable it at the conclusion.
*/

/*!
    \property size Image::sourceSize

    This property holds the actual width and height of the loaded image.

    Unlike the {Item::}{width} and {Item::}{height} properties, which scale
    the painting of the image, this property sets the actual number of pixels
    stored for the loaded image so that large images do not use more
    memory than necessary. For example, this ensures the image in memory is no
    larger than 1024x1024 pixels, regardless of the Image's width and 
    height values:

    \code
    Rectangle {
        width: ...
        height: ...

        Image {
           anchors.fill: parent
           source: "reallyBigImage.jpg"
           sourceSize.width: 1024
           sourceSize.height: 1024
        }
    }
    \endcode

    If the image's actual size is larger than the sourceSize, the image is scaled down.
    If only one dimension of the size is set to greater than 0, the
    other dimension is set in proportion to preserve the source image's aspect ratio.
    (The fillMode is independent of this.)

    If the source is an instrinsically scalable image (eg. SVG), this property
    determines the size of the loaded image regardless of intrinsic size.
    Avoid changing this property dynamically; rendering an SVG is \e slow compared
    to an image.

    If the source is a non-scalable image (eg. JPEG), the loaded image will
    be no greater than this property specifies. For some formats (currently only JPEG),
    the whole image will never actually be loaded into memory.

    The sourceSize can be cleared to the natural size of the image
    by setting sourceSize to \c undefined.
 
    \note \e Changing this property dynamically causes the image source to be reloaded,
    potentially even from the network, if it is not in the disk cache.
*/

/*!
    \property url Image::source

    Image can handle any image format supported by Qt, loaded from any URL scheme supported by Qt.

    The URL may be absolute, or relative to the URL of the component.
*/

/*!
    \property bool Image::asynchronous

    Specifies that images on the local filesystem should be loaded
    asynchronously in a separate thread.  The default value is
    false, causing the user interface thread to block while the
    image is loaded.  Setting \a asynchronous to true is useful where
    maintaining a responsive user interface is more desirable
    than having images immediately visible.

    Note that this property is only valid for images read from the
    local filesystem.  Images loaded via a network resource (e.g. HTTP)
    are always loaded asynchonously.
*/

/*!
    \property bool Image::cache

    Specifies whether the image should be cached. The default value is
    true. Setting \a cache to false is useful when dealing with large images,
    to make sure that they aren't cached at the expense of small 'ui component' images.
*/

/*!
    \property bool Image::mirror

    This property holds whether the image should be horizontally inverted
    (effectively displaying a mirrored image).

    The default value is false.
*/

};

#endif // IMAGE_P_H
