/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef HEADERVIEW_P_H
#define HEADERVIEW_P_H

class HeaderSection
{

    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(int width READ width WRITE setWidth)
    Q_PROPERTY(enumeration resizeMode READ resizeMode WRITE setResizeMode)
    Q_PROPERTY(int index READ index CONSTANT)
    Q_PROPERTY(bool visible READ isVisible WRITE setVisible)

signals:
    void clicked();

/*!
    \class HeaderSection
    \brief The HeaderSection component represents a section in a HeaderView

    \ingroup org-hildon-components
    \ingroup org-hildon-components-views
    
    \sa HeaderView
*/

/*!
    \property string HeaderSection::text
    \brief The text that appears in the section
*/

/*!
    \property int HeaderSection::width
    \brief Sets the width of the section
*/

/*!
    \property enumeration HeaderSection::resizeMode
    \brief Sets the resizing behaviour of the section
    
    Possible values are:
    
    \li \c Interactive - The user can resize the section.
    \li \c Fixed - The user cannot resize the section.
    \li \c Stretch - HeaderView will automatically resize the section 
    to fill the available space. The size cannot be changed by the 
    user or programmatically.
    \li \c ResizeToContents - HeaderView will automatically resize the section 
    to its optimal size based on the contents of the entire column or row. 
    The size cannot be changed by the user or programmatically.
    
    The default value is \c Interactive.
*/

/*!
    \property int HeaderSection::index
    \brief The index of the section in the HeaderView
*/

/*!
    \property bool HeaderSection::visible
    \brief Whether the section is visible in the HeaderView
*/

/*!
    \fn void HeaderSection::clicked()
    
    This signal is emitted when the section is clicked.
*/

};

class HeaderView : public Item
{

    Q_PROPERTY(list<HeaderSection> sections READ sections)
    Q_PROPERTY(bool clickable READ isClickable WRITE setClickable)
    Q_PROPERTY(bool showSortIndicator READ isSortIndicatorShown WRITE setSortIndicatorShown)
    Q_PROPERTY(bool highlightSections READ highlightSections WRITE setHighlightSections)
    Q_PROPERTY(bool stretchLastSection READ stretchLastSection WRITE setStretchLastSection)
    Q_PROPERTY(bool cascadingSectionResizes READ cascadingSectionResizes WRITE setCascadingSectionResizes)
    Q_PROPERTY(int defaultSectionSize READ defaultSectionSize WRITE setDefaultSectionSize)
    Q_PROPERTY(int minimumSectionSize READ minimumSectionSize WRITE setMinimumSectionSize)
    Q_PROPERTY(flags defaultAlignment READ defaultAlignment WRITE setDefaultAlignment)
    Q_PROPERTY(QModelIndex currentIndex READ currentIndex WRITE setCurrentIndex)

signals:
    void sectionMoved(int logicalIndex, int oldVisualIndex, int newVisualIndex);
    void sectionResized(int logicalIndex, int oldSize, int newSize);
    void sectionPressed(int logicalIndex);
    void sectionClicked(int logicalIndex);
    void sectionEntered(int logicalIndex);
    void sectionDoubleClicked(int logicalIndex);
    void sectionCountChanged(int oldCount, int newCount);
    void sectionHandleDoubleClicked(int logicalIndex);
    void sectionAutoResize(int logicalIndex, enumeration mode);
    void geometriesChanged();
    void sortIndicatorChanged(int logicalIndex, enumeration order);
    
/*!
    \class HeaderView
    \brief The HeaderView component provides a header for an item view
    
    \ingroup org-hildon-components
    \ingroup qml-view-components
    
    \sa HeaderSection, TableView, TreeView
*/

/*!
    \property bool HeaderView::highlightSections
    \brief Whether the sections containing selected items are highlighted

    By default, this property is false.
*/

/*!
    \property list<HeaderSection> HeaderView::sections
    \brief The sections visible in the header
*/

/*!
    \property bool HeaderView::showSortIndicator
    \brief Whether the sort indicator is shown

    By default, this property is false.

    \sa clickable
*/

/*!
    \property bool HeaderView::stretchLastSection
    \brief Whether the last visible section in the header takes up all the
    available space

    The default value is false.

    \note The horizontal headers provided by TreeView are configured with this
    property set to true, ensuring that the view does not waste any of the
    space assigned to it for its header. If this value is set to true, this
    property will override the resize mode set on the last section in the
    header.

    \sa HeaderSection::resizeMode
*/

/*!
    \property bool HeaderView::cascadingSectionResizes
    \brief Whether interactive resizing will be cascaded to the following
    sections once the section being resized by the user has reached its
    minimum size

    This property only affects sections that have \c HeaderSection.Interactive as their
    resize mode.

    The default value is false.

    \sa HeaderSection::resizeMode
*/

/*!
    \property int HeaderView::defaultSectionSize
    \brief The default size of the header sections before resizing.

    This property only affects sections that have \c HeaderSection.Interactive or 
    \c HeaderSection.Fixed as their resize mode.

    \sa HeaderSection::resizeMode, minimumSectionSize
*/

/*!
    \property int HeaderView::minimumSectionSize
    \brief The minimum size of the header sections.

    The minimum section size is the smallest section size allowed.

    \sa HeaderSection::resizeMode. defaultSectionSize
*/

/*!
    \property flags HeaderView::defaultAlignment
    \brief The default alignment of the text in each header section
*/

/*!
    \fn void HeaderView::sectionMoved(int logicalIndex, int oldVisualIndex,
    int newVisualIndex)

    This signal is emitted when a section is moved. The section's logical index
    is specified by \a logicalIndex, the old index by \a oldVisualIndex, and
    the new index position by \a newVisualIndex.
*/

/*!
    \fn void HeaderView::sectionResized(int logicalIndex, int oldSize,
    int newSize)

    This signal is emitted when a section is resized. The section's logical
    number is specified by \a logicalIndex, the old size by \a oldSize, and the
    new size by \a newSize.
*/

/*!
    \fn void HeaderView::sectionPressed(int logicalIndex)

    This signal is emitted when a section is pressed. The section's logical
    index is specified by \a logicalIndex.

    \sa clickable
*/

/*!
    \fn void HeaderView::sectionClicked(int logicalIndex)

    This signal is emitted when a section is clicked. The section's logical
    index is specified by \a logicalIndex.

    Note that the sectionPressed signal will also be emitted.

    \sa clickable, sectionPressed()
*/

/*!
    \fn void HeaderView::sectionEntered(int logicalIndex)

    This signal is emitted when the cursor moves over the section and the left
    mouse button is pressed. The section's logical index is specified by
    \a logicalIndex.

    \sa clickable, sectionPressed()
*/

/*!
    \fn void HeaderView::sectionDoubleClicked(int logicalIndex)

    This signal is emitted when a section is double-clicked. The section's
    logical index is specified by \a logicalIndex.

    \sa clickable
*/

/*!
    \fn void HeaderView::sectionCountChanged(int oldCount, int newCount)

    This signal is emitted when the number of sections changes, i.e., when
    sections are added or deleted. The original count is specified by
    \a oldCount, and the new count by \a newCount.
*/

/*!
    \fn void HeaderView::sectionHandleDoubleClicked(int logicalIndex)

    This signal is emitted when a section is double-clicked. The section's
    logical index is specified by \a logicalIndex.

    \sa clickable
*/

/*!
    \fn void HeaderView::sortIndicatorChanged(int logicalIndex,
    enumeration order)

    This signal is emitted when the section containing the sort indicator or
    the order indicated is changed. The section's logical index is specified
    by \a logicalIndex and the sort order is specified by \a order.
*/

/*!
    \fn void HeaderView::sectionAutoResize(int logicalIndex,
    enumeration mode)

    This signal is emitted when a section is automatically resized. The
    section's logical index is specified by \a logicalIndex, and the resize
    mode by \a mode.

    \sa HeaderSection::resizeMode, stretchLastSection
*/

/*!
    \fn void HeaderView::geometriesChanged()

    This signal is emitted when the header's geometries have changed.
*/

};

#endif // HEADERVIEW_P_H
