/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef FILESYSTEMMODEL_P_H
#define FILESYSTEMMODEL_P_H

class FileSystemModel
{

    Q_PROPERTY(string rootPath READ rootPath WRITE setRootPath)
    Q_PROPERTY(string filterPath READ filterPath WRITE setFilterPath)
    Q_PROPERTY(regExp filterRegExp READ filterRegExp WRITE setFilterRegExp)
    Q_PROPERTY(enumeration filterCaseSensitivity READ filterCaseSensitivity WRITE setFilterCaseSensitivity)
    Q_PROPERTY(enumeration sortCaseSensitivity READ sortCaseSensitivity WRITE setSortCaseSensitivity)
    Q_PROPERTY(bool isSortLocaleAware READ isSortLocaleAware WRITE setSortLocaleAware)
    Q_PROPERTY(bool showDirs READ showDirs WRITE setShowDirs)
    Q_PROPERTY(bool showFiles READ showFiles WRITE setShowFiles)
    Q_PROPERTY(bool showHidden READ showHidden WRITE setShowHidden)
    Q_PROPERTY(bool showDotAndDotDot READ showDotAndDotDot WRITE setShowDotAndDotDot)
    Q_PROPERTY(bool showSymlinks READ showSymlinks WRITE setShowSymlinks)
    Q_PROPERTY(bool resolveSymlinks READ resolveSymlinks WRITE setResolveSymlinks)
    Q_PROPERTY(bool readOnly READ isReadOnly WRITE setReadOnly)
    Q_PROPERTY(bool nameFilterDisables READ nameFilterDisables WRITE setNameFilterDisables)
    Q_PROPERTY(stringlist nameFilters READ nameFilters WRITE setNameFilters)
    Q_PROPERTY(stringlist loadedDirectories READ loadedDirectories)

public:
    bool directoryIsLoaded(string directory);
    
    int count(QModelIndex parent = QModelIndex());

    QModelIndex modelIndex(string path, int column = 0);

    string filePath(QModelIndex index);
    bool isDir(QModelIndex index);
    int size(QModelIndex index);
    string type(QModelIndex index);
    date lastModified(QModelIndex index);

    QModelIndex mkdir(QModelIndex parent, string name);
    bool rmdir(QModelIndex index);
    string fileName(QModelIndex index);
    string fileIcon(QModelIndex index);
    bool remove(QModelIndex index);
    
    void sort(int column, enumeration order = Qt.AscendingOrder);

signals:
    void directoryLoaded(string directory);

/*!
    \class FileSystemModel
    \brief The FileSystemModel component provides a data model enabling 
    read/write access to the file system
    
    \ingroup org-hildon-components
    \ingroup org-hildon-components-data
    
    FileSystemModel can be used with one of the item view components 
    to provide a file browser:
    
    \code
    ListView {
        id: view
        
        windowTitle: "File browser"
        model: FileSystemModel {
            id: fileModel
        }
        rootIndex: fileModel.modelIndex("/home/user/MyDocs/")
        onClicked: fileModel.isDir(currentIndex) ? rootIndex = currentIndex
                                                 : Qt.openUrlExternally(fileModel.filePath(currentIndex))
    }
    \endcode
*/

/*!
    \property string FileSystemModel::rootPath
    
    Specifies the root path to be watched by the model. 
    Changes to files below the root path will be detected 
    by FileSystemModel, and the model will be updated accordingly.
*/

/*!
    \property string FileSystemModel::filterPath
    
    Specifies the path to which the regExp filter will be applied.
    When setting the regExp filter, only files under this path will 
    be filtered.
*/

/*!
    \property regExp FileSystemModel::filterRegExp
    \brief The regular expression used to filter the contents of the source model

    Setting this property overwrites the current filterCaseSensitivity.
    By default, the regExp is an empty string matching all contents.

    If no regExp or an empty string is set, everything in the source model
    will be accepted.

    \sa filterCaseSensitivity
*/

/*!
    \property enumeration FileSystemModel::filterCaseSensitivity

    \brief The case sensitivity of the regExp pattern used to filter the
    contents of the source model

    By default, the filter is case sensitive.

    \sa filterRegExp, sortCaseSensitivity
*/

/*!
    \property enumeration FileSystemModel::sortCaseSensitivity
    \brief The case sensitivity setting used for comparing strings when sorting

    By default, sorting is case sensitive.

    \sa filterCaseSensitivity
*/

/*!
    \property bool FileSystemModel::isSortLocaleAware
    \brief The local aware setting used for comparing strings when sorting

    By default, sorting is not local aware.

    \sa sortCaseSensitivity
*/

/*!
    \property bool FileSystemModel::dynamicSortFilter
    \brief Whether the proxy model is dynamically sorted and filtered
    whenever the contents of the source model change

    Note that you should not update the source model through the proxy
    model when dynamicSortFilter is true.

    The default value is false.
*/

/*!
    \property bool FileSystemModel::showDirs
    
    Specifies whether directories will be listed in the model.
    
    The default value is true.
*/

/*!
    \property bool FileSystemModel::showFiles
    
    Specifies whether files will be listed in the model.
    
    The default value is true.
*/

/*!
    \property bool FileSystemModel::showHidden
    
    Specifies whether hidden files/directories will be listed in the model.
    
    The default value is false.
*/

/*!
    \property bool FileSystemModel::showDotAndDotDot
    
    Specifies whether '.' and '..' will be listed in the model.
    
    The default value is true.
*/

/*!
    \property bool FileSystemModel::showSymlinks
    
    Specifies whether symlinks will be listed in the model.
    
    The default value is true.
*/

/*!
    \property bool FileSystemModel::showDirs
    
    Specifies whether symlinks will be resolved by the model.
    
    The default value is true.
*/

/*!
    \property bool FileSystemModel::readOnly
    
    Specifies whether the model should be read-only.
    
    The default value is false.
*/

/*!
    \property bool FileSystemModel::nameFilterDisables
    
    Specifies whether items that do not match the name filter will 
    be disabled, rather than hidden.
    
    The default value is false.
*/

/*!
    \property stringlist FileSystemModel::nameFilters
    
    Specifies the name filters to be used. Items not matching 
    the filters will either be hidden or disabled, according to 
    the nameFilterDisables property.
*/

/*!
    \property stringlist FileSystemModel::loadedDirectories
    
    The list of directories that have been loaded by the model.
*/

/*!
    \fn bool FileSystemModel::directoryIsLoaded(string directory)
    
    Returns true if \a directory has been loaded by 
    the model.
*/

/*!
    \fn QModelIndex FileSystemModel::modelIndex(string path, int column = 0)
    
    This function returns the QModelIndex that represents \a path and \a column.
*/

/*!
    \fn int FileSystemModel::count(QModelIndex parent)
    
    Returns the number of items with \a parent index.
*/

/*!
    \fn string FileSystemModel::filePath(QModelIndex index)
    
    Returns that filePath represented by \a index.
*/

/*!
    \fn bool FileSystemModel::isDir(QModelIndex index)
    
    Returns whether the item at \a index is a directory.
*/

/*!
    \fn int FileSystemModel::size(QModelIndex index)
    
    Returns the size (in bytes) of the item at \a index.
*/

/*!
    \fn string FileSystemModel::type(QModelIndex index)
    
    Returns the mime type of the item at \a index.
*/

/*!
    \fn date FileSystemModel::lastModified(QModelIndex index)
    
    Returns the last modified date of the item at \a index.
*/

/*!
    \fn QModelIndex FileSystemModel::mkdir(QModelIndex parent, string name)
    
    Creates a new directory with name \a name, under the path represented 
    by \a index. If the directory is successfully created, a QModelIndex 
    representing the new directory will be returned, otherwise the QModelIndex will be invalid.
*/

/*!
    \fn bool FileSystemModel::rmdir(QModelIndex index)
    
    This function removes the directory represented by \a index, 
    and returns true if successful.
*/

/*!
    \fn string FileSystemModel::fileName(QModelIndex index)
    
    This function returns the fileName of the item at \a index.
*/

/*!
    \fn string FileSystemModel::fileIcon(QModelIndex index)
    
    This function returns the icon source of the item at \a index.
*/

/*!
    \fn bool FileSystemModel::remove(QModelIndex index)
    
    This function removes the item at \a index, 
    and returns true if successful.
*/

/*!
    \fn void FileSystemModel::sort(int column, enumeration order)
    
    Sorts the data in \a column according to \a order.
    
    \a order can be one of the following:
    
    \li \c Qt.AscendingOrder
    \li \c Qt.DescendingOrder
*/

};

#endif // FILESYSTEMMODEL_P_H
