/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef DOUBLESPINBOX_P_H
#define DOUBLESPINBOX_P_H

class DoubleSpinBox : public Item
{
    
    Q_PROPERTY(bool wrapping READ wrapping WRITE setWrapping)
    Q_PROPERTY(bool frame READ hasFrame WRITE setFrame)
    Q_PROPERTY(flags alignment READ alignment WRITE setAlignment)
    Q_PROPERTY(bool readOnly READ isReadOnly WRITE setReadOnly)
    Q_PROPERTY(enumeration buttonSymbols READ buttonSymbols WRITE setButtonSymbols)
    Q_PROPERTY(string specialValueText READ specialValueText WRITE setSpecialValueText)
    Q_PROPERTY(string text READ text)
    Q_PROPERTY(bool accelerated READ isAccelerated WRITE setAccelerated)
    Q_PROPERTY(enumeration correctionMode READ correctionMode WRITE setCorrectionMode)
    Q_PROPERTY(bool acceptableInput READ hasAcceptableInput)
    Q_PROPERTY(bool keyboardTracking READ keyboardTracking WRITE setKeyboardTracking)
    Q_PROPERTY(string prefix READ prefix WRITE setPrefix)
    Q_PROPERTY(string suffix READ suffix WRITE setSuffix)
    Q_PROPERTY(string cleanText READ cleanText)
    Q_PROPERTY(int decimals READ decimals WRITE setDecimals)
    Q_PROPERTY(double minimum READ minimum WRITE setMinimum)
    Q_PROPERTY(double maximum READ maximum WRITE setMaximum)
    Q_PROPERTY(double singleStep READ singleStep WRITE setSingleStep)
    Q_PROPERTY(double value READ value WRITE setValue)

public:
    void stepUp();
    void stepDown();
    void selectAll();
    void clear();
    
signals:
    void editingFinished();
    void valueChanged(double d);

/*!
    \class DoubleSpinBox
    \brief The DoubleSpinBox component provides a spin box widget that
    takes doubles.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-text

    DoubleSpinBox allows the user to choose a value by clicking the
    up and down buttons or by pressing Up or Down on the keyboard to
    increase or decrease the value currently displayed. The user can
    also type the value in manually.

    Every time the value changes DoubleSpinBox emits the
    valueChanged() signal. The current value can be fetched with
    and set with value.

    Note: DoubleSpinBox will round numbers so they can be displayed
    with the current precision. In a DoubleSpinBox with decimals set
    to 2, calling setValue(2.555) will cause value to return 2.56.

    Clicking the up and down buttons or using the keyboard accelerator's
    Up and Down arrows will increase or decrease the current value in
    steps of size singleStep. The minimum and
    maximum value and the step size can be set using one of the
    constructors, and can be changed later with minimum,
    maximum and singleStep. The spinbox has a default
    precision of 2 decimal places but this can be changed using
    decimals.

    Most spin boxes are directional, but DoubleSpinBox can also
    operate as a circular spin box, i.e. if the range is 0.0-99.9 and
    the current value is 99.9, clicking "up" will give 0 if wrapping
    is set to true. Use wrapping if you want circular behavior.

    The displayed value can be prepended and appended with arbitrary
    strings indicating, for example, currency or the unit of
    measurement. See prefix and suffix. The text in the spin
    box is retrieved with text (which includes any prefix and
    suffix), or with cleanText (which has no prefix, no suffix
    and no leading or trailing whitespace).

    It is often desirable to give the user a special (often default)
    choice in addition to the range of numeric values. See
    specialValueText for how to do this with DoubleSpinBox.

    \sa SpinBox, Slider
*/

/*!
    \property enumeration DoubleSpinBox::buttonSymbols

    \brief The current button symbol mode

    The possible values can be either \c UpDownArrows or \c PlusMinus.
    The default is \c UpDownArrows.

    Note that some styles might render PlusMinus and UpDownArrows
    identically.
*/

/*!
    \property string DoubleSpinBox::text

    \brief The spin box's text, including any prefix and suffix

    There is no default text.
*/

/*!
    \property string DoubleSpinBox::specialValueText
    \brief The special-value text

    If set, the spin box will display this text instead of a numeric
    value whenever the current value is equal to minimum(). Typical use
    is to indicate that this choice has a special (default) meaning.

    To turn off the special-value text display, call this function
    with an empty string. The default is no special-value text, i.e.
    the numeric value is shown as usual.
*/

/*!
    \property bool DoubleSpinBox::wrapping

    \brief Whether the spin box is circular.

    If wrapping is true stepping up from maximum value will take you
    to the minimum value and vica versa. Wrapping only make sense if
    you have minimum and maximum values set.

    \sa minimum, maximum
*/

/*!
    \property bool DoubleSpinBox::readOnly
    \brief Whether the spin box is read only.

    In read-only mode, the user can still copy the text to the
    clipboard, or drag and drop the text;
    but cannot edit it.

    The text field in the spin box does not show a cursor in
    read-only mode.
*/

/*!
    \property bool DoubleSpinBox::keyboardTracking
    \brief Whether keyboard tracking is enabled for the spinbox.

    If keyboard tracking is enabled (the default), the spinbox
    emits the valueChanged() signal while the new value is being
    entered from the keyboard.

    E.g. when the user enters the value 600 by typing 6, 0, and 0,
    the spinbox emits 3 signals with the values 6, 60, and 600
    respectively.

    If keyboard tracking is disabled, the spinbox doesn't emit the
    valueChanged() signal while typing. It emits the signal later,
    when the return key is pressed, when keyboard focus is lost, or
    when other spinbox functionality is used, e.g. pressing an arrow
    key.
*/

/*!
    \property bool DoubleSpinBox::frame
    \brief Whether the spin box draws itself with a frame

    If enabled (the default) the spin box draws itself inside a frame,
    otherwise the spin box draws itself without any frame.
*/

/*!
    \property bool DoubleSpinBox::accelerated
    \brief Whether the spin box will accelerate the frequency of the steps when
    pressing the step Up/Down buttons.

    If enabled the spin box will increase/decrease the value faster
    the longer you hold the button down.
*/

/*!
    \property enumeration DoubleSpinBox::correctionMode
    \brief The mode to correct a value if editing finishes
           
    Possible values are:

    \li \c CorrectToPreviousValue The spinbox will revert to the last
                                  valid value.

    \li \c CorrectToNearestValue The spinbox will revert to the nearest
                                 valid value.

    The default mode is CorrectToPreviousValue
    
    \sa acceptableInput
*/

/*!
    \property bool DoubleSpinBox::acceptableInput
    \brief Whether the input satisfies the current validation

    \sa correctionMode
*/

/*!
    \property flags DoubleSpinBox::alignment
    \brief The alignment of the spin box

    Available flags are: 
    
    \li \c Qt.AlignLeft
    \li \c Qt.AlignRight
    \li \c Qt.AlignHCenter

    By default, the alignment is Qt.AlignLeft

    Attempting to set the alignment to an illegal flag combination
    does nothing.
*/

/*!
    \property double DoubleSpinBox::value
    \brief the value of the spin box

    Will emit valueChanged() if the new value is different
    from the old one.

    Note: The value will be rounded so it can be displayed with the
    current setting of decimals.

    \sa decimals
*/

/*!
    \property string DoubleSpinBox::prefix
    \brief The spin box's prefix

    The prefix is prepended to the start of the displayed value.
    Typical use is to display a unit of measurement or a currency
    symbol.

    To turn off the prefix display, set this property to an empty
    string. The default is no prefix. The prefix is not displayed when
    value == minimum and specialValueText is set.

    \sa suffix, specialValueText
*/

/*!
    \property string DoubleSpinBox::suffix
    \brief The suffix of the spin box

    The suffix is appended to the end of the displayed value. Typical
    use is to display a unit of measurement or a currency symbol.

    To turn off the suffix display, set this property to an empty
    string. The default is no suffix. The suffix is not displayed for
    the minimum if specialValueText is set.

    \sa prefix, specialValueText
*/

/*!
    \property string DoubleSpinBox::cleanText

    \brief The text of the spin box excluding any prefix, suffix,
    or leading or trailing whitespace.

    \sa text, prefix, suffix
*/

/*!
    \property double DoubleSpinBox::singleStep
    \brief The step value

    When the user uses the arrows to change the spin box's value the
    value will be incremented/decremented by the amount of the
    singleStep. The default value is 1.0. Setting a singleStep value
    of less than 0 does nothing.
*/

/*!
    \property double DoubleSpinBox::minimum

    \brief The minimum value of the spin box

    When setting this property the maximum is adjusted
    if necessary to ensure that the range remains valid.

    The default minimum value is 0.0.

    Note: The minimum value will be rounded to match the decimals
    property.

    \sa decimals, specialValueText
*/

/*!
    \property double DoubleSpinBox::maximum

    \brief The maximum value of the spin box

    When setting this property the minimum is adjusted
    if necessary, to ensure that the range remains valid.

    The default maximum value is 99.99.

    Note: The maximum value will be rounded to match the decimals
    property.

    \sa decimals
*/

/*!
     \property int DoubleSpinBox::decimals

     \brief The precision of the spin box, in decimals

     Sets how many decimals the spinbox will use for displaying and
     interpreting doubles.

     \warning The maximum value for \a decimals is DBL_MAX_10_EXP +
     DBL_DIG (ie. 323) because of the limitations of the double type.

     Note: The maximum, minimum and value might change as a result of
     changing this property.
*/

/*!
    \fn void DoubleSpinBox::stepUp()
    
    Steps up by one linestep
*/

/*!
    \fn void DoubleSpinBox::stepDown()
    
    Steps down by one linestep
*/

/*!
    \fn void DoubleSpinBox::selectAll()
    
    Selects all the text in the spinbox except the prefix and suffix.
*/

/*!
    \fn void DoubleSpinBox::clear()
    
    Clears the spinbox of all text but prefix and suffix.
*/

/*!
    \fn void DoubleSpinBox::editingFinished()

    This signal is emitted editing is finished. This happens when the
    spinbox loses focus and when enter is pressed.
*/

/*!
    \fn void DoubleSpinBox::valueChanged(double d);

    This signal is emitted whenever the spin box's value is changed.
    The new value is passed in \a d.
*/

};

#endif // DOUBLESPINBOX_P_H
