/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef DIALOG_P_H
#define DIALOG_P_H

class Dialog : public Item
{

    Q_PROPERTY(bool modal READ isModal WRITE setModal)
    Q_PROPERTY(bool showProgressIndicator READ showingProgressIndicator WRITE showProgressIndicator)
    Q_PROPERTY(Item content READ content WRITE setContent)
    Q_PROPERTY(list<Button> buttons READ buttons WRITE setButtons)
    
public:
    void open();
    int exec();
    void done(int r);
    void accept();
    void reject();

signals:
    void finished(int result);
    void accepted();
    void rejected();

/*!
    \class Dialog
    \brief The Dialog component is the base component of dialogs

    \ingroup org-hildon-components
    \ingroup org-hildon-components-dialogs

    A dialog window is a top-level window mostly used for short-term
    tasks and brief communications with the user.
*/

/*!
    \property bool Dialog::modal
    \brief Whether show() should pop up the dialog as modal or modeless

    By default, this property is false and show() pops up the dialog
    as modeless.

    exec() ignores the value of this property and always pops up the
    dialog as modal.

    \sa show(), exec()
*/

/*!
    \property Item Dialog::content
    \brief The Item to be displayed in the content area
    
    \sa Item
*/

/*!
    \property list<Button> Dialog::buttons
    \brief The list of Button items to appear in the button box
    
    \sa Button
*/

/*!
    \fn void Dialog::open()

    Shows the dialog as a window modal dialog,
    returning immediately.

    \sa exec(), show(), modal
*/

/*!
    \fn int Dialog::exec()
    
    Shows the dialog as a modal dialog,
    blocking until the user closes it. The function returns a 
    DialogCode result.

    If the dialog is application modal, users cannot
    interact with any other window in the same application until they close
    the dialog. If the dialog is window modal, only
    interaction with the parent window is blocked while the dialog is open.
    By default, the dialog is application modal.

    \sa open(), show(), modal
*/

/*!
    \fn void Dialog::done(int r)
    
    Closes the dialog and sets its result code to \a r. If this dialog
    is shown with exec(), done() causes the local event loop to finish,
    and exec() to return \a r.

    \sa accept(), reject()
*/

/*!
    \fn void Dialog::accept()
    
    Hides the modal dialog and sets the result code to \c Accepted.

    \sa reject(), done()
*/

/*!
    \fn void Dialog::reject()
    
    Hides the modal dialog and sets the result code to \c Rejected.

    \sa accept(), done()
*/

/*! \fn void Dialog::finished(int result)

    This signal is emitted when the dialog's \a result code has been
    set, either by the user or by calling done(), accept(), or
    reject().

    Note that this signal is \e not emitted when hiding the dialog
    with hide() or setVisible(false). This includes deleting the
    dialog while it is visible.

    \sa accepted(), rejected()
*/

/*! \fn void Dialog::accepted()

    This signal is emitted when the dialog has been accepted either by
    the user or by calling accept() or done() with the
    Dialog.Accepted argument.

    Note that this signal is \e not emitted when hiding the dialog
    with hide() or setVisible(false). This includes deleting the
    dialog while it is visible.

    \sa finished(), rejected()
*/

/*! \fn void Dialog::rejected()

    This signal is emitted when the dialog has been rejected either by
    the user or by calling reject() or done() with the
    Dialog.Rejected argument.

    Note that this signal is \e not emitted when hiding the dialog
    with hide() or setVisible(false). This includes deleting the
    dialog while it is visible.

    \sa finished(), accepted()
*/

};

#endif // DIALOG_P_H
