/*
 * Copyright (C) 2014 Stuart Howarth <showarth@marxoft.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 3, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef BUTTON_P_H
#define BUTTON_P_H

class Button : public Item
{

    Q_PROPERTY(string text READ text WRITE setText)
    Q_PROPERTY(string icon READ icon WRITE setIcon)
    Q_PROPERTY(size iconSize READ iconSize WRITE setIconSize)
    Q_PROPERTY(string shortcut READ shortcut WRITE setShortcut)
    Q_PROPERTY(bool checkable READ isCheckable WRITE setCheckable)
    Q_PROPERTY(bool checked READ isChecked WRITE setChecked DESIGNABLE isCheckable)
    Q_PROPERTY(bool autoRepeat READ autoRepeat WRITE setAutoRepeat)
    Q_PROPERTY(bool autoExclusive READ autoExclusive WRITE setAutoExclusive)
    Q_PROPERTY(int autoRepeatDelay READ autoRepeatDelay WRITE setAutoRepeatDelay)
    Q_PROPERTY(int autoRepeatInterval READ autoRepeatInterval WRITE setAutoRepeatInterval)
    Q_PROPERTY(bool down READ isDown WRITE setDown)
    Q_PROPERTY(bool autoDefault READ autoDefault WRITE setAutoDefault)
    Q_PROPERTY(bool default READ isDefault WRITE setDefault)
    Q_PROPERTY(bool flat READ isFlat WRITE setFlat)

public:
    void animateClick(int msec = 100);
    void click();
    void toggle();

signals:
    void pressed();
    void released();
    void clicked(bool checked = false);
    void toggled(bool checked);

/*!
    \class Button
    \brief The Button component provides a command button.

    \ingroup org-hildon-components
    \ingroup org-hildon-components-controls


    The push button, or command button, is perhaps the most commonly
    used item in any graphical user interface. Push (click) a button
    to command the computer to perform some action, or to answer a
    question. Typical buttons are OK, Apply, Cancel, Close, Yes, No
    and Help.

    A command button is rectangular and typically displays a text
    label describing its action. A shortcut key can be specified by
    preceding the preferred character with an ampersand in the
    text. 

    Push buttons display a textual label, and optionally a small
    icon. These can be set using the constructors and changed later
    using setText() and setIcon().  If the button is disabled the
    appearance of the text and icon will be manipulated with respect
    to the GUI style to make the button look "disabled".

    A push button emits the signal clicked() when it is activated by
    the mouse, the Spacebar or by a keyboard shortcut. Connect to
    this signal to perform the button's action. Push buttons also
    provide less commonly used signals, for example, pressed() and
    released().

    Command buttons in dialogs are by default auto-default buttons,
    i.e. they become the default push button automatically when they
    receive the keyboard input focus. A default button is a push
    button that is activated when the user presses the Enter or Return
    key in a dialog. You can change this with setAutoDefault(). Note
    that auto-default buttons reserve a little extra space which is
    necessary to draw a default-button indicator. If you do not want
    this space around your buttons, call setAutoDefault(false).

    Being so central, the button item has grown to accommodate a
    great many variations in the past decade. The Microsoft style
    guide now shows about ten different states of Windows push buttons
    and the text implies that there are dozens more when all the
    combinations of features are taken into consideration.

    The most important modes or states are:

    \li Available or not (grayed out, disabled).
    \li Standard push button, toggling push button or menu button.
    \li On or off (only for toggling push buttons).
    \li Default or normal. The default button in a dialog can generally
       be "clicked" using the Enter or Return key.
    \li Auto-repeat or not.
    \li Pressed down or not.

    If you need toggle behavior (see setCheckable()) or a button
    that auto-repeats the activation signal when being pushed down
    like the arrows in a scroll bar (see setAutoRepeat()), a command
    button is probably not what you want. When in doubt, use a tool
    button.

    A variation of a command button is a menu button. These provide
    not just one command, but several, since when they are clicked
    they pop up a menu of options. Use the method setMenu() to
    associate a popup menu with a push button.

    Other classes of buttons are option buttons (see RadioButton) and
    check boxes (see CheckBox).

    \sa ToolButton, RadioButton, CheckBox
*/

/*!
    \property bool Button::autoDefault
    \brief Whether the push button is an auto default button

    If this property is set to true then the push button is an auto
    default button.

    In some GUI styles a default button is drawn with an extra frame
    around it, up to 3 pixels or more. Qt automatically keeps this
    space free around auto-default buttons, i.e. auto-default buttons
    may have a slightly larger size hint.

    This property's default is true for buttons that have a Dialog
    parent; otherwise it defaults to false.

    See the default property for details of how default and
    auto-default interact.
*/

/*!
    \property bool Button::default
    \brief Whether the push button is the default button

    Default and autodefault buttons decide what happens when the user
    presses enter in a dialog.

    A button with this property set to true (i.e., the dialog's
    \e default button,) will automatically be pressed when the user presses enter,
    with one exception: if an \a autoDefault button currently has focus, the autoDefault
    button is pressed. When the dialog has autoDefault buttons but no default button,
    pressing enter will press either the autoDefault button that currently has focus, or if no
    button has focus, the next autoDefault button in the focus chain.

    In a dialog, only one push button at a time can be the default
    button. This button is then displayed with an additional frame
    (depending on the GUI style).

    The default button behavior is provided only in dialogs. Buttons
    can always be clicked from the keyboard by pressing Spacebar when
    the button has focus.

    If the default property is set to false on the current default button
    while the dialog is visible, a new default will automatically be
    assigned the next time a pushbutton in the dialog receives focus.

    This property's default is false.
*/

/*!
    \property bool Button::flat
    \brief Whether the button border is raised

    This property's default is false. If this property is set, most
    styles will not paint the button background unless the button is
    being pressed.
*/

/*!
    \property string Button::text
    \brief The text shown on the button

    If the button has no text, the text() function will return a an empty
    string.

    If the text contains an ampersand character ('&'), a shortcut is
    automatically created for it. The character that follows the '&' will
    be used as the shortcut key. Any previous shortcut will be
    overwritten, or cleared if no shortcut is defined by the text.

    There is no default text.
*/

/*!
    \property string Button::icon
    \brief The icon source

    The icon's default size is defined by the GUI style, but can be
    adjusted by setting the iconSize property.
*/

/*!
    \property string Button::shortcut
    \brief The mnemonic associated with the button
*/

/*!
    \property bool Button::checkable
    \brief Whether the button is checkable

    By default, the button is not checkable.

    \sa checked
*/

/*!
    \property bool Button::checked
    \brief Whether the button is checked

    Only checkable buttons can be checked. By default, the button is unchecked.

    \sa checkable
*/

/*!
    \property bool Button::down
    \brief Whether the button is pressed down

    If this property is true, the button is pressed down. The signals
    pressed() and clicked() are not emitted if you set this property
    to true. The default is false.
*/

/*!
    \property bool Button::autoRepeat
    \brief Whether autoRepeat is enabled

    If autoRepeat is enabled, then the pressed(), released(), and clicked() signals are emitted at
    regular intervals when the button is down. autoRepeat is off by default.
    The initial delay and the repetition interval are defined in milliseconds by
    autoRepeatDelay and autoRepeatInterval.

    Note: If a button is pressed down by a shortcut key, then auto-repeat is enabled and timed by the
    system and not by this class. The pressed(), released(), and clicked() signals will be emitted
    like in the normal case.
*/

/*!
    \property int Button::autoRepeatDelay
    \brief the initial delay of auto-repetition

    If autoRepeat is enabled, then autoRepeatDelay defines the initial
    delay in milliseconds before auto-repetition kicks in.

    \sa autoRepeat, autoRepeatInterval
*/

/*!
    \property int Button::autoRepeatInterval
    \brief the interval of auto-repetition

    If autoRepeat is enabled, then autoRepeatInterval defines the
    length of the auto-repetition interval in millisecons.

    \sa autoRepeat, autoRepeatDelay
*/

/*!
    \property bool Button::autoExclusive
    \brief whether auto-exclusivity is enabled

    If auto-exclusivity is enabled, checkable buttons that belong to the
    same parent item behave as if they were part of the same
    exclusive button group. In an exclusive button group, only one button
    can be checked at any time; checking another button automatically
    unchecks the previously checked one.

    The property has no effect on buttons that belong to a button
    group.

    autoExclusive is off by default, except for radio buttons.

    \sa RadioButton
*/

/*!
    \property size Button::iconSize
    \brief the icon size used for this button.

    The default size is defined by the GUI style. This is a maximum
    size for the icons. Smaller icons will not be scaled up.
*/

/*! 
    \fn void Button::animateClick(int msec)
    
    Performs an animated click: the button is pressed immediately, and
    released \a msec milliseconds later (the default is 100 ms).

    Calling this function again before the button was released will reset
    the release timer.

    All signals associated with a click are emitted as appropriate.

    This function does nothing if \link enabled\endlink 
    is false.

    \sa click()
*/

/*! 
    \fn void Button::click()
    
    Performs a click.

    All the usual signals associated with a click are emitted as
    appropriate. If the button is checkable, the state of the button is
    toggled.

    This function does nothing if \link enabled\endlink 
    is false.

    \sa animateClick()
*/

/*! 
    \fn void Button::toggle()

    Toggles the state of a checkable button.

    \sa checked
*/

/*!
    \fn void Button::pressed()

    This signal is emitted when the button is pressed down.

    \sa released(), clicked()
*/

/*!
    \fn void Button::released()

    This signal is emitted when the button is released.

    \sa pressed(), clicked(), toggled()
*/

/*!
    \fn void Button::clicked(bool checked)

    This signal is emitted when the button is activated (i.e. pressed down
    then released while the mouse cursor is inside the button), when the
    shortcut key is typed, or when click() or animateClick() is called.
    Notably, this signal is \e not emitted if you call setDown(),
    setChecked() or toggle().

    If the button is checkable, \a checked is true if the button is
    checked, or false if the button is unchecked.

    \sa pressed(), released(), toggled()
*/

/*!
    \fn void Button::toggled(bool checked)

    This signal is emitted whenever a checkable button changes its state.
    \a checked is true if the button is checked, or false if the button is
    unchecked.

    This may be the result of a user action, click() slot activation,
    or because setChecked() was called.

    The states of buttons in exclusive button groups are updated before this
    signal is emitted. This means that slots can act on either the "off"
    signal or the "on" signal emitted by the buttons in the group whose
    states have changed.

    For example, a slot that reacts to signals emitted by newly checked
    buttons but which ignores signals from buttons that have been unchecked
    can be implemented using the following pattern:

    \sa checked, clicked()
*/
};

#endif // BUTTON_P_H
