#include "playlist.h"
#include "link.h"
#include "user.h"
#include <QDebug>


/* Playlist callbacks */

static void tracksAddedCallback( sp_playlist* pl, sp_track* const* tracks, int num_tracks,
                                          int position, void* userdata )
{
    Q_UNUSED(userdata)
    qDebug()<< "Tracks added to playlist..";
   // QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"tracksAdded",Qt::QueuedConnection,
     //                        Q_ARG(sp_track**,const_cast<sp_track**>(tracks)),Q_ARG(qint32,num_tracks),Q_ARG(qint32,position));
}

static void tracksRemovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, void* userdata )
{
    Q_UNUSED(userdata)
    qDebug() << "Tracks removed from playlist..";
   // QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"tracksRemoved",Qt::QueuedConnection,
     //                         Q_ARG(qint32*,const_cast<qint32*>(tracks)),Q_ARG(qint32,num_tracks));
}

static void tracksMovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                            void* userdata)
{
}

static void playlistRenamedCallback( sp_playlist* pl, void* userdata )
{
    Q_UNUSED(userdata)
    qDebug()<<"Playlist renamed..";
    QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"playlistRenamed",Qt::QueuedConnection);
}

static void playlistStateChangedCallback( sp_playlist* pl, void* userdata )
{
    Q_UNUSED(userdata)
    qDebug() << "Playlist state changed..";
    QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"playlistStateChanged",Qt::QueuedConnection);
}

static void playlistUpdateInProgressCallback( sp_playlist* pl, bool done, void* userdata )
{
    Q_UNUSED(userdata)
    qint32 state;
    qDebug() << "Playlist update in progress..";
    if(done)
        state=1;
    else
        state=0;
    QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"playlistUpdateInProgress",Qt::QueuedConnection,Q_ARG(qint32,state));
}

static void playlistMetadataUpdatedCallback( sp_playlist* pl, void* userdata )
{
    Q_UNUSED(userdata)
    qDebug() << "Playlist metadata updated..";
    QMetaObject::invokeMethod(Spotify::Session::callingSession->m_playlistHash.value(pl),"playlistMetadataUpdated",Qt::QueuedConnection);
}

namespace Spotify
{
    Playlist::Playlist(sp_session* session, int type, sp_playlist* playlist, QObject* parent ):
            TrackCollection( parent ),
            m_session( session ),
            m_playlist( playlist ),
            m_owner( 0 ),
            m_position( 0 ),
            m_isOffline( false ),
            m_type( type ),
            m_hasRef( false )
    {
        m_callbacks = new sp_playlist_callbacks(); //allocate playlist callbacks instance
        m_callbacks->tracks_added = tracksAddedCallback;
        m_callbacks->tracks_removed = tracksRemovedCallback;
        m_callbacks->tracks_moved = tracksMovedCallback;
        m_callbacks->playlist_renamed = playlistRenamedCallback;
        m_callbacks->playlist_state_changed = playlistStateChangedCallback;
        m_callbacks->playlist_metadata_updated = playlistMetadataUpdatedCallback;
        m_callbacks->playlist_update_in_progress = playlistUpdateInProgressCallback;
        m_callbacks->description_changed = 0;
        m_callbacks->track_message_changed = 0;
        m_callbacks->subscribers_changed = 0;
        m_callbacks->track_created_changed = 0;
        m_callbacks->track_seen_changed = 0;
        sp_playlist_add_callbacks(m_playlist,m_callbacks,NULL); //register playlist callbacks

        m_isLoaded = false;
        if(type==PLAYLIST_TYPE_STARRED)
            m_name = tr("Starred tracks");
        load(); //attempt to load metadata for playlist
    }

    bool Playlist::load()
    {
        if( m_playlist )
        {
            if( m_name.isEmpty() )
            {
                const char* charData = sp_playlist_name( m_playlist );
                m_name = QString::fromUtf8( charData, strlen( charData ) );
            }
            m_numTracks = sp_playlist_num_tracks( m_playlist );

            if( !m_owner )
                m_owner = new Spotify::User( sp_playlist_owner( m_playlist ) );

            m_collaborative = sp_playlist_is_collaborative( m_playlist );

            // First time we get tracks for this playlist
          if( m_tracks.count() == 0 )
            {
                for( int i = 0; i < m_numTracks; ++i )
                {
                    sp_track* track = sp_playlist_track( m_playlist, i );
                    Spotify::Track* strack = new Spotify::Track(m_session, track, false, this );
                    m_tracks.append( strack );
                    m_tracksOriginal.append( strack );
                }
            }

            //get current offline sync status for playlist
            m_offlineStatus = (ePlaylistOfflineStatus)sp_playlist_get_offline_status(m_session,m_playlist);

            m_isLoaded = sp_playlist_is_loaded(m_playlist);
            if(m_isLoaded)
                emit loaded();

            return m_isLoaded;
        }
    }

    void Playlist::setPosition(int pos)
    {
        m_position = pos;
    }

    int Playlist::getPosition()
    {
        return m_position;
    }

    int Playlist::getNumTracks()
    {
        if(m_playlist && sp_playlist_is_loaded(m_playlist))
        {
            return sp_playlist_num_tracks( m_playlist );
        }
        else
        {
            return 0;
        }
    }

    int Playlist::getType()
    {
        return m_type;
    }

    ePlaylistOfflineStatus Playlist::getOfflineStatus()
    {
        return m_offlineStatus;
    }

    Playlist::~Playlist()
    {
        if( m_owner )
            m_owner->deleteLater();
        if(m_callbacks)
        {
            sp_playlist_remove_callbacks(m_playlist,m_callbacks,NULL);
            delete m_callbacks;
        }

        if(m_hasRef) //corresponding sp_playlist object needs to be released explicitly
           sp_playlist_release(m_playlist);
    }

    void Playlist::tracksAdded(sp_track** tracks, qint32 num_tracks, qint32 position)
    { //Tracks added to current playlist
        qDebug() << "Adding " << num_tracks << " tracks to playlist " << this->getName() << ", pos " << position;
        QList< Spotify::Track* > trackList;

        for( qint32 i = 0; i < num_tracks; i++)
        {
            sp_track* tr = tracks[i];
            if( tr )
            {
                Spotify::Track* track = new Spotify::Track(m_session, tr, false, this );
                trackList.append( track );
            }
        }
        this->insertTracks( trackList, position );
        emit updated();
    }


    void Playlist::tracksRemoved(qint32* tracks, qint32 num_tracks)
    {
        qDebug() << "Should remove " << num_tracks << " tracks from playlist " << this->getName() ;
        this->removeTracks((const int*) tracks, num_tracks );
    }


    void Playlist::tracksMovedWrapper( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                               void* userdata )
    {
        Q_UNUSED( userdata );
        qDebug() << "Should move " << num_tracks << " tracks";
        /*if( m_playlistHash.contains( pl ) )
        {
            Spotify::Playlist* playlist = m_playlistHash.value( pl );
            if( playlist )
            {
                if( new_position > tracks[0] ) new_position -= 1;
                playlist->moveTracks( tracks, num_tracks, new_position );
            }
        }
        */
    }

    void Playlist::playlistRenamed()
    { //playlist has been renamed
        if( m_playlist )
        {
            const char* charData = sp_playlist_name( m_playlist );
            m_name = QString::fromUtf8( charData, strlen( charData ) );
            emit updated();
        }
    }

    void Playlist::playlistStateChanged()
    { //Playlist state has changed
        if(m_collaborative != sp_playlist_is_collaborative( m_playlist ))
        {
            m_collaborative = sp_playlist_is_collaborative( m_playlist );
        }
        if(m_offlineStatus!=(ePlaylistOfflineStatus)sp_playlist_get_offline_status(m_session, m_playlist))
        {
            m_offlineStatus = (ePlaylistOfflineStatus)sp_playlist_get_offline_status(m_session, m_playlist);
        }
		
		if(sp_playlist_is_loaded( m_playlist))
		{
		    load();
			updateTracks();
		}

         emit updated();
    }

    void Playlist::playlistUpdateInProgress( qint32 done )
    { //playlist update in progresss notification
        if(done)
        {
            load();
        }
    }


    void Playlist::playlistMetadataUpdated()
    {//playlist metadata has been updated
        load();
        updateTracks();
    }

} // namespace Spotify
