//PlaybackThread.h: Thread class responsible for writing audio data using the PulseAudio interface

#ifndef PLAYBACKTHREAD_H
#define PLAYBACKTHREAD_H

#include <QThread>
#include <QList>
#include <QVector>
#include <QByteArray>
#include <QMutex>
#include <pulse/simple.h>

#define SAMPLE_RATE 44100
#define CHANNELS    2

class PlaybackThread : public QThread
{
    Q_OBJECT

public:

    PlaybackThread(QObject* parent = 0);
    ~PlaybackThread();

    void stop(); //stop thread execution

    //public accessor methods

    QMutex* getMutex();
    void clearFIFO();
    void flushPAbuffer();
    void FIFOAppend(const QByteArray& buffer);
    qint32 getFIFOSize();
    void setFIFOSize(qint32 size);
    void incFIFOSize(qint32 size);
    void setBufferingPaused(bool paused);
    void setPlaybackPaused(bool paused);
    bool getPlaybackPaused();
    void setEndOfTrack(bool endOfTrack);
    void setMinBufferSize(qint32 secs); //set min. FIFO buffer size (in secs)
    qint32 getMinBufferSize();
    void setPreBufActive(bool active);
    void setNextTrackStartPos();
    void setNextTrackStartPos(qint32 pos);
    void setNewTrackStarted(bool started);
    bool getNewTrackStarted();


private:

     void run(); //main thread loop implementation

     pa_simple* m_pAudioStream;

     bool m_exitThread; //set to true if thread execution should be stopped

     QVector<QByteArray> m_FIFO; //FIFO datastructure containing audio data to be played

     qint32 m_FIFOSize; //current size of audio FIFO (in bytes)

     QMutex m_audio_mutex; //mutex for buffer change synchronisation between main and playback threads

     bool m_isPaused; //true in case music buffering is currently paused (set by main session thread)

     bool m_playbackPaused; //true in case e.g., user has requested to pause playback of current track

     bool m_endOfTrack; //true when current end of track has been reached

     qint32 m_minBufferSize;  //min. FIFO buffer size (in secs)

     bool m_preBufActive; //set to true in case track pre-buffering is active

     qint32 m_nextTrackStartPos; //stores the start position of the next track audio data in the FIFO (used when pre-buffering active only)

     bool m_newTrackStarted; //set to true when the data of next track should be stored in FIFO (used as condition when nextTrackStartPos should be set)

     QList<qint32> m_trackStartPosList; //used for track pre-buffering (holds track data end position(s) in FIFO)

     void updateTrackStartPositions(); //private method iterating through the track start positions list and updating the start positions as elements are removed from the FIFO.

signals:

     void playbackFinished(); //signal that playback of current track has finished
     void playing(bool); //signal emitted to indicate playing / paused state
     void resumeBuffering(); //signal emitted when FIFO falls below certain value (request to resume data buffering from Spotify)
     void playbackPositionUpdated(qint64); //signal emitted to indicate track progress / position (in secs). Parameter gives length (in secs) of currently added data buffer

};


#endif // PLAYBACKTHREAD_H
