#ifndef SPOTIFYSESSION_H
#define SPOTIFYSESSION_H
#include <spotify/api.h>
#include <QMetaType>
#include <QList>
#include <QImage>
#include <QHash>
#include <QBuffer>
#include <QApplication>
#include <QTimer>
#include <QThread>
#include <QWaitCondition>
#include <QMutex>
#include "version.h"
#include "error.h"
#include "track.h"
#include "playlist.h"
#include "artistcatalogue.h"
#include "albumbrowser.h"
#include "search.h"
#include "user.h"
#include "PlaybackThread.h"



namespace Spotify
{

    class Session: public QObject
    {
        Q_OBJECT
        public:

            Session( QObject* parent = 0 );
            virtual ~Session();

            void init(const sp_bitrate& bitrate = SP_BITRATE_160k); //init a new Spotify session (optional parameter the preferred streaming bitrate)

            void initAudio(); //init the audio / playback thread

            bool playlistLoadingInProgress(); //returns true in case playlist loading (getPlaylists) is currently in progress

            void stopPlaylistLoading(); //request to stop the playlist loading (getPlaylists).

            void setMinBufferSize(qint32 secs); //set the size of the internal FIFO queue used to hold audio data

            void setPreBufferingActive(bool active); //set whether track pre-buffering is active (i.e., buffer next track data in advance)

            bool getPreBufferingActive(); //returns current pre-buffering state

            void setNextTrackStartPos(); //set the next track start position in the audio FIFO

            void setNewTrackStarted(bool started); //indicate start of new track audio buffer data (in playback thread FIFO)

            void setPlaybackPos(qint64 pos);

            void setBufferingPos(qint64 pos);

            void setPlaying(bool playing);

            void resetPlaybackMembers();  //reset the playback related instance members (and removes any pending events from the playback thread)


            /**
             * Pointer to the Session object. Used to map
             * C-callbacks to C++ members.
             */
            static Spotify::Session* callingSession;

            /**
             * Check if the session is valid.
             * @return True if the session is valid, false if it is not.
             */
            bool isValid() { return m_valid; }

            bool m_ProcessEvents;

            bool m_bPlaylistsLoading; //TRUE in case playlist loading is in progress

            bool m_bStopPlaylistLoading; //TRUE in case playlist loading should be stopped (interrupted)

            bool m_DisableNotifyThread;

            bool m_preBufActive; //true if track pre-buffering is enabled

            QTimer *m_pProcEvtTimer;



            /**
             * Audio playback thread
             */
            PlaybackThread* m_pAudioThread;

            /**
             * Fetches the currently logged in user.
             * @return The logged in user (or NULL if not logged in).
             */
            Spotify::User* getUser() const { return m_currentUser; }


            /**
             * Request (root) playlist container (contains all playlists)
             */

            void getPlaylistContainer();


            /**
             * Logs a user in to spotify.
             * @param username Username
             * @param password Password
             * @return Error
             */
            Spotify::Error login( QString username, QString password );

            /**
             * Load a track
             */
            Spotify::Error load( Spotify::Track* track );

            /**
             * Start playback of a Spotify::Track
             * @param track Track to play.
             * @param preBuffer Track should be buffered only (at end of current FIFO)
             */
            Spotify::Error play( Spotify::Track* track, bool preBuffer = false);

            /**
             * Seek to a given position in the stream.
             * @param pos Position to seek to.
             * @return Error
             */
            Spotify::Error seek( int pos );

            /**
             * Stop playback.
             */
            Spotify::Error stop();

            /**
             * Toggle music delivery pause from Spotify.
             */
            Spotify::Error pause();

            /**
             * Toggle playback pause / resume (e.g, stop / continue writing data to PulseAudio).
             */
            void pauseResumePlayback();


            /**
             * Search
             *
             */
            Spotify::Error search( const QString& query, const int trackOffset, const int trackCount,
                                   const int albumOffset,const int albumCount, const int artistOffset,
                                   const int artistCount, void* userData );


            /**
             * Request to load image (identified by imageID) from the Spotify backend
             * @param imageID ID of the image to be loaded.
             * @param Pointer to Spotify::ImageContainer instance for storing the loaded image
             * @return Spotify::Error
             */
            Spotify::Error loadImage(const byte* imageID, Spotify::ImageContainer* pContainer);

            /**
             * Get the collection of playlists.
             */
            Spotify::Error getPlaylists();

            /**
             * Get / update the collection of tracks for specified playlist.
             */
            Spotify::Error getPlaylistTracks(Spotify::Playlist* pPlaylist);


            /**
             * Add a new playlist to the playlist container by name.
             * @param name Name of the new playlist.
             * @return Error
             */
            bool addPlaylist( QString name );

            /**
             * Add a new playlist to the playlist container by link.
             * @param lin Link to the existing playlist.
             * @return Error code.
             */
            bool addPlaylist( Spotify::Link link );

            /**
             * Remove a playlist.
             * @param playlist The index of the playlist to remove.
             * @return Error code.
             */
            bool removePlaylist( const int index );

            /**
             * Move a playlist
             * @param index The index of the playlist to move.
             * @param position The position to move the playlist to.
             * @return Error code.
             */
            bool movePlaylist( const int index, int position );

            /**
             * Browse an artist.
             * @param artist Artist to be browsed. The artist metadata does not have to be loaded.
             * @return TRUE on success, otherwise FALSE
             */
            bool browse( Spotify::Artist* artist );

            /**
             * Browse an album
             * @param album Album to be browsed. The album metadata does not have to be loaded.
             * @param pCallerID Pointer to identifier of of the caller (requester)
             * @return TRUE on success, otherwise FALSE.
             */
            bool browse( Spotify::Album* album, qint32* pCallerID);


            //libopenspotify callbacks

            // Artist Browsing callbacks:
            static void SP_CALLCONV artistBrowseCompleteCallback( sp_artistbrowse* ab, void* userdata );

            // Album Browsing callbacks:
            static void SP_CALLCONV albumBrowseCompleteCallback( sp_albumbrowse* ab, void* userdata );

            // Playlist Container callbacks:
            static void SP_CALLCONV playlistAddedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int position, void* userdata );

            static void SP_CALLCONV playlistRemovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                                 int position, void* userdata );


            static void SP_CALLCONV playlistMovedCallback( sp_playlistcontainer* pc, sp_playlist* playlist,
                                               int oldPosition, int newPosition, void* userdata );


            // Playlist callbacks:
            static void SP_CALLCONV tracksAddedCallback( sp_playlist* pl, sp_track* const* tracks, int num_tracks,
                                            int position, void* userdata );


            static void SP_CALLCONV tracksRemovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, void* userdata );

            static void SP_CALLCONV tracksMovedCallback( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                             void* userdata );

            static void SP_CALLCONV playListContainerLoadedCallback(sp_playlistcontainer *pc, void* userdata);


            static void SP_CALLCONV playlistRenamedCallback( sp_playlist* pl, void* userdata );

            static void SP_CALLCONV playlistStateChangedCallback( sp_playlist* pl, void* userdata );

            static void SP_CALLCONV playlistUpdateInProgressCallback( sp_playlist* pl, bool done, void* userdata );

            static void SP_CALLCONV playlistMetadataUpdatedCallback( sp_playlist* pl, void* userdata );

            // Image callbacks:
            static void SP_CALLCONV imageLoadedCallback( sp_image* image, void* userdata );

            // Search callbacks:
            static void SP_CALLCONV searchCompleteCallback( sp_search* search, void* userdata );

            // Session callbacks:
            static void SP_CALLCONV loggedInCallback( sp_session* session, sp_error error );

            static void SP_CALLCONV loggedOutCallback( sp_session* session );

            static void SP_CALLCONV metadataUpdatedCallback( sp_session* session );

            static void SP_CALLCONV connectionErrorCallback( sp_session* session, sp_error error );

            static void SP_CALLCONV  messageToUserCallback( sp_session* session, const char* message );

            static void SP_CALLCONV notifyMainThreadCallback( sp_session* session );

            static int SP_CALLCONV musicDeliveryCallback( sp_session* session, const sp_audioformat* format,
                                               const void* frames, int num_frames );

            static void SP_CALLCONV playTokenLostCallback( sp_session* session );

            static void SP_CALLCONV logMessageCallback( sp_session* session, const char* message );

            static void SP_CALLCONV endOfTrackCallback( sp_session* session );


        public slots:

            /**
             * Log the current user out of Spotify.
             */
            void logout();

        private slots:
            void updatePlaylistsMetadataSlot();
            void searchDeletedSlot( QObject* parent );
            void artistCatalogueDeletedSlot( QObject* parent );
            void albumBrowserDeletedSlot( QObject* parent );
            void OnProcessThreadEvents();
            void OnRootListContainerLoaded(sp_playlistcontainer* pc);
            void OnPlayListAdded(sp_playlist* playlist, int position);
            void OnPlayListRenamed(sp_playlist* playlist);
            void OnTracksAdded(sp_playlist* pl, sp_track* const* tracks, int num_tracks, int position);
            void OnPlaylistMetadataUpdated(sp_playlist* pl);
            void OnImageLoaded(sp_image* image, void* userdata);
            void OnMusicDelivery(int num_frames, const sp_audioformat* format, QByteArray buffer);
            void OnSearchComplete(sp_search* search);
            void OnAlbumBrowserReady(sp_albumbrowse* ab, qint32*);
            void OnArtistBrowseComplete( sp_artistbrowse* ab );
            void OnResumeBuffering();
            void OnPlaybackPositionUpdated(qint64 duration);
            void OnPlaybackFinished();
            void OnPlayingState(bool state);

        signals:
            /**
             * Signal emitted when an error has occured.
             * @param error Error object indicating the error.
             */
            void error( Spotify::Error error );

            /**
             * Signal emitted when a user is logged in.
             * @param error Error object indicating whether the login was successfull or not.
             */
            void loggedIn( Spotify::Error error );

            /**
             * Signal emitted when a user is logged out.
             */
            void loggedOut();

            /**
             * Signal emitted when meta data has been updated.
             */
            void metadataUpdated();

            /**
             * Signal emitted when a search has completed. The receiver will have
             * ownership of all Spotify::Track pointers and is responsible for deleting the data
             * when these are no longer needed.
             * @param tracks List of Spotify::Track pointers.
             * @param userdata Pointer to an arbitrary structure. Can be used to identify the owner
             * of the search.
             */
            void searchComplete( Spotify::Search* search );

            void artistCatalogueReady( Spotify::ArtistCatalogue* catalogue );

            void albumBrowserReady( Spotify::AlbumBrowser*, qint32 );
            /**
             * Signal emitted when a message to the user has been received.
             */
            void messageToUser( QString message );

            /**
             * Signal emitted when cover art has been received.
             * @param image QImage representation of the cover art.
             * @param callerID ID of intended receiver of result.
             */
            void coverArtReady( QImage image, qint32 callerID );

            /**
             * Signal emitted when the root playlist container has been loaded (internal usage)
             */

            void rootListContainerLoaded(sp_playlistcontainer*);

            /**
             * Signal emitted when a playlist has been added / loaded to playlist container at specified position (internal usage)
             */

            void onPlaylistAdded(sp_playlist*, int);

            /**
             * Signal emitted when a playlist has been renamed (internal usage)
             */

            void onPlaylistRenamed(sp_playlist*);

            /**
             * Signal emitted when tracks have been added to a playlist (internal usage)
             */

            void onTracksAdded(sp_playlist*, sp_track* const*, int, int);

	     /**
             * Signal emitted when image data has been loaded (e.g., album coverart)
             */
            void onImageLoaded(sp_image*, void*); 

            /**
             * Signal emitted when playlist containers have been received. The Spotify::Session object
             * owns the Spotify::Playlist data and the Spotify::Track data within and will delete
             * these when destructed.
             * @param playlists List of Spotify::Playlist pointers.
             */
            void playlistContainersReady( QList<Spotify::Playlist*> );

	      /**
             * Signal emitted when track data for the playlist is ready.
             */
            void playlistTracksReady(Spotify::Playlist*);

            /**
             * Signal emitted when a new playlist has been added. The Spotify::Session object
             * owns the Spotify::Playlist data and the Spotify::Track data within and will delete
             * these when destructed.
             * @playlist The playlist that has been added.
             * @position The index of the new playlist.
             */
            void playlistAdded( Spotify::Playlist* playlist, int position );


            /**
             * Signal emitted when a playlist has been moved.
             * @param oldPosition The old index of the playlist.
             * @param newPosition The new index of the playlist.
             */
            void playlistMoved( int oldPosition, int newPosition );

            /**
             * Signal emitted when a playlist has been removed.
             * @param position The index of the playlist that has been removed.
             */
            void playlistRemoved( int position );

            /**
             * Signal emitted when playback has finished.
             */
            void playbackFinished();


            /**
             * Signal emitted to indicate current playback state (playing (true) / paused (false))
             */

            void playing(bool);

            /**
             * Signal emitted when the current buffering amount has been updated.
             * @param bufferedAmout The amount buffered of the currently loaded track (in milliseconds).
             */
            void trackBufferingAmountUpdated( qint32 bufferedAmount );

            /**
             * Signal emitted when the current playback position in the track has been updated.
             * @param pos The current playback position in the track (in milliseconds).
             */
            void playbackPositionUpdated( qint64 pos );


            /**
             * Signal emitted when the playtoken has been lost.
             */
            void playTokenLost();

            void onNotifyMainThread();

            /**
             * Signal emitted when metadata has been updated for the specified playlist
             */
            void playlistMetadataUpdated(sp_playlist*);

            void onMusicDelivery(int, const sp_audioformat*, QByteArray);

            void onSearchComplete(sp_search*);

            void onAlbumBrowserReady(sp_albumbrowse*, qint32*);

            void onArtistBrowseComplete(sp_artistbrowse*);

            void bufferNextTrack();


        private:

            Spotify::Error m_error;
            Spotify::Track* m_currentTrack;

            sp_session_config m_config;
            sp_session* m_session;
            sp_playlistcontainer* m_playlistcontainer;

            bool m_valid;
            QString m_username;
            QString m_password;
            Spotify::User* m_currentUser;

            qint32 m_bufferedAmount; //current buffered amount (of the currently loaded track)
            qint64 m_playbackPos; //current playback position for the currently loaded track

            bool m_playing;

            // For mapping sp_playlist pointers to Spotify::Playlist pointers
            QHash< sp_playlist*, Spotify::Playlist* > m_playlistHash;

            // For maintaing a sorted (by index) list of Spotify::Playlist pointers
            QList< Spotify::Playlist* > m_playlistList;

            // For mapping sp_search pointers to Spotify::Search pointers
            QHash< sp_search*, Spotify::Search* > m_searchHash;

            // For mapping sp_artistbrowse pointers to Spotify::ArtistCatalogue pointers:
            QHash< sp_artistbrowse*, Spotify::ArtistCatalogue* > m_artistBrowseHash;

            // For mapping sp_albumbrowse pointers to Spotify::AlbumBrowser pointers:
            QHash< sp_albumbrowse*, Spotify::AlbumBrowser* > m_albumBrowseHash;

            //methods to be invoked by corresponding callbacks

            void artistBrowseCompleteWrapper( sp_artistbrowse* ab, void* userdata );

            void albumBrowseCompleteWrapper( sp_albumbrowse* ab, void* userdata );


            void playlistAddedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist, int position,
                                       void* userdata );


            void playlistRemovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                         int position, void* userdata );


            void playlistMovedWrapper( sp_playlistcontainer* pc, sp_playlist* playlist,
                                       int oldPosition, int newPosition, void* userdata );

            void tracksAddedWrapper( sp_playlist* pl, sp_track* const* tracks, int num_tracks,
                                    int position, void* userdata );

            void tracksRemovedWrapper( sp_playlist* pl, const int* tracks, int num_tracks, void* userdata );

            void tracksMovedWrapper( sp_playlist* pl, const int* tracks, int num_tracks, int new_position,
                                     void* userdata );

            void playListContainerLoadedWrapper(sp_playlistcontainer *pc, void* userdata);


            void playlistRenamedWrapper( sp_playlist* pl, void* userdata );

            void playlistStateChangedWrapper( sp_playlist* pl, void*  userdata );

            void playlistUpdateInProgressWrapper( sp_playlist* pl, bool done, void* userdata );

            void playlistMetadataUpdatedWrapper( sp_playlist* pl, void* userdata );

            void imageLoadedWrapper( sp_image* image, void* userdata );


            void searchCompleteWrapper( sp_search* search, void* userdata );

            void loggedInWrapper( Spotify::Error error );

            void loggedOutWrapper();

            void metadataUpdatedWrapper();

            void connectionErrorWrapper( Spotify::Error error );

            void messageToUserWrapper( QString message );

            void notifyMainThreadWrapper();

            int musicDeliveryWrapper( const sp_audioformat* format, const void* frames, int num_frames );

            void playTokenLostWrapper( sp_session* session );

            void logMessageWrapper( QString message );

            void endOfTrackWrapper();

            /* returns true in case all playlist containers have been completely loaded and added to internal playlists list */
            bool isPlaylistContainersReady();


    };
}



Q_DECLARE_METATYPE( Spotify::Playlist* );
Q_DECLARE_METATYPE( Spotify::Track* );
Q_DECLARE_METATYPE( Spotify::Search* );
#endif // SPOTIFYSESSION_H
