//AlbumView.cpp: Implementation of the AlbumView class.

#include "AlbumView.h"

AlbumView::AlbumView(Spotify::Session* pSession, QWidget *pParent)
    : QDialog(pParent)
{
    //default constructor, set up the UI elements etc.

    setWindowTitle(tr("Browse Album"));

    m_AlbumReviewDlg.setWindowTitle(tr("Album Review"));

    m_pAlbumBrowser = NULL;

    m_CallerID = ALBUMVIEW_ID;

    m_pSession = pSession; //set Spotify session instance pointer

    m_pAlbumCoverContainer = new Spotify::ImageContainer(this); //allocate coverart imagecontainer instance

    //setup signal / slot connections
    connect(m_pAlbumCoverContainer,SIGNAL(loaded()),SLOT(OnAlbumCoverLoaded()));
    connect(m_pSession,SIGNAL(albumBrowserReady(Spotify::AlbumBrowser*,qint32)),SLOT(OnAlbumBrowserReady(Spotify::AlbumBrowser*,qint32)));
    connect(&m_AlbumReviewBtn,SIGNAL(clicked()),SLOT(OnAlbumReviewBtn()));

    m_pAlbumTrackView = new QListView(this);
    m_pAlbumTrackDelegate = new ListViewDelegate(this);
    m_pAlbumTrackDelegate->setHeaderFontPointSize(15);
    m_pAlbumTrackDelegate->setSubHeaderFontPointSize(14);
    m_pAlbumTrackDelegate->setItemTextDisabledColor(QColor(190,190,190)); //set disabled item color (gray)
    m_pAlbumTrackModel = new QStandardItemModel(this);

    //connect view to item delegate and model
    m_pAlbumTrackView->setModel(m_pAlbumTrackModel);
    m_pAlbumTrackView->setItemDelegate(m_pAlbumTrackDelegate);
    connect(m_pAlbumTrackView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));

    QFont captionFont;
    captionFont.setFamily(QApplication::font().family());
    captionFont.setPointSize(19);
    m_AlbumCaption.setFont(captionFont);
    m_AlbumCaption.setWordWrap(true);

    m_AlbumCover.setFixedSize(ALBUMVIEW_COVERART_WIDTH,ALBUMVIEW_COVERART_HEIGHT);

    m_ArtistViewBtn.setFixedWidth(ALBUMVIEW_COVERART_WIDTH);
    m_ArtistViewBtn.setDisabled(true); //currently disabled

    m_AlbumReviewBtn.setText(tr("Album Review"));

    m_LeftSubLayout.addWidget(&m_AlbumCover);
    m_LeftSubLayout.addWidget(&m_ArtistViewBtn);
    m_LeftSubLayout.setSpacing(10);

    m_MainLayout.addLayout(&m_LeftSubLayout,0,0,3,1);
    m_MainLayout.addWidget(&m_AlbumCaption,0,1,1,1);
    m_MainLayout.addWidget(&m_AlbumReviewBtn,1,1,1,1);
    m_MainLayout.addWidget(m_pAlbumTrackView,2,1,1,1);

    setLayout(&m_MainLayout);
}

AlbumView::~AlbumView()
{
}

void AlbumView::setAlbum(Spotify::Album* pAlbum)
{
    //public method for setting the album instance to be displayed
    bool status;

    if(m_pSession && pAlbum)
    {
        m_AlbumCover.clear();

        m_pAlbumTrackModel->clear(); //clear model rows

        m_AlbumCaption.clear();

        m_AlbumReviewBtn.setVisible(false);

        status = m_pSession->browse(pAlbum,&m_CallerID); //request to browse specified album details

        if(!status)
        {
            //error occured
            QMaemo5InformationBox::information(this,tr("Fetching of album details failed!"));
        }
        else
        {
            //enable progress indication, browse call succeeded
            if(pAlbum->getYear()!=0)
                m_AlbumCaption.setText(pAlbum->getName() + " (" + QString::number(pAlbum->getYear()) + ")"); //set album title
            else
                m_AlbumCaption.setText(pAlbum->getName()); //set album title (no year)
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true);
        }
    }
}

void AlbumView::OnAlbumBrowserReady(Spotify::AlbumBrowser* browser, qint32 callerID)
{
     //Album browser request complete, set up the Album View

    Spotify::Error error;

    QFont artistBtnFont;
    artistBtnFont.setFamily(QApplication::font().family());
    artistBtnFont.setPointSize(15);
    QFontMetrics fm(artistBtnFont);

    if(callerID==m_CallerID) //check that results was requested by album view
    {
        if(browser)
        {
            error = sp_albumbrowse_error(browser->getAlbumBrowser());
            if(error.isError())
            {
                //error occured (album browse request failed)
                QMaemo5InformationBox::information(this,tr("Fetching of album details failed: ") + error.toString());
            }
            else
            {
                m_pAlbumBrowser = browser;
                m_pAlbumBrowser->load();
                if(m_pAlbumBrowser->getReview()!="")
                    m_AlbumReviewBtn.setVisible(true); //album review available
                m_pAlbumBrowser->sortByDiscIndex();
                if(m_pAlbumBrowser->getAlbum())
                {
                    m_pSession->loadImage(m_pAlbumBrowser->getAlbum()->getCover(),m_pAlbumCoverContainer); //request album coverart

                }
                if(m_pAlbumBrowser->getArtist())
                {
                  m_ArtistViewBtn.setText(fm.elidedText(m_pAlbumBrowser->getArtist()->getName(),Qt::ElideRight,290));
                }
                //add tracks to listview
                addTracks();
            }
        }

        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
    }
}

void AlbumView::OnAlbumReviewBtn()
{
    //display the album review dialog
    if(m_pAlbumBrowser)
    {
        m_AlbumReviewDlg.setText(m_pAlbumBrowser->getReview());
        m_AlbumReviewDlg.showMaximized();
    }
}

void AlbumView::addTracks()
{
    //add album tracks to the listview

    QPixmap iconMap;

    iconMap.load("/usr/share/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    for(int pos = 0; pos < m_pAlbumBrowser->getTrackCount(); pos++)
    {
        Spotify::Track* currTrack = m_pAlbumBrowser->getTrack(pos);
        if(currTrack)
        {
            QStandardItem* trackItem = new QStandardItem();
            if(!currTrack->isAvailable())
            {
                trackItem->setEnabled(false); //not possible to play unavailable track, but display it anyway in the listview
                trackItem->setData(-1,ListViewDelegate::customDataRole); //assign -1 as position for disabled track
            }
            else
            {
               trackItem->setData(pos,ListViewDelegate::customDataRole); //assign position in internal album track list
            }
            trackItem->setEditable(false);
            trackItem->setData(trackTypeIcon,ListViewDelegate::iconRole);
            trackItem->setData(currTrack->getName(),ListViewDelegate::headerRole);
            if(currTrack->getArtist())
            {
                trackItem->setData(currTrack->getArtist()->getName(),ListViewDelegate::subHeaderLeftJustifyRole);
            }
            trackItem->setData(currTrack->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
            m_pAlbumTrackModel->appendRow(trackItem);
        }
    }

    if(m_pAlbumTrackModel->rowCount()>0)
    { //scroll to top, no current / selected row by default
        m_pAlbumTrackView->scrollToTop();
        QModelIndex index = m_pAlbumTrackModel->index(-1,-1);
        m_pAlbumTrackView->setCurrentIndex(index);
    }
}

void AlbumView::OnTrackActivated(QModelIndex trackIndex)
{
    //A track has been activated (e.g., selected) in the list of album tracks

    if(trackIndex.data(ListViewDelegate::customDataRole).toInt()!=-1)
    {
        m_pAlbumTrackView->setEnabled(false); //disable listview temporarily

        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //show progress indicator
        emit playQueueSet(m_pAlbumBrowser->getTracks(false),m_pAlbumTrackView->currentIndex().row()); //signal that playqueue has been set / updated
        hide();
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
        m_pAlbumTrackView->setEnabled(true); //re-enable the listview
    }
 }


void AlbumView::OnAlbumCoverLoaded()
{
    //Album coverart has been loaded (for current album),
    //assign it to the album coverart container.
    if(m_pAlbumCoverContainer)
        m_AlbumCover.setPixmap(QPixmap::fromImage(m_pAlbumCoverContainer->getImage()).scaled(ALBUMVIEW_COVERART_WIDTH,ALBUMVIEW_COVERART_HEIGHT));
}
