#include "artist.h"
#include "link.h"
#include <QDebug>

namespace Spotify
{
    Artist::Artist( Spotify::Artist& artist, bool ref ):
        QObject( 0 ),
        m_hasRef( ref )
    {
        m_artist = artist.getArtist();
        if( m_artist )
        {
            if( m_hasRef )
            {
                qDebug() << "An artist was copied and a reference was added.";
                sp_artist_add_ref( m_artist );
            }

            m_isLoaded = artist.isLoaded();

            if( m_isLoaded )
                m_name = artist.getName();

            setParent( artist.parent() );
            if( thread() != artist.thread() )
                moveToThread( artist.thread() );
        }
    }

    Artist::Artist( sp_artist* artist, bool ref, QObject *parent):
            QObject(parent),
            m_isLoaded( false ),
            m_artist( artist ),
            m_hasRef( ref )
    {
        if( m_artist )
        {
            if( m_hasRef )
            {
                qDebug() << "An artist was created with ref";
                sp_artist_add_ref( m_artist );
            }
            load();
        }
    }

    bool Artist::load()
    {
        if( m_artist && sp_artist_is_loaded( m_artist ) )
        {
            // Set meta data:
            const char* charData = sp_artist_name( m_artist );
            m_name = QString::fromUtf8( charData, strlen( charData ) );
            m_isLoaded = true;
            emit loaded();
        }
        else
            qDebug() << "Artist wasn't ready..";
        return m_isLoaded;
    }

    Spotify::Artist* Spotify::Artist::fromLink( Spotify::Link& link, QObject* parent )
    {
        if( link.isValid() && link.type() == Spotify::Link::ARTIST )
        {
            sp_artist* artist = sp_link_as_artist( link.getLink() );
            if( artist )
                return new Spotify::Artist( artist, true, parent );
        }
        return 0;
    }

    Artist::~Artist()
    {
        if( m_artist && m_hasRef )
        {
            qDebug() << "A reffed artist was dereffed and deleted.";
            sp_artist_release( m_artist );
        }
    }

} // namespace Spotify
