//QueueView.cpp: Implementation of the QueueView class.

#include "QueueView.h"


QueueView::QueueView(Spotify::Session* pSession, QWidget* parent)
    : QMainWindow(parent)
{
    //construct the QueueView UI

    setAttribute(Qt::WA_Maemo5StackedWindow);

    setWindowTitle(tr("Now Playing"));

    m_pSession = pSession; //set the Spotify session instance

    m_pCurrTrack = NULL;
    m_currQueuePos = 0;
    m_currShufflePos = 0;
    m_Active = false;
    m_UpdateBufPos = true;
    m_Shuffle = false;


    //define slider used for displaying buffering and track progress
    m_pProgressSlider = new CustomSlider(this,355,40);
    m_pProgressSlider->setBgColor(QColor(51,51,51)); //background color of progress slider
    m_pProgressSlider->setP1Color(QColor(8,117,189)); //color of buffering progress bar
    m_pProgressSlider->setP2Color(QColor(41,166,241)); //color of track progress bar
    m_pProgressSlider->setTextColor(QColor(255,255,255)); //set text color to white
    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);

    m_pSpectrum = new SpectrumAnalyzer(this,50,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    //init the trackinfo listview instance
    m_pTrackInfoView = new QListView(this);
    m_pTrackInfoModel = new QStandardItemModel(this);
    m_pTrackInfoDelegate = new ListViewDelegate(this);
    m_pTrackInfoDelegate->setHeaderFontPointSize(15);
    m_pTrackInfoDelegate->setSubHeaderFontPointSize(14);

    m_pTrackInfoView->setItemDelegate(m_pTrackInfoDelegate); //connect the delegate to view
    m_pTrackInfoView->setModel(m_pTrackInfoModel); //connect the model to view
    m_pTrackInfoView->setProperty("FingerScrollable",true); //make sure that the current item is not accidentally de-selected when scrolling

    //set up signal and slot connections
    connect(m_pSession,SIGNAL(trackPositionUpdated(qint64)),SLOT(OnTrackPosUpdated(qint64)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(coverArtReady(QImage)),SLOT(OnAlbumCoverReady(QImage)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(playbackFinished()),SLOT(OnTrackPlaybackFinished()));
    connect(m_pSession,SIGNAL(bufferNextTrack()),SLOT(OnBufferNextTrack()));
    connect(m_pTrackInfoView,SIGNAL(activated(QModelIndex)),SLOT(OnTrackActivated(QModelIndex)));
    connect(&m_PauseResumeBtn,SIGNAL(clicked()),SLOT(OnPauseResumeToggle()));
    connect(&m_ShuffleBtn,SIGNAL(clicked()),SLOT(OnShuffleToggle()));
    connect(&m_PrevBtn,SIGNAL(clicked()),SLOT(OnPrevTrackBtn()));
    connect(&m_NextBtn,SIGNAL(clicked()),SLOT(OnNextTrackBtn()));

    m_AlbumCoverContainer.setFixedSize(355,355);

    //style the player control buttons
    m_PauseResumeBtn.setCheckable(true); //toggle button (between pause / resume states)
    m_PauseResumeBtn.setEnabled(false);

    m_PauseResumeBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 100px;}"
            "QPushButton {min-width: 100px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/usr/share/images/play.png);}"
            "QPushButton:on {background-image: url(/usr/share/images/pause.png);}"
            "QPushButton:pressed {background-image: url(/usr/share/images/play-on.png);}");

    m_PrevBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px; border-style: none; background-image: url(/usr/share/images/previous.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/previous-on.png);}");

    m_NextBtn.setStyleSheet("QPushButton {min-width: 80px; min-height: 44px; max-width: 80px; max-height: 44px;border-style: none; background-image: url(/usr/share/images/next.png);}"
                            "QPushButton:pressed {background-image: url(/usr/share/images/next-on.png);}");

    m_ShuffleBtn.setCheckable(true);

    m_ShuffleBtn.setStyleSheet(
            "QPushButton {max-height: 44px;}"
            "QPushButton {min-height: 44px;}"
            "QPushButton {max-width: 60px;}"
            "QPushButton {min-width: 60px;}"
            "QPushButton {border-style: none;}"
            "QPushButton {background-image: url(/usr/share/images/shuffle.png);}"
            "QPushButton:on {background-image: url(/usr/share/images/shuffle-down.png);}");

    //sub-layout containing player controls, progress slider and spectrum analyzer
    m_PlayControls.setSpacing(0);
    m_PlayControls.addWidget(&m_PrevBtn,0,0,1,1,Qt::AlignRight);
    m_PlayControls.addWidget(&m_PauseResumeBtn,0,1,1,1,Qt::AlignLeft);
    m_PlayControls.addWidget(&m_NextBtn,0,2,Qt::AlignLeft);
    m_PlayControls.setColumnStretch(2,6);
    m_PlayControls.addWidget(&m_ShuffleBtn,0,3,1,1,Qt::AlignLeft);
    m_PlayControls.setColumnStretch(3,5);
    m_PlayControls.addWidget(m_pProgressSlider,0,4,1,1);
    m_PlayControls.setColumnMinimumWidth(4,370);
    m_PlayControls.setColumnStretch(4,10);
    m_PlayControls.addWidget(m_pSpectrum,0,5,1,1,Qt::AlignRight);
    m_PlayControls.setContentsMargins(10,5,10,0);

    //set up the QueueView UI layout
    m_GridLayout.addWidget(&m_AlbumCoverContainer,0,0,7,1,Qt::AlignTop);
    m_GridLayout.addWidget(m_pTrackInfoView,0,1,7,1);
    m_GridLayout.addLayout(&m_PlayControls,7,0,1,-1);
    m_GridLayout.setContentsMargins(0,5,0,10); //set layout margin

    m_MainLayout.setLayout(&m_GridLayout);

    setCentralWidget(&m_MainLayout);
}

QueueView::~QueueView()
{

}

void QueueView::setNowPlaying(QList<Spotify::Track*> queue, int currQueuePos, bool playing, qint64 currPlaybackPos, qint64 currBufPos, bool doShuffle, qint32 currShufflePos)
{
        //set the view into play queue (now playing mode), displaying the current play queue

           m_PlayQueue = queue;

           if(doShuffle)
           { //shuffle mode is active, and queue should be shuffle
               m_currShufflePos = currShufflePos;

               if(m_Shuffle)
               {
                   shuffleQueue();
               }
           }

           addTracks(m_PlayQueue); //add track data to track list

           if(currQueuePos!=-1)
           {
               m_pCurrTrack = m_PlayQueue.at(currQueuePos);

               m_currQueuePos = currQueuePos;

              emit playQueuePosUpdated(m_currQueuePos,m_currShufflePos);

              //m_TrackInfo.setCurrentRow(m_currQueuePos);
              QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
              if(index.isValid())
                m_pTrackInfoView->setCurrentIndex(index);


             if(m_pCurrTrack)
             {
                 loadTrackData(m_pCurrTrack);
             }

             m_PauseResumeBtn.setEnabled(true);
             m_PauseResumeBtn.setChecked(playing);

             //set the state of the progress indicators
             OnTrackPosUpdated(currBufPos);
             OnPlaybackPositionUpdated(currPlaybackPos);
            }
           else
           {
               clearViewSelection();
           }
}

void QueueView::clearViewSelection()
{
    //clear the current view selection (i.e., set to state when no track is active)

    m_AlbumCoverContainer.clear();

    m_pTrackInfoView->clearSelection();

    m_pProgressSlider->setPos1(0);
    m_pProgressSlider->setPos2(0);
    m_pProgressSlider->setText("");

    m_pCurrTrack = NULL;

    m_PauseResumeBtn.setChecked(false);
    m_PauseResumeBtn.setEnabled(false);
}

void QueueView::OnPlayingState(bool state)
{
    if(m_Active)
    {
        if(state)
        { //playing state
            m_pSpectrum->Start();
        }
        else
        { //paused state
           m_pSpectrum->Stop();
        }
    }
}

void QueueView::OnBufferNextTrack()
{
    Spotify::Error error;
    Spotify::Track* track;
    int shufflePos;

   //invoked when ready to start buffering the following track (pre-buffering mode)
    if(m_Active) //current view instance is active
    {
        if(m_Shuffle)
        { //shuffle mode active; get next track position from shuffle queue (if queue is not empty)
           if(m_currShufflePos < m_ShuffleQueue.size()-1)
           {
               shufflePos = m_ShuffleQueue.at(m_currShufflePos+1);
           }
           else
           {
               shufflePos = m_PlayQueue.size()-1;
           }
        }

        if((!m_Shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_Shuffle && (shufflePos < m_PlayQueue.size())))
        {
            if(!m_Shuffle)
                track = m_PlayQueue.at(m_currQueuePos+1);
            else
                track = m_PlayQueue.at(shufflePos);
            if(m_pCurrTrack)
            {
                m_pSession->setNewTrackStarted(true); //indicate start of new track audio buffer data (in playback thread FIFO)
                error = m_pSession->play(track,true); //initiate buffering of track
                if(error.isError())
                {
                    qDebug()<< "Error starting track playback: " << error.toString();
                }
                m_UpdateBufPos = false; //do not update the buffering position (current track still playing)
            }
        }
        else
        {

          //end of playqueue reached, set next track pos to end of current track
            m_pSession->setNewTrackStarted(false);
            m_pSession->setNextTrackStartPos();
        }
    }
}

void QueueView::loadTrackData(Spotify::Track* track)
{
    //Load track data for specified track (e.g., album cover art)

    QString strCurrTrackDuration, strCurrTrackProgressInfo;
    QTextStream outStream(&strCurrTrackProgressInfo);

    if(track)
    {
        if(track->getAlbum())
        {
            m_pSession->getAlbumCover(track->getAlbum()); //request to load album coverart
        }

        m_currTrackDuration = track->getDuration().hour()*3600+track->getDuration().minute()*60+track->getDuration().second(); //current track duration (in secs)
        strCurrTrackDuration = track->getDurationAsString();
        outStream << "0:00" << " / " << strCurrTrackDuration;
        m_pProgressSlider->setText(strCurrTrackProgressInfo);
    }
}

void QueueView::addTracks(QList<Spotify::Track*> trackList)
{
    //add tracks from specified list to main listview.
    //Overriden method.

    QPixmap iconMap;
    QString strSubHeader;

    iconMap.load("/usr/share/images/icon-type-song-color.png"); //load song (track) type icon from file

    QIcon trackTypeIcon(iconMap);

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

    //m_TrackInfo.clear();
    m_pTrackInfoModel->clear();

     for(int trackPos = 0; trackPos < trackList.size(); trackPos++)
     {
             Spotify::Track* currTrack = trackList.at(trackPos);
             QStandardItem* trackInfo = new QStandardItem();
             if(currTrack)
             {
                 strSubHeader = "";
                 trackInfo->setData(trackTypeIcon,ListViewDelegate::iconRole);
                 trackInfo->setData(currTrack->getName(),ListViewDelegate::headerRole);
                 trackInfo->setData(currTrack->getDurationAsString(),ListViewDelegate::subHeaderRightJustifyRole);
                 if(currTrack->getArtist())
                 {
                     strSubHeader = currTrack->getArtist()->getName();
                 }
                 trackInfo->setData(strSubHeader,ListViewDelegate::subHeaderLeftJustifyRole);
                 m_pTrackInfoModel->appendRow(trackInfo);
             }
     }

     setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void QueueView::OnTrackActivated(QModelIndex index)
{
    //A track has been activated (e.g., selected) in the track list
    //Display artist and album cover pic (if available)

    Q_UNUSED(index);

    Spotify::Error error;

        m_currQueuePos = m_pTrackInfoView->currentIndex().row();

        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);

        if(m_pCurrTrack) //we have got a valid track instance
        {
            m_pSession->resetPlaybackMembers();
            loadTrackData(m_pCurrTrack);
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //request to start playback of track
            //emit playing();
            if(error.isError())
            { //error occured
                QMaemo5InformationBox::information(this,tr("Error starting track playback: ")+error.toString());
            }
            else
            {
                m_PauseResumeBtn.setChecked(true);
                m_PauseResumeBtn.setEnabled(true);
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos,m_currShufflePos);
        }
}

void QueueView::OnTrackPosUpdated(qint64 trackPos)
{
    //Track buffering position has been updated

    if(m_Active && m_UpdateBufPos)
    {
        int currPos = (trackPos*355) / (m_currTrackDuration*1000);
        m_pProgressSlider->setPos1(currPos); //set curr. slider position
    }

}

void QueueView::OnPlaybackPositionUpdated(qint64 trackPos)
{
    //Track playback position updated

    if(m_Active)
    {

        QString strCurrTrackProgressInfo, strCurrPlaybackPos, strCurrTrackDuration;

        QTextStream outStream(&strCurrTrackProgressInfo);

        QTime currTrackPos(trackPos/3600000,trackPos/60000,(trackPos/1000)%60,0);

        if(m_pCurrTrack)
        { //set track progress info in format current playback position / total track duration
                if(currTrackPos.hour()>0)
                    strCurrPlaybackPos = currTrackPos.toString("h:mm:ss"); //position exceeds 60 mins
                else
                    strCurrPlaybackPos = currTrackPos.toString("m:ss");
                if(m_pCurrTrack->getDuration().hour()>0) //duration longer than 60 mins
                    strCurrTrackDuration = m_pCurrTrack->getDuration().toString("h:mm:ss");
                else
                    strCurrTrackDuration = m_pCurrTrack->getDuration().toString("m:ss");
                outStream << strCurrPlaybackPos << " / " << strCurrTrackDuration;
                m_pProgressSlider->setText(strCurrTrackProgressInfo);
        }

        int currPos = (trackPos*355) / (m_currTrackDuration*1000);

        m_pProgressSlider->setPos2(currPos); //set curr. slider position
    }
}

void QueueView::OnPauseResumeToggle()
{
    //Play / Pause button has bee clicked (toggled)

    if(m_pCurrTrack) //some track has been activated
    {
        m_pSession->pausePlayback(); //pause / resume playback of current track
    }
}

void QueueView::setActive(bool active, bool shuffleActive)
{
    //set whether the current instance is active (i.e., whether signals should be processed),
    //as well as whether the shuffle function is active.
    m_Active = active;

    m_Shuffle = shuffleActive;
    if(m_Shuffle)
    {
        shuffleQueue();
        m_ShuffleBtn.setChecked(true);
        setWindowTitle(tr("Now Playing") + " - " + tr("Shuffle"));
    }
    else
    {
        m_ShuffleBtn.setChecked(false);
        setWindowTitle(tr("Now Playing"));
    }
}

bool QueueView::getShuffleActive()
{
    //returns whether the shuffle mode is active
    return m_Shuffle;
}

void QueueView::OnNextTrackBtn()
{
    //move to next track in play queue

    Spotify::Error error;

    if(m_Shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos < m_ShuffleQueue.size()-1)
        {
             m_currShufflePos++;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = m_PlayQueue.size()-1;
        }
    }

    if((!m_Shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_Shuffle && (m_currQueuePos < m_PlayQueue.size())))
    {
        if(!m_Shuffle)
            m_currQueuePos++;
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            m_pSession->resetPlaybackMembers();
            loadTrackData(m_pCurrTrack);
            qDebug()<<"Starting track at pos " << m_currQueuePos << " in playqueue";
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //initiate track playback
            if(error.isError())
            {
                qDebug()<< "Error starting track playback: " << error.toString();
            }
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos,m_currShufflePos); //emit pos updated
        }
    }
}

void QueueView::OnPrevTrackBtn()
{
    //move to previous track in play queue

    Spotify::Error error;

    if(m_Shuffle)
    { //shuffle mode active, get position of next track to play from shuffle queue
        if(m_currShufflePos > 0)
        {
            m_currShufflePos--;
            m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
        }
        else
        { //no tracks left in shuffle queue
            m_currQueuePos = -1;
        }
    }

    if((!m_Shuffle && (m_currQueuePos > 0))||(m_Shuffle && (m_currQueuePos>=0)))
    {
        if(!m_Shuffle)
            m_currQueuePos--;
        m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
        if(m_pCurrTrack)
        {
            QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
            if(index.isValid())
              m_pTrackInfoView->setCurrentIndex(index);
            m_pProgressSlider->setPos1(0);
            m_pProgressSlider->setPos2(0);
            m_pSession->resetPlaybackMembers();
            loadTrackData(m_pCurrTrack);
            qDebug()<<"Starting track at pos " << m_currQueuePos << " in playqueue";
            m_PauseResumeBtn.setChecked(true); //toggle playing state
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator
            QMaemo5InformationBox::information(this,tr("Buffering"));
            error = m_pSession->play(m_pCurrTrack); //initiate track playback
            if(error.isError())
            {
                qDebug()<< "Error starting track playback: " << error.toString();
            }
             setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
            emit playQueuePosUpdated(m_currQueuePos,m_currShufflePos); //emit pos updated
        }
    }
}

void QueueView::OnTrackPlaybackFinished()
{
    if(m_Active)
    {
        //Playback of current track has finished
        //move to next track in play queue, if possible

        qDebug() << "Playback finished..";

        Spotify::Error error;

        if(m_Shuffle)
        { //shuffle mode active, get position of next track to play from shuffle queue
            if(m_currShufflePos<m_ShuffleQueue.size()-1)
            {
                m_currShufflePos++;
                m_currQueuePos = m_ShuffleQueue.at(m_currShufflePos);
            }
            else
            { //no tracks left in shuffle queue
                m_currQueuePos = m_PlayQueue.size();
            }
        }

        if((!m_Shuffle && (m_currQueuePos < m_PlayQueue.size()-1))||(m_Shuffle && (m_currQueuePos<m_PlayQueue.size())))
        {
            if(!m_Shuffle) //normal playback mode
                m_currQueuePos++;
            m_pCurrTrack = m_PlayQueue.at(m_currQueuePos);
            if(m_pCurrTrack)
            {
                m_pProgressSlider->setPos1(0);
                m_pProgressSlider->setPos2(0);
                QModelIndex index = m_pTrackInfoModel->index(m_currQueuePos,0);
                if(index.isValid())
                  m_pTrackInfoView->setCurrentIndex(index);
                 loadTrackData(m_pCurrTrack);
                if(!m_pSession->getPreBufferingActive()) //pre-buffering is not active, buffering / playback should be initiated
                {
                    m_pSession->resetPlaybackMembers();
                    error = m_pSession->play(m_pCurrTrack); //initiate track playback
                    if(error.isError())
                    {
                        qDebug()<< "Error starting track playback: " << error.toString();
                    }
                }
                else
                { //track pre-buffering mode active; reset playback position and playing state
                    m_pSession->setPlaybackPos(0); //reset current playback position
                    m_UpdateBufPos = true; //allow update of buffering state
                    OnTrackPosUpdated(m_pSession->getBufferingPos()); //update the current buffering state
                }
                emit playQueuePosUpdated(m_currQueuePos,m_currShufflePos); //emit pos updated
            }
        }
        else
        {
            //end of playback queue reached
            clearViewSelection();
            emit playQueuePosUpdated(-1,m_currShufflePos); //emit pos updated (indicates that playqueue has reached the end)
        }
    }
}

void QueueView::OnAlbumCoverReady(QImage image)
{
    //Album coverart has been loaded (for currently selected track),
    //assign it to the album coverart container.
    m_AlbumCoverContainer.setPixmap(QPixmap::fromImage(image).scaled(355,355));
}

void QueueView::OnShuffleToggle()
{
    //toggle shuffle state (on / off)

    if(m_Shuffle)
    {
        m_Shuffle = false;
        setWindowTitle(tr("Now Playing"));
    }
    else
    {
        shuffleQueue(); //shuffle the current playqueue
        m_Shuffle = true;
        setWindowTitle(tr("Now Playing") + " - " + tr("Shuffle"));
    }
}

void QueueView::shuffleQueue()
{
    //Shuffle the current playback queue (the m_ShuffleQueue list contains the shuffled queue positions).
    //The shuffle is implemented using the Fisher-Yates (Knuth's) algorithm.

    qsrand(QTime::currentTime().msec()); //initialise the random number seed

    m_ShuffleQueue.clear();

    //initialise the position indexes in the shuffle queue
    for (int pos=0; pos<m_PlayQueue.size(); pos++)
    {
        m_ShuffleQueue.append(pos);
    }

    for(int i = m_ShuffleQueue.size()-1; i>=0; i--)
    {
        qint32 j = qrand() % (i+1);
        if(i==j)
            continue;
        m_ShuffleQueue.swap(j,i); //swap list elements
    }

    m_currShufflePos = -1;
}
