//PlaybackThread.cpp: Implementation of the PlaybackThread class.

#include "PlaybackThread.h"
#include <QDebug>
#include <QList>
#include <QMutex>
#include <QByteArray>

QList<QByteArray> g_FIFO; //FIFO datastructure containing audio data to be played
qint32 g_FIFOSize; //current size of audio FIFO (in bytes)
QMutex g_audio_mutex; //mutex for buffer change synchronisation between main and playback threads
bool g_isPaused; //true in case music buffering is currently paused (set by main session thread)
bool g_playbackPaused; //true in case e.g., user has requested to pause playback of current track
bool g_endOfTrack; //true when current end of track has been reached

#define SAMPLE_RATE 44100
#define CHANNELS    2

PlaybackThread::PlaybackThread(QObject* parent)
    : QThread(parent)
{
    m_exitThread = false;

    //init pulseaudio interface

    pa_sample_spec ss;
    pa_channel_map map;
    pa_buffer_attr buffer;

    ss.format = PA_SAMPLE_S16NE;
    ss.channels = CHANNELS;
    ss.rate = SAMPLE_RATE;

    m_pAudioStream = NULL;

    buffer.maxlength = pa_usec_to_bytes(750000, &ss);
    buffer.tlength = -1;
    buffer.prebuf = -1;
    buffer.minreq = -1;

    pa_channel_map_init_stereo(&map);

    m_pAudioStream = pa_simple_new(NULL,"SpotyMaemo", PA_STREAM_PLAYBACK, NULL, "SpotyStream",&ss,&map, &buffer,NULL);
}

PlaybackThread::~PlaybackThread()
{
    //cleanup
    int err=0;
    g_audio_mutex.lock(); //lock mutex
    pa_simple_flush(m_pAudioStream,&err); //flush playback buffer
    pa_simple_free(m_pAudioStream); //close and free pulseaudio connection
    g_audio_mutex.unlock(); //release lock
}

void PlaybackThread::stop()
{
    //stop the thread execution
    g_audio_mutex.lock(); //lock mutex
    m_exitThread = true;
    g_audio_mutex.unlock(); //release lock
}

void PlaybackThread::run()
{
   //thread main loop

    QByteArray buffer;
    int error = 0;

    forever
    {
        g_audio_mutex.lock(); //lock mutex

        if(m_exitThread)
        {
            m_exitThread = false;
            g_audio_mutex.unlock(); //release lock
            break;
        }

        if(g_endOfTrack && g_FIFO.isEmpty())
        { //end of track reached, emit notification signal
            g_endOfTrack = false;
            emit playbackFinished();
        }

        if(!g_FIFO.isEmpty() && !g_playbackPaused)
        { //data in audio FIFO and playback has not been paused
            emit playing(true);
            buffer = g_FIFO.takeFirst();
            g_FIFOSize -= buffer.size();

            if(g_FIFOSize<30*SAMPLE_RATE*sizeof(int16_t)*CHANNELS && g_isPaused)
            { //request to resume buffering (less than 30 secs of data in buffer)
                g_isPaused = false;
                emit resumeBuffering();
            }

            g_audio_mutex.unlock();

            qint64 duration = (buffer.size()*1000)/(SAMPLE_RATE*sizeof(int16_t)*CHANNELS); //buffer duration (msecs)

            pa_simple_write(m_pAudioStream, buffer.data(),buffer.size(),&error); //write audio stream data to pulseaudio server

            emit playbackPositionUpdated(duration);

        }
        else
        {
          emit playing(false);
          g_audio_mutex.unlock(); //release lock
        }

        msleep(5); //sleep for a while
    }
}

