#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);

    QCoreApplication::setApplicationName("QSpot"); //Setup app name

    setWindowTitle(tr("QSpot - Playlists"));

    m_pSpectrum = new SpectrumAnalyzer(this,55,40);
    m_pSpectrum->setColor(QColor(255,255,255)); //set spectrum color to white

    m_pSession = new Spotify::Session(this);

    m_pAppSettings = new QSettings("qspot","qspot", this);

    m_pLoginDlg = new LoginDialog(m_pSession,m_pAppSettings,this);

    //set up the the main window menu

    menuBar()->addAction(tr("Settings"));
    menuBar()->addAction(tr("About"));

     //setup signals etc.

    connect(m_pLoginDlg,SIGNAL(rejected()),SLOT(OnLoginDialogRejected()));
    connect(m_pSession,SIGNAL(loggedIn(Spotify::Error)),SLOT(OnLoggedIn(Spotify::Error)));
    connect(m_pSession,SIGNAL(playlistContainersReady(QList<Spotify::Playlist*>)),this,SLOT(OnPlaylistContainersReady(QList<Spotify::Playlist*>)));
    connect(m_pSession,SIGNAL(playing(bool)),SLOT(OnPlayingState(bool)));
    connect(m_pSession,SIGNAL(searchComplete(Spotify::Search*)),SLOT(OnSearchComplete(Spotify::Search*)));
    connect(m_pSession,SIGNAL(trackPositionUpdated(qint64)),SLOT(OnTrackPosUpdated(qint64)));
    connect(m_pSession,SIGNAL(playbackPositionUpdated(qint64)),SLOT(OnPlaybackPositionUpdated(qint64)));
    connect(m_pSession,SIGNAL(playTokenLost()),SLOT(OnPlaybackTokenLost()),Qt::QueuedConnection);
    connect(&m_PlaylistInfo,SIGNAL(itemActivated(QListWidgetItem*)),SLOT(OnPlaylistActivated(QListWidgetItem*)));
    connect(menuBar(),SIGNAL(triggered(QAction*)),SLOT(OnMainMenuAction(QAction*)));
    connect(this,SIGNAL(showLogin()),SLOT(OnShowLogin()),Qt::QueuedConnection);
    connect(&m_NowPlayingIcon,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_NowPlayingInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(&m_NowPlayingPositionInfo,SIGNAL(clicked()), SLOT(OnNowPlaying()));
    connect(m_pSpectrum,SIGNAL(clicked()),SLOT(OnNowPlaying()));
    connect(&m_SearchBtn,SIGNAL(clicked()),SLOT(OnSearch()));
    connect(&m_SearchInput,SIGNAL(returnPressed()),SLOT(OnSearch()));

    m_SearchInput.setDisabled(true);

    m_SearchBtn.setStyleSheet("QPushButton {min-width: 78px; min-height: 48px; max-width: 78px; max-height: 48px;border-style: none; background-image: url(/usr/share/images/icon-search.png);}"
                              "QPushButton:pressed {background-image: url(/usr/share/images/icon-search-on.png);}");
    m_SearchBtn.setDisabled(true);

    m_PlaylistInfo.clear();

    QPixmap iconMap;

    iconMap.load("/usr/share/images/icon-nowplaying.png"); //load icon from file

    m_NowPlayingIcon.setPixmap(iconMap); //set now playing icon
    m_NowPlayingIcon.setFixedSize(24,24);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft);

    m_NowPlayingFont.setFamily(QApplication::font().family());
    m_NowPlayingFont.setPointSize(14);
    m_NowPlayingInfo.setFont(m_NowPlayingFont);
    m_NowPlayingInfo.setFixedWidth(558);
    m_NowPlayingInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);
    m_NowPlayingInfo.setText("----"); //default (empty) value for now playing info
    m_NowPlayingPositionInfo.setFont(m_NowPlayingFont);
    m_NowPlayingPositionInfo.setFixedWidth(132);
    m_NowPlayingPositionInfo.setAlignment(Qt::AlignLeft|Qt::AlignVCenter);

    m_GridLayout.addWidget(&m_SearchInput,0,0,1,3);
    m_GridLayout.addWidget(&m_SearchBtn,0,3);
    m_GridLayout.addWidget(&m_PlaylistInfo,1,0,1,-1);
    m_GridLayout.addWidget(&m_NowPlayingIcon,2,0);
    m_GridLayout.addWidget(&m_NowPlayingInfo,2,1,1,1,Qt::AlignLeft);
    m_GridLayout.addWidget(&m_NowPlayingPositionInfo,2,2,1,1,Qt::AlignLeft);
    m_GridLayout.addWidget(m_pSpectrum,2,3);

    m_MainLayout.setLayout(&m_GridLayout);

    setCentralWidget(&m_MainLayout);

    m_pTrackView = new TrackView(m_pSession,this);
    connect(m_pTrackView,SIGNAL(playQueueSet(QList<Spotify::Track*>)),SLOT(OnSetPlayQueue(QList<Spotify::Track*>)));
    connect(m_pTrackView,SIGNAL(playQueuePosUpdated(qint32)),SLOT(OnPlayQueuePosUpdated(qint32)));

    m_pQueueView = new QueueView(m_pSession,this);
    connect(m_pQueueView,SIGNAL(playQueuePosUpdated(qint32)),SLOT(OnPlayQueuePosUpdated(qint32)));

    emit showLogin(); //emit signal requesting to show login dialog
}

MainWindow::~MainWindow()
{
}

void MainWindow::OnShowLogin()
{
  m_pLoginDlg->showMaximized(); //show the login dialog
}

void MainWindow::OnPlayingState(bool state)
{
    if(state)
    { //playing state
        m_pSpectrum->Start();
        m_playing = true;
    }
    else
    { //paused state
        m_pSpectrum->Stop();
        m_playing = false;
    }
}

void MainWindow::OnTrackPosUpdated(qint64 pos)
{
    //set current buffering position
    m_currBuffPos = pos;

}

void MainWindow::OnPlaybackPositionUpdated(qint64 pos)
{
        //set current playback position, and update the track progress part of the now playing infobar

        QString strCurrPlaybackPos, strCurrTrackDuration, strCurrTrackProgressInfo;

        m_currPlaybackPos = pos;

        QTextStream outStream(&strCurrTrackProgressInfo);

        QTime currTrackPos(0,pos/60000,(pos/1000)%60,0);

        Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

        if(currTrack)
        {

            //set track progress info in format current playback position / total track duration
            if(currTrackPos.hour()>0)
                strCurrPlaybackPos = currTrackPos.toString("h:mm:ss"); //position exceeds 60 mins
            else
                strCurrPlaybackPos = currTrackPos.toString("m:ss");
            if(currTrack->getDuration().hour()>0) //duration longer than 60 mins
               strCurrTrackDuration = currTrack->getDuration().toString("h:mm:ss");
            else
               strCurrTrackDuration = currTrack->getDuration().toString("m:ss");
            outStream << "(" << strCurrPlaybackPos << "/" << strCurrTrackDuration << ")";
            m_NowPlayingPositionInfo.setText(strCurrTrackProgressInfo);
        }
}

void MainWindow::OnNowPlaying()
{
    //show the current playqueue view (now playing)

    if(m_PlayQueue.size()>0) //some tracks in playqueue
    {
        if(m_pQueueView)
        {
            m_pQueueView->setNowPlaying(m_PlayQueue,m_currQueuePos,m_playing,m_currPlaybackPos,m_currBuffPos); //set queue (now playing) view mode
            m_pQueueView->show();
        }
    }
}

void MainWindow::OnSetPlayQueue(QList<Spotify::Track*> queue)
{
    //set the current (active) playqueue
    m_PlayQueue = queue;
}

void MainWindow::OnPlayQueuePosUpdated(qint32 pos)
{
     m_currQueuePos = pos;

     if(m_currQueuePos>=0)
     {

         Spotify::Track* currTrack = m_PlayQueue.at(m_currQueuePos);

         if(currTrack)
         {
             m_strNowPlayingInfo = currTrack->getName();
             if(currTrack->getArtist())
             {
                 m_strNowPlayingInfo += ", ";
                 m_strNowPlayingInfo += currTrack->getArtist()->getName();
             }
             m_NowPlayingInfo.setText(m_strNowPlayingInfo);
         }
     }
     else //empty now playing infobar (when playqueue has reached end)
     {
        m_NowPlayingInfo.setText("----");
        m_NowPlayingPositionInfo.setText("");
     }
}

void MainWindow::logout()
{
  //logout the current user and display the login dialog
  m_pSession->logout();
  close();
}

void MainWindow::OnLoginDialogRejected()
{
//request to reject the login dialog, exit app
   qApp->quit();
}

void MainWindow::OnLoggedIn(Spotify::Error error)
{
    //Slot invoked when response from login request received

    QString strLoginMsg;
    QMessageBox msgBox;

    if(error.isError())
    {
        strLoginMsg = tr("Error:") + error.toString(); //write error string
        msgBox.setText(strLoginMsg);
        msgBox.exec();
        m_pLoginDlg->resetState(); //re-enable login button etc.
    }
    else
    { //successful login to Spotify service, request user playlist / track data
    	m_pLoginDlg->setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
    	setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display main window progress indicator
    	m_pLoginDlg->hide();
		m_SearchInput.setFocus(); //set default focus to search field
    	m_pSession->getPlaylistContainer();
    	QMaemo5InformationBox::information(this,tr("Loading playlists")); //display loading notification
    }
}



void MainWindow::OnPlaylistContainersReady(QList<Spotify::Playlist*> playlists)
{
   //add items to playlists list widget

   QPixmap iconMap;

   iconMap.load("/usr/share/images/icon-type-list.png"); //load playlist type icon from file

   QIcon listTypeIcon(iconMap);

   m_Playlists = playlists; //store list of playlists

   m_PlaylistInfo.clear();

   for(int pos=0; pos < playlists.size(); pos++)
   {
       Spotify::Playlist* pl = playlists.at(pos);
       QListWidgetItem *item = new QListWidgetItem();
       item->setIcon(listTypeIcon);
       item->setText(pl->getName());
       item->setData(Qt::UserRole,pos); //store associated position in playlist instance list
       m_PlaylistInfo.addItem(item);
   }

   m_SearchInput.setDisabled(false);
   m_SearchBtn.setDisabled(false);

   setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false);
}

void MainWindow::OnPlaylistActivated(QListWidgetItem *item)
{
	//Display the trackview for the activated playlist

	int pos = item->data(Qt::UserRole).toInt();

	Spotify::Playlist* pPlaylist = m_Playlists.at(pos);

        if(m_pTrackView)
        {
            m_pTrackView->setPlaylist(pPlaylist); //set active playlist for track view
            m_pTrackView->show();
        }

}

void MainWindow::OnMainMenuAction(QAction* action)
{

  //User has selected an item from the app main menu
}

void MainWindow::OnSearch()
{
    //Search request

    Spotify::Error error;

    if(m_SearchInput.text().trimmed()!="")
    {
        QMaemo5InformationBox::information(this,tr("Searching"));
        setAttribute(Qt::WA_Maemo5ShowProgressIndicator,true); //display progress indicator

        error = m_pSession->search(m_SearchInput.text(),0,500,0,10,0,10,NULL);
        if(error.isError())
        { //some error occured
            QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
            setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
        }
    }
}

void MainWindow::OnSearchComplete(Spotify::Search* search)
{
    //Search request completed

    if(search)
    {
        Spotify::Error error = sp_search_error(search->getSearch()); //TODO: move into Search class
        if(error.isError())
        { //some error has occured
            QMaemo5InformationBox::information(this,tr("Error: ") + error.toString());
        }
        else
        {
            if(search->getTracks().size()>0)
            {//some track / tracks returned by search
                if(m_pTrackView)
                {
                    m_SearchInput.setText(""); //clear search field
                    m_SearchInput.setCursorPosition(0);
                    m_pTrackView->setSearchResults(search);
                    m_pTrackView->show();
                }
            }
            else
            {
                QMaemo5InformationBox::information(this,tr("No tracks found!"));
            }
        }
    }

    setAttribute(Qt::WA_Maemo5ShowProgressIndicator,false); //hide progress indicator
}

void MainWindow::OnPlaybackTokenLost()
{
    //playback token has been lost (e.g., same account has logged in from somewhere else); noitify user
     QMaemo5InformationBox::information(this,tr("Playback paused due to the same account being used somewhere else!"));
}



