import QtQuick 1.0
import "components"
import "scripts/createobject.js" as ObjectCreator
import "scripts/youtube.js" as YT
import "scripts/videolistscripts.js" as Scripts

Page {
    id: root

    property int topMargin: 0
    property bool showingFavourites : videoFeed == _FAVOURITES_FEED
    property string videoFeed
    property string searchFeed
    property alias checkList: videoList.checkList
    property bool enableSearch: false

    function setVideoFeed(feed) {
        videoFeed = feed;
        YT.getYouTubeVideos(feed);
    }

    function resetVideoFeed() {
        searchFeed = "";
        videoListModel.reset();
    }

    function searchVideoFeed(query) {
        var encodedQuery = encodeURIComponent(query.replace(/\s.\s/gi, " "));
        var safe = Settings.safeSearch ? "strict" : "none";
        searchFeed = videoFeed + "&safeSearch=" + safe + "&q=%22" + encodedQuery + "%22%7C" + encodedQuery.replace(/\s/g, "+") + "&orderby=relevance";
        videoListModel.reset();
    }

    function goToVideoInfo(video) {
        pageStack.push(Qt.resolvedUrl("YTInfoPage.qml"));
        pageStack.currentPage.setVideo(video);
    }


    function selectAll() {
        Scripts.selectAll()
    }

    function selectNone() {
        Scripts.selectNone();
    }

    function addVideosToFavourites() {
        YT.addVideosToFavourites();
    }

    function deleteVideosFromFavourites() {
        YT.deleteVideosFromFavourites();
    }

    function showPlaylistDialog() {
        YT.showPlaylistDialog();
    }

    function addVideosToWatchLaterPlaylist() {
        YT.addVideosToWatchLaterPlaylist();
    }

    function addVideosToPlaybackQueue() {
        YT.addVideosToPlaybackQueue();
    }

    function addVideosToDownloadQueue(convertToAudio) {
        YT.addVideosToDownloadQueue(convertToAudio);
    }

    function deleteVideosFromUploads() {
        var list = [];
        for (var i = 0; i < videoList.checkList.length; i++) {
            list.push(videoListModel.get(videoList.checkList[i]).id);
        }
        YouTube.deleteFromUploads(list);
        videoList.checkList = [];
    }

    tools: Menu {

        menuItems: [

            MenuSearchItem {
                placeholderText: qsTr("Search")
                visible: enableSearch
                onSearch: searchVideoFeed(text)
                onClicked: searchVideoFeed(text)
            },

            MenuItem {
                text: qsTr("Show all")
                visible: searchFeed != ""
                onClicked: resetVideoFeed()
            },

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                closeMenuWhenClicked: false
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? selectAll() : selectNone()
            },

            MenuItem {
                text: qsTr("Delete")
                visible: videoFeed == _UPLOADS_FEED
                enabled: videoList.checkList.length > 0
                onClicked: deleteVideosFromUploads()
            },

            MenuItem {
                text: showingFavourites ? qsTr("Delete from favourites") : qsTr("Add to favourites")
                visible: YouTube.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showingFavourites ? deleteVideosFromFavourites() : addVideosToFavourites()
            },

            MenuItem {
                text: qsTr("Add to playlist")
                visible: YouTube.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showPlaylistDialog()
            },

            MenuItem {
                text: qsTr("Watch later")
                visible: YouTube.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToWatchLaterPlaylist()
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToPlaybackQueue()
            },

            MenuItem {
                text: qsTr("Download")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(true)
            }
        ]
    }

    Connections {
        target: YouTube
        onAddedToFavourites: if (showingFavourites) videoListModel.reset();
        onDeletedFromFavourites: if (showingFavourites) videoListModel.reset();
        onDeletedFromUploads: if (videoFeed == _UPLOADS_FEED) videoListModel.reset();
    }

    ListView {
        id: videoList

        property variant checkList : []

        anchors { fill: parent; topMargin: root.topMargin }
        maximumFlickVelocity: Settings.maximumFlickVelocity
        highlightRangeMode: ListView.NoHighlightRange
        cacheBuffer: 2500
        clip: true
        onAtYEndChanged: {
            if ((atYEnd) && (videoListModel.count < videoListModel.totalResults) && (!videoListModel.loading)) {
                if (searchFeed == "") {
                    YT.getYouTubeVideos(videoFeed);
                }
                else {
                    YT.getYouTubeVideos(searchFeed);
                }
            }
        }

        footer: Item {
            id: footer

            width: videoList.width
            height: 100
            visible: videoListModel.loading
            opacity: footer.visible ? 1 : 0

            BusyDialog {
                anchors.centerIn: footer
                opacity: footer.opacity
            }
        }

        model: ListModel {
            id: videoListModel

            property bool loading : false
            property bool error: false
            property int totalResults
            property int page : 0

            function reset() {
                videoListModel.clear();
                videoListModel.page = 0;
                videoListModel.totalResults = 0;
                videoListModel.error = false;
                if (searchFeed == "") {
                    YT.getYouTubeVideos(videoFeed);
                }
                else {
                    YT.getYouTubeVideos(searchFeed);
                }
            }
        }

        delegate: YTListDelegate {
            id: delegate

            function addOrRemoveFromCheckList() {
                var cl = videoList.checkList;
                if (!delegate.checked) {
                    cl.push(index);
                }
                else {
                    for (var i = 0; i < cl.length; i++) {
                        if (cl[i] == index) {
                            cl.splice(i, 1);
                        }
                    }
                }
                videoList.checkList = cl;
            }

            checked: Scripts.indexInCheckList(index)
            onClicked: {
                if (videoList.checkList.length === 0) {
                    goToVideoInfo(ObjectCreator.cloneObject(videoListModel.get(index)));
                }
                else {
                    addOrRemoveFromCheckList();
                }
            }
            onPressAndHold: addOrRemoveFromCheckList()
            onPlayClicked: if (videoList.checkList.length === 0) playVideos([ObjectCreator.cloneObject(videoListModel.get(index))]);
        }

        ScrollBar {}
    }

    Label {
        id: noResultsText

        anchors.centerIn: videoList
        font.pixelSize: _LARGE_FONT_SIZE
        font.bold: true
        color: "#4d4d4d"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("No videos found")
        visible: (!videoListModel.loading) && (videoListModel.count == 0)
    }
}
