#include "dailymotion.h"
#include "json.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QString>
#include <QUrl>
#include <QStringList>
#include <QDateTime>
#include <QTimer>

#define EXCLUDED_CHARS " \n\t#[]{}=+$&*()<>@|',/!\":;?"

Dailymotion::Dailymotion(QObject *parent) :
    QObject(parent), nam(0), clientId("c1eda48dcc5e41f23cbb"), clientSecret("66d40e4bf9949dea9313722059acb53e53017f8b"), uploading(false) {
    emit clientIdChanged();
    emit clientSecretChanged();
    refreshTimer = new QTimer(this);
    refreshTimer->setSingleShot(false);
    connect(refreshTimer, SIGNAL(timeout()), this, SLOT(refreshAccessToken()));
}

void Dailymotion::setNetworkAccessManager(QNetworkAccessManager *manager) {
    nam = manager;
}

void Dailymotion::setPlaybackPath(const QString &path) {
    playbackPath = path;
}

void Dailymotion::setBusyMessage(const QString &message) {
    busyMessage = message;
    emit busyMessageChanged();
}

void Dailymotion::clearBusyMessage() {
    busyMessage = QString();
    emit busyMessageChanged();
}

void Dailymotion::signIn(const QString &user, const QString &pass) {
    setBusyMessage(tr("Signing in..."));
    QUrl url("https://api.dailymotion.com/oauth/token");
    QByteArray data("grant_type=password&client_id="
                    + clientId.toAscii() + "&client_secret=" + clientSecret.toAscii()
                    + "&username=" + user.toAscii() + + "&password=" + pass.toAscii()
                    + "&scope=userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = nam->post(request, data);
    connect(reply, SIGNAL(finished()), this, SLOT(checkForToken()));
}

void Dailymotion::setAccount(const QString &user, const QString &aToken, const QString &rToken, int expiry) {
    if ((!rToken.isEmpty()) && (expiry == 0)) {
        refreshAccessToken(rToken);
    }
    else {
        setUsername(user);
        setAccessToken(aToken);
        setRefreshToken(rToken, expiry);
    }
}

void Dailymotion::setUsername(const QString &user) {
    username = user;
    emit usernameChanged();
}

void Dailymotion::setAccessToken(const QString &token) {
    accessToken = token;
    emit accessTokenChanged();
    if ((accessToken.isEmpty()) || (!refreshTimer->isActive())) {
        emit userSignedInChanged();
    }
}

void Dailymotion::setRefreshToken(const QString &token, int expiry) {
    refreshToken = token;
    if ((!token.isEmpty()) && (expiry > 0)) {
        setTokenExpiry(expiry);
    }
}

void Dailymotion::setTokenExpiry(int expiry) {
    tokenExpiry = expiry * 900;
    refreshTimer->start(tokenExpiry);
}

void Dailymotion::refreshAccessToken(QString token) {
    if (token.isEmpty()) {
        token = refreshToken;
    }
    else {
        setBusyMessage(tr("Refreshing access token..."));
    }
    QUrl url("https://api.dailymotion.com/oauth/token");
    QByteArray data("grant_type=refresh_token&client_id=" + clientId.toAscii()
                    + "&client_secret=" + clientSecret.toAscii()
                    + "&refresh_token=" + token.toAscii());
    QNetworkRequest request(url);
    QNetworkReply* reply = nam->post(request, data);
    connect(reply, SIGNAL(finished()), this, SLOT(checkForToken()));
}

void Dailymotion::signOut() {
    setBusyMessage(tr("Signing out..."));
    QUrl url("https://api.dailymotion.com/logout");
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", "OAuth " + accessToken.toAscii());
    QNetworkReply* reply = nam->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedOut()));
}

void Dailymotion::checkIfSignedOut() {
    clearBusyMessage();
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("Unable to revoke access to your Dailymotion account"));
        return;
    }
    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if (statusCode == 200) {
        emit signedOut();
        emit alert(tr("You have signed out of your Dailymotion account"));
    }
    else {
        emit alert(tr("Unable to revoke access to your Dailymotion account"));
    }
    reply->deleteLater();
}

void Dailymotion::checkForToken() {
    clearBusyMessage();
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("Cannot sign in to Dailymotion"));
        return;
    }

    using namespace QtJson;

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();
    if (!ok) {
        QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
        emit alert(statusText);
    }
    else {
        QVariantMap error = result.value("error").toMap();
        if (!error.isEmpty()) {
            QString errorString = error.value("message").toString();
            emit alert(errorString);
        }
        else {
            QString user = result.value("uid").toString();
            QString aToken = result.value("access_token").toString();
            QString rToken = result.value("refresh_token").toString();
            int expiry = result.value("expires_in").toInt();
            if ((aToken.isEmpty()) || (rToken.isEmpty()) || (expiry == 0)) {
                emit alert(tr("Cannot sign in to Dailymotion"));
            }
            else {
                signedIn(user, aToken, rToken, expiry);
            }
        }
    }
    reply->deleteLater();
}

void Dailymotion::postRequest(const QUrl &url, const QByteArray &data) {
    /* Helper method that posts HTTP POST requests */

    QNetworkRequest request(url);
    request.setRawHeader("Authorization", "OAuth " + accessToken.toAscii());
    QNetworkReply* reply = nam->post(request, data);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void Dailymotion::deleteRequest(const QUrl &url) {
    /* Helper method that posts HTTP DELETE requests */

    QNetworkRequest request(url);
    request.setRawHeader("Authorization", "OAuth " + accessToken.toAscii());
    QNetworkReply* reply = nam->deleteResource(request);
    connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
}

void Dailymotion::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit alert(tr("Dailymotion server unreachable"));
        return;
    }

    using namespace QtJson;

    bool ok;
    QString response(reply->readAll());
    QVariantMap result = Json::parse(response, ok).toMap();
    if (!ok) {
        QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
        emit alert(statusText);
    }
    else {
        QVariantMap error = result.value("error").toMap();
        if (!error.isEmpty()) {
            QString errorString = error.value("message").toString();
            emit alert(errorString);
        }
        else {
            QString id = result.value("id").toString();
            emit postSuccessful(id);
        }
    }
    QTimer::singleShot(1000, this, SLOT(disconnectSignals()));
    reply->deleteLater();
}

void Dailymotion::disconnectSignals() {
    disconnect(this, SIGNAL(postSuccessful(QString)), 0, 0);
    disconnect(this, SIGNAL(postFailed()), 0, 0);
}

void Dailymotion::addToFavourites(const QString &id) {
    QUrl url("https://api.dailymotion.com/me/favorites/" + id);
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(addedToFavourites()));
}

void Dailymotion::deleteFromFavourites(const QString &id) {
    QUrl url("https://api.dailymotion.com/me/favorites/" + id);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(deletedFromFavourites()));
}

void Dailymotion::addToPlaylist(const QString &id, const QString &playlistId) {
    QUrl url("https://api.dailymotion.com/video/" + id + "/playlists/" + playlistId);
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(addedToPlaylist()));
}

void Dailymotion::deleteFromPlaylist(const QString &id, const QString &playlistId) {
    QUrl url("https://api.dailymotion.com/video/" + id + "/playlists/" + playlistId);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(deletedFromPlaylist()));
}

void Dailymotion::createPlaylist(const QString &title) {
    QUrl url("https://api.dailymotion.com/me/playlists");
    QByteArray data("name=" + title.toAscii().toPercentEncoding(EXCLUDED_CHARS));
    postRequest(url, data);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(playlistCreated(QString)));
}

void Dailymotion::deletePlaylist(const QString &id) {
    QUrl url("https://api.dailymotion.com/playlist/" + id);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(playlistDeleted()));
}

void Dailymotion::joinGroup(const QString &id) {
    QUrl url("https://api.dailymotion.com/group/" + id + "/members/" + username);
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(groupJoined()));
}

void Dailymotion::leaveGroup(const QString &id) {
    QUrl url("https://api.dailymotion.com/group/" + id + "/members/" + username);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(groupLeft()));
}

void Dailymotion::addComment(const QString &id, const QString &comment) {
    QUrl url("https://api.dailymotion.com/video/" + id + "/comments");
    QByteArray data("message=" + comment.toAscii().toPercentEncoding(EXCLUDED_CHARS));
    postRequest(url, data);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(commentAdded()));
}

void Dailymotion::follow(const QString &userId) {
    QUrl url("https://api.dailymotion.com/me/following/" + userId);
    postRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(followed()));
}

void Dailymotion::unfollow(const QString &userId) {
    QUrl url("https://api.dailymotion.com/me/following/" + userId);
    deleteRequest(url);
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(unfollowed()));
}

void Dailymotion::getVideoUrl(const QString &id) {
    QString url = "http://iphone.dailymotion.com/video/" + id;
    QNetworkReply* reply = nam->get(QNetworkRequest(QUrl(url)));
    connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void Dailymotion::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error()) {
        emit alert("Unable to play video");
        emit videoUrlError();
    }
    else {
        QString response(reply->readAll());
        QString videoUrl = response.split("type=\"video/x-m4v\" href=\"").at(1).split("\"").at(0);
        if (!videoUrl.startsWith("http")) {
            emit alert(tr("Unable to play video"));
            emit videoUrlError();
        }
        else {
            emit gotVideoUrl(videoUrl);
        }
    }
    reply->deleteLater();
}

void Dailymotion::uploadVideo(const QVariantMap &video) {
    videoToUpload = video;
    QString filename(video.value("filePath").toString());
    fileToBeUploaded = new QFile(filename);
    if (!fileToBeUploaded->exists()) {
        emit uploadFailed(tr("File does not exist"));
        return;
    }
    emit uploadStarted();
    QNetworkRequest request(QUrl("https://api.dailymotion.com/file/upload"));
    request.setRawHeader("Authorization", "OAuth " + accessToken.toAscii());
    QNetworkReply *reply = nam->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(setUploadUrl()));
}

void Dailymotion::setUploadUrl() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit uploadFailed(tr("Server unreachable"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    if (statusCode == 200) {
        QString response(reply->readAll().replace(" ", "").replace("\\", ""));
        QUrl url(response.split("upload_url\":\"").at(1).split("\"").first());
        performVideoUpload(url);
    }
    else {
        QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
        emit uploadFailed(statusText);
    }
}

void Dailymotion::performVideoUpload(const QUrl &url) {
    fileToBeUploaded->open(QIODevice::ReadOnly);
    emit sizeChanged(fileToBeUploaded->size());
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", "OAuth " + accessToken.toAscii());
    uploadReply = nam->post(request, fileToBeUploaded);
    connect(uploadReply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(updateUploadProgress(qint64,qint64)));
    connect(uploadReply, SIGNAL(finished()), this, SLOT(uploadFinished()));
    uploadTime.start();
}

void Dailymotion::updateUploadProgress(qint64 bytesSent, qint64 bytesTotal) {
    float progress = float (fileToBeUploaded->size() + bytesSent) / float (fileToBeUploaded->size() + bytesTotal);
    int eta = int (((uploadTime.elapsed() / progress) - uploadTime.elapsed()) / 1000);
    emit progressChanged(progress, eta);
}

void Dailymotion::abortVideoUpload() {
    uploadReply->abort();
}

void Dailymotion::uploadFinished() {
    fileToBeUploaded->close();

    if (uploadReply->error()) {
        if (uploadReply->error() == QNetworkReply::OperationCanceledError) {
            emit uploadCancelled();
            return;
        }
        else {
            QString statusText = uploadReply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit uploadFailed(statusText);
        }
    }
    else {
        QString response(uploadReply->readAll().replace(" ", ""));
        QString url = response.split("url\":\"").at(1).split("\"").first();
        if (!url.isEmpty()) {
            setUploadMetadata(url);
        }
        else {
            emit uploadFailed(tr("Cannot set upload metadata"));
        }
    }
}

void Dailymotion::setUploadMetadata(const QString &uploadUrl) {
    QUrl url("https://api.dailymotion.com/me/videos");
    QByteArray params;
    params.append("url=" + uploadUrl.toAscii());
    foreach (QString key, videoToUpload.keys()) {
        params.append(key.toAscii() + "=" + videoToUpload.value(key).toByteArray().toPercentEncoding(EXCLUDED_CHARS) + "&");
    }
    postRequest(url, params.left(params.size() - 1));
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(uploadCompleted()));
    connect(this, SIGNAL(postFailed()), this, SIGNAL(uploadFailed()));
}

void Dailymotion::updateVideoMetadata(const QString &id, const QVariantMap &metadata) {
    QUrl url("https://api.dailymotion.com/video/" + id);
    QByteArray params;
    foreach (QString key, metadata.keys()) {
        params.append(key.toAscii() + "=" + metadata.value(key).toByteArray().toPercentEncoding(EXCLUDED_CHARS) + "&");
    }
    postRequest(url, params.left(params.size() - 1));
    connect(this, SIGNAL(postSuccessful(QString)), this, SIGNAL(videoMetadataUpdated()));
}
