#include "settings.h"
#include <QDir>
#include <QDesktopServices>
#include <QFile>
#include <QLocale>

Settings::Settings(QSettings *parent) :
    QSettings(QString("cuteTube-QML"), QString("cuteTube-QML"), parent) {
}

Settings::~Settings() {

}

void Settings::restoreSettings() {
    beginGroup("Media");
    setYouTubeDownloadQuality(value("youtubeDownloadQuality", QString("hq")).toString());
    setHomescreen(value("homescreen", QString("homepage")).toString());
    setPauseWhenMinimized(value("pauseWhenMinimized", true).toBool());
    setMonitorClipboard(value("monitorClipboard", false).toBool());
    setPlayClipboardVideos(value("playClipboardVideos", false).toBool());
    setDownloadClipboardVideos(value("downloadClipboardVideos", false).toBool());
    setMediaPlayer(value("mediaPlayer", QString("cutetubePlayer")).toString());
    setDefaultSite(value("defaultSite", QString("youtube")).toString());
    setXTubeEnabled(value("enableXTube", false).toBool());
    endGroup();

    beginGroup("Transfers");
    setDefaultTransferStatus(value("defaultTransferStatus", QString("queued")).toString());
    setDownloadPath(value("downloadPath", QDesktopServices::storageLocation(QDesktopServices::DocumentsLocation).append("/cuteTube-QML/")).toString());
    setTransferPriority(value("transferPriority", QString("uploads")).toString());
    endGroup();

    beginGroup("Search");
    setSafeSearch(value("safeSearch", false).toBool());
    setRecentSearches(value("recentSearches", QStringList()).toStringList());
    setSearchOrder(value("searchOrder", QString("relevance")).toString());
    endGroup();

    beginGroup("Appearance");
    setTheme(value("theme", QString("darkblue")).toString());
    setActiveColor(value("activeColor", QString("#066bbe")).toString());
    setActiveColorString(value("activeColorString", QString("color8")).toString());
    setGradientColor(value("gradientColor", QString("black")).toString());
    setGradientColorHigh(value("gradientColorHigh", QString("#524e4e")).toString());
    setLanguage(value("language", QString("en")).toString());
    endGroup();

    beginGroup("System");
    setOrientation(value("screenOrientation", QString("auto")).toString());
    endGroup();

    beginGroup("Accounts");
    setYouTubeUsername(value("youtubeUsername", QString()).toString());
    setYouTubeAccessToken(value("youtubeAccessToken", youtubeAccessToken).toString());
    setDailymotionAccount(value("dailymotionUsername", QString()).toString(), value("dailymotionAccessToken", QString()).toString(), value("dailymotionRefreshToken", QString()).toString(), 0);
    setFacebookAccessToken(value("facebookAccessToken", QString()).toString());
    setTwitterAccount(value("twitterAccessToken", QString()).toString(), value("twitterTokenSecret", QString()).toString());
    endGroup();
}

void Settings::saveSettings() {
    beginGroup("Media");
    setValue("youtubeDownloadQuality", youtubeDownloadQuality);
    setValue("homescreen", homescreen);
    setValue("pauseWhenMinimized", pauseWhenMinimized);
    setValue("monitorClipboard", monitorClipboard);
    setValue("playClipboardVideos", playClipboardVideos);
    setValue("downloadClipboardVideos", downloadClipboardVideos);
    setValue("mediaPlayer", mediaPlayer);
    setValue("defaultSite", defaultSite);
    setValue("enableXTube", xtubeEnabled);
    endGroup();

    beginGroup("Transfers");
    setValue("defaultTransferStatus", defaultTransferStatus);
    setValue("downloadPath", downloadPath);
    setValue("xtubeDownloadPath", xtubeDownloadPath);
    setValue("transferPriority", transferPriority);
    endGroup();

    beginGroup("Search");
    setValue("safeSearch", safeSearch);
    setValue("recentSearches", recentSearches);
    setValue("searchOrder", searchOrder);
    endGroup();

    beginGroup("Appearance");
    setValue("theme", theme);
    setValue("activeColor", activeColor);
    setValue("activeColorString", activeColorString);
    setValue("gradientColor", gradientColor);
    setValue("gradientColorHigh", gradientColorHigh);
    setValue("language", language);
    endGroup();

    beginGroup("System");
    setValue("screenOrientation", screenOrientation);
    endGroup();

    beginGroup("Accounts");
    setValue("youtubeUsername", youtubeUsername);
    setValue("youtubeAccessToken", youtubeAccessToken);
    setValue("dailymotionUsername", dailymotionUsername);
    setValue("dailymotionAccessToken", dailymotionAccessToken);
    setValue("dailymotionRefreshToken", dailymotionRefreshToken);
    setValue("facebookAccessToken", facebookAccessToken);
    setValue("twitterAccessToken", twitterAccessToken);
    setValue("twitterTokenSecret", twitterTokenSecret);
    endGroup();
}

void Settings::setYouTubeDownloadQuality(const QString &quality) {
    youtubeDownloadQuality = quality;
    emit youtubeDownloadQualityChanged(quality);
}

void Settings::setHomescreen(const QString &screen) {
    homescreen = screen;
    emit homescreenChanged();
}

void Settings::setDefaultTransferStatus(const QString &status) {
    defaultTransferStatus = status;
    emit defaultTransferStatusChanged();
}

void Settings::setTransferPriority(const QString &priority) {
    transferPriority = priority;
    emit transferPriorityChanged(priority);
}

void Settings::setPauseWhenMinimized(bool pause) {
    pauseWhenMinimized = pause;
    emit pauseWhenMinimizedChanged();
}

void Settings::setMonitorClipboard(bool monitor) {
    monitorClipboard = monitor;
    emit monitorClipboardChanged(monitor);
}

void Settings::setPlayClipboardVideos(bool play) {
    playClipboardVideos = play;
    emit playClipboardVideosChanged();
}

void Settings::setDownloadClipboardVideos(bool download) {
    downloadClipboardVideos = download;
    emit downloadClipboardVideosChanged();
}

void Settings::setDownloadPath(const QString &path) {
    downloadPath = path;
    QDir pathCheck;
    pathCheck.mkpath(path + ".thumbnails/");
    emit downloadPathChanged(path);
    setXTubeDownloadPath(QString(path + "XTube/"));
}

void Settings::setSafeSearch(bool safe) {
    safeSearch = safe;
    emit safeSearchChanged();
}

void Settings::setRecentSearches(const QStringList &searches) {
    recentSearches = searches;
    recentSearches.removeDuplicates();
    emit recentSearchesChanged();
    if (searches.isEmpty()) {
        emit alert(tr("Your previous searches have been cleared"));
    }
}

void Settings::setSearchOrder(const QString &order) {
    searchOrder = order;
    emit searchOrderChanged();
}

void Settings::setTheme(const QString &aTheme) {
    theme = aTheme;
    if (aTheme == "light") {
        setGradientColor(QString("#edece8"));
        setGradientColorHigh("white");
    }
    else {
        setGradientColor(QString("black"));
        setGradientColorHigh("#524e4e");
    }
    emit themeChanged();
}

void Settings::setActiveColor(const QString &color) {
    activeColor = color;
    emit activeColorChanged();
}

void Settings::setActiveColorString(const QString &colorString) {
    activeColorString = colorString;
    emit activeColorStringChanged();
}

void Settings::setGradientColor(const QString &color) {
    gradientColor = color;
    emit gradientColorChanged();
}

void Settings::setGradientColorHigh(const QString &color) {
    gradientColorHigh = color;
    emit gradientColorHighChanged();
}

void Settings::setLanguage(const QString &lang) {
    language = lang;
    emit languageChanged();
}

void Settings::setOrientation(const QString &orientation) {
    screenOrientation = orientation;
    emit orientationChanged(orientation);
}

void Settings::setNetworkProxy(const QString &proxy) {
    networkProxy = proxy;
    emit networkProxyChanged(proxy);
}

void Settings::setMediaPlayer(const QString &player) {
    mediaPlayer = player;
    emit mediaPlayerChanged(player);
}

void Settings::setDefaultSite(const QString &site) {
    defaultSite = site;
    emit defaultSiteChanged();
}

void Settings::setYouTubeUsername(const QString &user) {
    youtubeUsername = user;
    emit youtubeUsernameChanged(user);
}

void Settings::setYouTubeAccessToken(const QString &token) {
    youtubeAccessToken = token;
    emit youtubeAccessTokenChanged(token);
}

void Settings::setDailymotionAccount(const QString &user, const QString &aToken, const QString &rToken, int tokenExpiry) {
    dailymotionUsername = user;
    dailymotionAccessToken = aToken;
    dailymotionRefreshToken = rToken;
    emit dailymotionAccountChanged(user, aToken, rToken, tokenExpiry);
}

void Settings::clearYouTubeAccount() {
    setYouTubeUsername(QString());
    setYouTubeAccessToken(QString());
    saveSettings();
}

void Settings::clearDailymotionAccount() {
    setDailymotionAccount(QString(), QString(), QString(), 0);
    saveSettings();
}

void Settings::setFacebookAccessToken(const QString &token) {
    facebookAccessToken = token;
    emit facebookAccessTokenChanged(token);
}

void Settings::setTwitterAccount(const QString &token, const QString &secret) {
    twitterAccessToken = token;
    twitterTokenSecret = secret;
    emit twitterAccountChanged(token, secret);
}

void Settings::clearFacebookAccount() {
    setFacebookAccessToken(QString());
    saveSettings();
}

void Settings::clearTwitterAccount() {
    setTwitterAccount(QString(), QString());
    saveSettings();
}

void Settings::setXTubeEnabled(bool enabled) {
    xtubeEnabled = enabled;
    emit xtubeEnabledChanged();
}

void Settings::setXTubeDownloadPath(const QString &path) {
    xtubeDownloadPath = path;
    QDir pathCheck;
    pathCheck.mkpath(path + ".thumbnails/");
    emit xtubeDownloadPathChanged(path);
}

QVariantList Settings::getLanguages() const {
    QVariantList languages;
    QVariantMap language;
    QString localeName;
    QString languageName;
    QDir directory(QString("/opt/qmltube/qml/qmltube/i18n/"));
    QStringList translationFiles = directory.entryList(QStringList("*.qm"));
    foreach (QString file, translationFiles) {
        localeName = file.section('_', 1).left(2);
        QLocale locale(localeName);
        languageName = QLocale::languageToString(locale.language());
        if (languageName.size() > 1) {
            language.insert("name", languageName);
            language.insert("value", localeName);
            languages.append(language);
        }
    }
    return languages;
}
