import QtQuick 1.0
import "scripts/youtube.js" as YT
import "scripts/dailymotion.js" as DM
import "scripts/createobject.js" as ObjectCreator

Rectangle {
    id: appWindow

    property bool __videoPlaying : pageStack.currentPage.videoPlaying ? true : false
    property bool inPortrait: appWindow.height > appWindow.width

    /* YouTube feeds */

    property string _RECOMMENDED_FEED: "http://gdata.youtube.com/feeds/api/users/default/recommendations?v=2&max-results=30&alt=json"
    property string _WATCH_LATER_FEED: "http://gdata.youtube.com/feeds/api/users/default/watch_later?v=2&max-results=30&alt=json"
    property string _USER_ACTIVITY_FEED: "http://gdata.youtube.com/feeds/api/users/default/events?v=2&inline=true&max-results=30&alt=json"
    property string _FRIENDS_ACTIVITY_FEED: "http://gdata.youtube.com/feeds/api/users/default/friendsactivity?v=2&inline=true&max-results=30&alt=json"
    property string _UPLOADS_FEED: "http://gdata.youtube.com/feeds/api/users/default/uploads?v=2&max-results=30&alt=json"
    property string _FAVOURITES_FEED: "http://gdata.youtube.com/feeds/api/users/default/favorites?v=2&max-results=30&alt=json"
    property string _PLAYLISTS_FEED: "http://gdata.youtube.com/feeds/api/users/default/playlists?v=2&max-results=30&alt=json"
    property string _SUBSCRIPTIONS_FEED: "http://gdata.youtube.com/feeds/api/users/default/subscriptions?v=2&max-results=30&alt=json"
    property string _NEW_SUB_VIDEOS_FEED: "http://gdata.youtube.com/feeds/api/users/default/newsubscriptionvideos?v=2&max-results=30&alt=json"
    property string _MOST_RECENT_FEED: "http://gdata.youtube.com/feeds/api/standardfeeds/most_recent?v=2&max-results=30&alt=json"
    property string _MOST_VIEWED_FEED: "http://gdata.youtube.com/feeds/api/standardfeeds/most_viewed?v=2&max-results=30&time=today&alt=json"
    property string _CATEGORY_FEED: "http://gdata.youtube.com/feeds/api/videos?v=2&max-results=30&alt=json&category="

    /* Dailymotion feeds */

    property string _DM_UPLOADS_FEED: "https://api.dailymotion.com/me/videos?limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS
    property string _DM_FAVOURITES_FEED: "https://api.dailymotion.com/me/favorites?limit=30&fields=" + _DM_VIDEO_FIELDS
    property string _DM_PLAYLISTS_FEED: "https://api.dailymotion.com/me/playlists?limit=30&fields=" + _DM_PLAYLIST_FIELDS
    property string _DM_GROUPS_FEED: "https://api.dailymotion.com/me/groups?limit=30&fields=" + _DM_PLAYLIST_FIELDS
    property string _DM_SUBSCRIPTIONS_FEED: "https://api.dailymotion.com/me/following?limit=30&fields=" + _DM_USER_FIELDS
    property string _DM_NEW_SUB_VIDEOS_FEED: "https://api.dailymotion.com/me/subscriptions?limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS
    property string _DM_MOST_RECENT_FEED: "https://api.dailymotion.com/videos?sort=recent&limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS
    property string _DM_MOST_VIEWED_FEED: "https://api.dailymotion.com/videos?sort=visited-today&limit=30&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS
    property string _DM_CATEGORY_FEED: "https://api.dailymotion.com/videos?limit=30&sort=recent&family_filter=" + Settings.safeSearch.toString() + "&fields=" + _DM_VIDEO_FIELDS + "&channel="
    property string _DM_VIDEO_FIELDS: "id,title,description,duration,owner_url,owner_username,owner,owner_avatar_medium_url,comments_total,thumbnail_medium_url,thumbnail_large_url,rating,ratings_total,created_time,views_total,tags"
    property string _DM_USER_FIELDS: "id,username"
    property string _DM_PLAYLIST_FIELDS: "id,name,description,owner,owner.username,owner.avatar_medium_url"

    /* Theme variables */

    property string _BACKGROUND_COLOR : Settings.appTheme == "light" ? "white" : "black"
    property string _TEXT_COLOR : Settings.appTheme == "light" ? "black" : "white"
    property int _SMALL_FONT_SIZE : 20
    property int _STANDARD_FONT_SIZE : 24
    property int _LARGE_FONT_SIZE : 40

    function toggleBusy(isBusy) {
        titleArea.isBusy = isBusy;
    }

    function search(query, site, order, type, durationFilter) {
        var ytSearch;
        if (type == "playlists") {
            ytSearch = YT.getYouTubePlaylistSearch(query);
            pageStack.push(Qt.resolvedUrl("YTPlaylistsPage.qml"));
            pageStack.currentPage.autoLoad = false;
            pageStack.currentPage.setPlaylistFeed(ytSearch);
        }
        else {
            var searchOrder;
            if (order) {
                searchOrder = order;
            }
            else {
                searchOrder = Settings.searchOrder;
            }
            ytSearch = YT.getYouTubeSearch(query, searchOrder, durationFilter);
            var dmSearch = DM.getSearch(query, searchOrder);
            pageStack.push(Qt.resolvedUrl("VideoListPage.qml"));
            pageStack.currentPage.setVideoFeeds({ "site": site, "youtube": ytSearch, "dailymotion": dmSearch });
        }
        pageStack.currentPage.title = qsTr("Search") + " ('" + query + "')";
    }

    function appendDownloadQueue(videoList, convertToAudio) {
        var downloadItem;
        for (var i = 0; i < videoList.length; i++) {
            downloadItem = videoList[i];
            if (!downloadItem.status) {
                downloadItem["status"] = Settings.defaultTransferStatus;
            }
            if (convertToAudio !== null) {
                downloadItem["convertToAudio"] = convertToAudio;
            }
            downloadItem["eta"] = undefined;
            downloadItem["progress"] = undefined;
            downloadItem["reason"] = "";
            downloadItem["size"] = undefined;
            downloadItem["type"] = transfers._DOWNLOAD;
            Database.storeDownload(downloadItem);
            transfers.appendTransfer(downloadItem);
        }
        messages.displayMessage(messages._VIDEO_DOWNLOAD_ADDED);
    }

    function playVideos(videoList) {
        if (Settings.mediaPlayer == "cutetubePlayer") {
            pageStack.push(Qt.resolvedUrl("VideoPlaybackPage.qml"));
            pageStack.currentPage.setPlaylist(videoList);
        }
        else {
            var video = videoList[0];
            if (video.archive) {
                Utils.playVideo(video.filePath);
            }
            else {
                toggleBusy(true);
                if (video.youtube) {
                    YouTube.getVideoUrl(video.videoId);
                }
                else if (video.dailymotion) {
                    Dailymotion.getVideoUrl(video.id);
                }
                else if (video.xvideos) {
                    XTube.getXVideosVideoUrl(video.videoId);
                }
                else if (video.youporn) {
                    XTube.getYouPornVideoUrl(video.videoId);
                }
            }
        }
    }

    width: 800
    height: 480
    anchors.fill: parent
    color: __videoPlaying ? "black" : _BACKGROUND_COLOR

    Component.onCompleted: {
        var downloads = Database.getStoredDownloads();
        if (downloads.length > 0) {
            appendDownloadQueue(downloads);
        }
    }

    Connections {
        /* Connect to signals from C++ object Utils */

        target: Utils
        onAlert: messages.displayMessage(message)
    }

    Connections {
        /* Connect to signals from C++ object Settings */

        target: Settings
        onAlert: messages.displayMessage(message)
    }

    Connections {
        /* Connect to signals from C++ object Database */

        target: Database
        onAlert: messages.displayMessage(message)
    }

    Connections {
        /* Connect to signals from C++ object DownloadManager */

        target: DownloadManager
        onAlert: messages.displayMessage(message)
        onDownloadStarted: transfers.setProperty(transfers.currentTransfer, "status", transfers._IN_PROGRESS)
        onDownloadPaused: transfers.setProperty(transfers.currentTransfer, "status", transfers._PAUSED)
        onDownloadFailed: transfers.transferFailed(reason)
        onDownloadCompleted: transfers.transferCompleted()
        onDownloadCancelled: {
            transfers.deleteTransfer(transfers.currentTransfer);
            transfers.getNextTransfer();
        }
        onProgressChanged: {
            if (pageStack.currentPage.transfersDisplayed) {
                transfers.set(transfers.currentTransfer, { "progress": progress, "eta": eta });
            }
        }
        onSizeChanged: transfers.setProperty(transfers.currentTransfer, "size", size);
    }

    Connections {
        /* Connect to signals from C++ object YouTube */

        target: YouTube
        onAlert: messages.displayMessage(message)
        onGotVideoUrl: toggleBusy(false)
        onGotVideoIds: YT.getVideosFromIds(ids)
        onAddedToFavourites: messages.displayMessage(messages._ADDED_TO_FAVOURITES)
        onDeletedFromFavourites: messages.displayMessage(messages._DELETED_FROM_FAVOURITES)
        onVideoInFavourites: messages.displayMessage(messages._VIDEO_IN_FAVOURITES)
        onAddedToPlaylist: {
            messages.displayMessage(messages._ADDED_TO_PLAYLIST);
            youtubePlaylistModel.reset();
        }
        onAddedToWatchLaterPlaylist: messages.displayMessage(messages._ADDED_TO_PLAYLIST)
        onDeletedFromPlaylist: youtubePlaylistModel.reset();
        onAccessTokenChanged: {
            youtubePlaylistModel.reset();
            youtubeSubscriptionsModel.reset();
        }
        onPlaylistCreated: {
            messages.displayMessage(messages._PLAYLIST_CREATED);
            youtubePlaylistModel.reset();
        }
        onPlaylistDeleted: {
            messages.displayMessage(messages._PLAYLIST_DELETED);
            youtubePlaylistModel.reset();
        }
        onSubscribed: youtubeSubscriptionsModel.reset()
        onUnsubscribed: youtubeSubscriptionsModel.reset()
        onUploadStarted: transfers.setProperty(transfers.currentTransfer, "status", transfers._IN_PROGRESS)
        onUploadFailed: transfers.transferFailed(reason)
        onUploadCompleted: transfers.transferCompleted()
        onUploadCancelled: {
            transfers.deleteTransfer(transfers.currentTransfer);
            transfers.getNextTransfer();
        }
        onProgressChanged: {
            if (pageStack.currentPage.transfersDisplayed) {
                transfers.set(transfers.currentTransfer, {"progress": progress, "eta": eta});
            }
        }
        onSizeChanged: transfers.setProperty(transfers.currentTransfer, "size", size);
    }

    Connections {
        /* Connect to signals from C++ object Dailymotion */

        target: Dailymotion
        onAlert: messages.displayMessage(message)
        onGotVideoUrl: toggleBusy(false)
        onAddedToFavourites: messages.displayMessage(messages._ADDED_TO_FAVOURITES)
        onDeletedFromFavourites: messages.displayMessage(messages._DELETED_FROM_FAVOURITES)
        onAddedToPlaylist: messages.displayMessage(messages._ADDED_TO_PLAYLIST)
        onUserSignedInChanged: {
            dailymotionPlaylistModel.reset();
            dailymotionGroupModel.reset();
            dailymotionSubscriptionsModel.reset();
            if ((Dailymotion.userSignedIn) && (Settings.homescreen == "subscriptionVideos")) {
                loadNewSubscriptionVideos(true);
            }
        }
        onPlaylistCreated: {
            messages.displayMessage(messages._PLAYLIST_CREATED);
            playlistTimer.restart();
        }
        onPlaylistDeleted: {
            messages.displayMessage(messages._PLAYLIST_DELETED);
            dailymotionPlaylistModel.reset();
        }
        onGroupJoined: dailymotionGroupModel.reset()
        onGroupLeft: dailymotionGroupModel.reset()
        onFollowed: dailymotionSubscriptionsModel.reset()
        onUnfollowed: dailymotionSubscriptionsModel.reset()
    }

    Connections {
        target: XTube
        onAlert: messages.displayMessage(message)
        onGotVideoUrl: toggleBusy(false)
    }

    Image {
        id: background

        anchors.fill: appWindow
        source: (appWindow.inPortrait) ? "images/backgroundportrait.png" : "images/backgroundlandscape.png"
        smooth: true
        visible: (!__videoPlaying) && (Settings.appTheme != "light")
    }

    ListModel {
        /* Holds the users YouTube playlists */

        id: youtubePlaylistModel

        property bool loading: false
        property int totalResults
        property int page: 0
        property bool error: false

        function reset() {
            youtubePlaylistModel.page = 0;
            youtubePlaylistModel.clear();
            if (YouTube.userSignedIn) {
                YT.getCurrentUserPlaylists(_PLAYLISTS_FEED);
            }
        }

        Component.onCompleted: reset()
    }

    Connections {
        target: youtubePlaylistModel
        onLoadingChanged: {
            if ((!youtubePlaylistModel.loading) && (!youtubePlaylistModel.error) && (youtubePlaylistModel.count < youtubePlaylistModel.totalResults)) {
                YT.getCurrentUserPlaylists(_PLAYLISTS_FEED);
            }
        }
    }

    ListModel {
        /* Holds the users YouTube subscriptions */

        id: youtubeSubscriptionsModel

        property bool loading: false
        property int totalResults
        property int page: 0
        property bool error: false

        function reset() {
            youtubeSubscriptionsModel.page = 0;
            youtubeSubscriptionsModel.clear();
            if (YouTube.userSignedIn) {
                YT.getCurrentUserSubscriptions(_SUBSCRIPTIONS_FEED);
            }
        }

        Component.onCompleted: reset()
    }

    Connections {
        target: youtubeSubscriptionsModel
        onLoadingChanged: {
            if ((!youtubeSubscriptionsModel.loading) && (!youtubeSubscriptionsModel.error) && (youtubeSubscriptionsModel.count < youtubeSubscriptionsModel.totalResults)) {
                YT.getCurrentUserSubscriptions(_SUBSCRIPTIONS_FEED);
            }
        }
    }

    ListModel {
        /* Holds the users Dailymotion playlists */

        id: dailymotionPlaylistModel

        property bool loading: false
        property bool moreResults: false
        property int page: 1
        property bool error: false

        function reset() {
            dailymotionPlaylistModel.clear();
            dailymotionPlaylistModel.page = 1;
            dailymotionPlaylistModel.moreResults = false;
            dailymotionPlaylistModel.error = false;
            if (Dailymotion.userSignedIn) {
                DM.getCurrentUserPlaylists();
            }
        }

        Component.onCompleted: reset()
    }

    Timer {
        id: playlistTimer

        interval: 1000
        onTriggered: dailymotionPlaylistModel.reset()
    }

    Connections {
        target: dailymotionPlaylistModel
        onLoadingChanged: {
            if ((!dailymotionPlaylistModel.loading) && (!dailymotionPlaylistModel.error) && (dailymotionPlaylistModel.moreResults)) {
                DM.getCurrentUserPlaylists();
            }
        }
    }

    ListModel {
        /* Holds the users Dailymotion groups */

        id: dailymotionGroupModel

        property bool loading: false
        property bool moreResults: false
        property int page: 1
        property bool error: false

        function reset() {
            dailymotionGroupModel.clear();
            dailymotionGroupModel.page = 1;
            dailymotionGroupModel.moreResults = false;
            dailymotionGroupModel.error = false;
            if (Dailymotion.userSignedIn) {
                DM.getCurrentUserGroups();
            }
        }

        Component.onCompleted: reset()
    }

    Connections {
        target: dailymotionGroupModel
        onLoadingChanged: {
            if ((!dailymotionGroupModel.loading) && (!dailymotionGroupModel.error) && (dailymotionGroupModel.moreResults)) {
                DM.getCurrentUserGroups();
            }
        }
    }

    ListModel {
        /* Holds the users Dailymotion subscriptions */

        id: dailymotionSubscriptionsModel

        property bool loading: false
        property bool moreResults: false
        property int page: 1
        property bool error: false

        function reset() {
            dailymotionSubscriptionsModel.clear();
            dailymotionSubscriptionsModel.page = 1;
            dailymotionSubscriptionsModel.moreResults = false;
            dailymotionSubscriptionsModel.error = false;
            if (Dailymotion.userSignedIn) {
                DM.getCurrentUserSubscriptions();
            }
        }

        Component.onCompleted: reset()
    }

    Connections {
        target: dailymotionSubscriptionsModel
        onLoadingChanged: {
            if ((!dailymotionSubscriptionsModel.loading) && (!dailymotionSubscriptionsModel.error) && (dailymotionSubscriptionsModel.moreResults)) {
                DM.getCurrentUserSubscriptions();
            }
        }
    }

    PageStack {
        id: pageStack
    }

    TitleArea {
        id: titleArea
    }

    InfoBanner {
        id: messages
    }

    TransferModel {
        id: transfers
    }

    MessageDialog {}
}
