import QtQuick 1.0
import QtMobility.gallery 1.1
import "components"
import "scripts/videolistscripts.js" as Scripts
import "scripts/createobject.js" as ObjectCreator

Page {
    id: root

    function playSelectedVideos() {
        var list = [];
        for (var i = 0; i < videoList.checkList.length; i++) {
            list.push(createArchiveItem(archiveModel.get(videoList.checkList[i])));
        }
        playVideos(list);
        videoList.checkList = [];
    }

    function showConfirmDeleteDialog() {
        var deleteDialog = ObjectCreator.createObject(Qt.resolvedUrl("ConfirmDeleteDialog.qml"), appWindow);
        deleteDialog.accepted.connect(deleteSelectedVideos);
        deleteDialog.open();
    }

    function deleteSelectedVideos() {
        var list = [];
        for (var i = 0; i < videoList.checkList.length; i++) {
            list.push(archiveModel.get(videoList.checkList[i]).filePath);
        }
        Database.deleteArchiveVideos(list);
        videoList.checkList = [];
    }

    function changeSortOrder(order) {
        archiveModel.sortProperties = [order];
        archiveModel.reload();
    }

    function createArchiveItem(video) {
        var archiveItem = {};
        var title = video.fileName.slice(0, video.fileName.lastIndexOf("."));
        archiveItem["itemId"] = video.itemId;
        archiveItem["title"] = title;
        archiveItem["name"] = title;
        archiveItem["filePath"] = video.filePath;
        archiveItem["archive"] = true;
        return archiveItem;
    }

    Connections {
        target: DownloadManager
        onDownloadCompleted: reloadTimer.restart()
    }

    Connections {
        target: Database
        onArchiveVideosDeleted: reloadTimer.restart()
    }

    Timer {
        id: reloadTimer

        interval: 3000
        onTriggered: archiveModel.reload()
    }

    title: qsTr("XTube Archive")
    tools: Menu {

        menuItems: [

            MenuSelectionItem {
                title: qsTr("Sort by")
                model: ListModel {
                    ListElement { name: QT_TR_NOOP("Date (asc)"); value: "+lastModified" }
                    ListElement { name: QT_TR_NOOP("Date (desc)"); value: "-lastModified" }
                    ListElement { name: QT_TR_NOOP("Title (asc)"); value: "+fileName" }
                    ListElement { name: QT_TR_NOOP("Title (desc)"); value: "-fileName" }
                }
                enabled: videoList.checkList.length === 0
                initialValue: archiveModel.sortProperties[0]
                onValueChosen: changeSortOrder(value)
            },

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                closeMenuWhenClicked: false
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? Scripts.selectAll() : Scripts.selectNone()
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                enabled: videoList.checkList.length > 0
                onClicked: playSelectedVideos()
            },

            MenuItem {
                text: qsTr("Delete")
                enabled: videoList.checkList.length > 0
                onClicked: showConfirmDeleteDialog()
            }
        ]
    }

    ListView {
        id: videoList

        property variant checkList : []

        anchors.fill: parent
        boundsBehavior: Flickable.DragOverBounds
        highlightRangeMode: ListView.StrictlyEnforceRange
        cacheBuffer: 2500
        clip: true
        model: DocumentGalleryModel {
            id: archiveModel

            rootItem: "folder::" + Settings.xtubeDownloadPath.slice(0, -1)
            rootType: DocumentGallery.File
            scope: DocumentGalleryModel.DirectDescendants
            properties: ["filePath", "fileName"]
            sortProperties: ["-lastModified"]
            filter: GalleryFilterUnion {
                filters: [
                    GalleryWildcardFilter {
                        property: "fileName"
                        value: "*.mp4"
                    },
                    GalleryWildcardFilter {
                        property: "fileName"
                        value: "*.m4a"
                    }
                ]
            }
        }
        delegate: XTubeArchiveDelegate {
            id: delegate

            function addOrRemoveFromCheckList() {
                var cl = videoList.checkList;
                if (!delegate.checked) {
                    cl.push(index);
                }
                else {
                    for (var i = 0; i < cl.length; i++) {
                        if (cl[i] == index) {
                            cl.splice(i, 1);
                        }
                    }
                }
                videoList.checkList = cl;
            }

            checked: Scripts.indexInCheckList(index)
            onClicked: {
                if (videoList.checkList.length === 0) {
                    playVideos([createArchiveItem(archiveModel.get(index))]);
                }
                else {
                    addOrRemoveFromCheckList();
                }
            }
            onPressAndHold: addOrRemoveFromCheckList()
        }

        ScrollBar {}
    }

    Text {
        id: noResultsText

        anchors.centerIn: videoList
        font.pixelSize: _LARGE_FONT_SIZE
        font.bold: true
        color: "#4d4d4d"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("Archive empty")
        visible: (archiveModel.status != DocumentGalleryModel.Active) && (archiveModel.count === 0)
    }

    BusyDialog {
        anchors.centerIn: videoList
        visible: archiveModel.status == DocumentGalleryModel.Active
    }
}
