#ifndef YouTube_H
#define YouTube_H

#include <QObject>
#include <QByteArray>
#include <QtNetwork/QNetworkAccessManager>
#include <QUrl>
#include <QFile>
#include <QTime>
#include <QVariantMap>
#include <QClipboard>

class QNetworkAccessManager;
class QNetworkReply;

class YouTube : public QObject {
    Q_OBJECT
    Q_PROPERTY(QString username
               READ getUsername
               NOTIFY usernameChanged)
    Q_PROPERTY(QString accessToken
               READ getAccessToken
               NOTIFY accessTokenChanged)
    Q_PROPERTY(bool userSignedIn
               READ getUserSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(bool isUploading
               READ isUploading
               NOTIFY uploadingChanged)

public:
    explicit YouTube(QObject *parent = 0);
    virtual ~YouTube();
    void setNetworkAccessManager(QNetworkAccessManager *manager);
    QString getUsername() const { return username; }
    QString getAccessToken() const { return accessToken; }
    bool getUserSignedIn() const { return !(accessToken.isEmpty()); }
    bool isUploading() const { return uploading; }

public slots:
    QString getDeveloperKey() const { return QString(developerKey); }
    void setUsername(const QString &user);
    void setAccessToken(const QString &token);
    void addToFavourites(const QStringList &ids);
    void deleteFromFavourites(const QStringList &ids);
    void addToPlaylist(const QStringList &ids, const QString &playlistId);
    void deleteFromPlaylist(const QStringList &ids, const QString &playlistId);
    void addToWatchLaterPlaylist(const QStringList &ids);
    void deleteFromWatchLaterPlaylist(const QStringList &ids, const QString &playlistId);
    void createPlaylist(const QString &title, const QString &description, const bool &isPrivate);
    void deletePlaylist(const QString &playlistId);
    void subscribeToChannel(const QString &user);
    void unsubscribeToChannel(const QString &subscriptionId);
    void rateVideo(const QString &videoId, const QString &likeOrDislike);
    void addComment(const QString &videoId, const QString &comment);
    void replyToComment(const QString &videoId, const QString &commentId, const QString &comment);
    void updateVideoMetadata(const QString &videoId, const QVariantMap &metadata);
    void setPlaybackQuality(const QString &quality);
    void getVideoUrl(const QString &videoId);
    void uploadVideo(const QVariantMap &video);
    void abortVideoUpload();
    void setMonitorClipboard(bool monitor);
    void signOut();

private slots:
    void setUploadUrl();
    void performVideoUpload();
    void updateUploadProgress(qint64 bytesSent, qint64 bytesTotal);
    void resumeVideoUpload();
    void uploadFinished();
    void postRequest(const QUrl &url, const QByteArray &xml, bool batch = true);
    void putRequest(const QUrl &url, const QByteArray &xml, bool batch = true);
    void deleteRequest(const QUrl &url, bool batch = true);
    void postFinished();
    void batchPostFinished();
    void parseVideoPage(QNetworkReply *reply);
    void setIsUploading(const bool isUploading) { uploading = isUploading; emit uploadingChanged(); }
    void getVideoLinks();
    void checkIfSignedOut();

private:
    QNetworkAccessManager *nam;
    QFile *fileToBeUploaded;
    QUrl uploadUrl;
    QTime uploadTime;
    int uploadRetries;
    QNetworkReply *uploadReply;
    QByteArray developerKey;
    QString username;
    QString accessToken;
    int playbackFormat;
    QHash<QString, int> pbMap;
    QString message;
    bool uploading;
    QClipboard *clipboard;

signals:
    void gotVideoUrl(const QString &videoUrl);
    void videoUrlError();
    void alert(const QString &message);
    void usernameChanged();
    void accessTokenChanged();
    void postSuccessful(const QString &id);
    void addedToFavourites();
    void deletedFromFavourites();
    void videoInFavourites();
    void addedToPlaylist(const QString &id);
    void deletedFromPlaylist(const QString &id);
    void addedToWatchLaterPlaylist();
    void deletedFromWatchLaterPlaylist();
    void playlistCreated();
    void playlistDeleted();
    void subscribed(const QString &id);
    void unsubscribed(const QString &id);
    void uploadStarted();
    void commentAdded(const QString &id);
    void videoRated(const QString &id);
    void cannotRate();
    void videoMetadataUpdated(const QString &id);
    void progressChanged(float progress, int eta);
    void uploadFailed(const QString &reason);
    void uploadCompleted();
    void uploadCancelled();
    void sizeChanged(qint64 size);
    void userSignedInChanged();
    void uploadingChanged();
    void gotVideoIds(const QStringList &ids);
    void signedOut();
};

#endif // YouTube_H
