#include "utils.h"
#include <QFile>
#include <QDir>
#include <QApplication>
#include <QClipboard>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtDBus/QDBusInterface>
#include <QTimer>
#include <QTime>
#include <QDesktopServices>

#ifdef Q_WS_MAEMO_5
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <QX11Info>
#endif

Utils::Utils(QObject *parent) :
    QObject(parent), viewer(0) {

    systemInfo = new QSystemInfo(this);
    deviceInfo = new QSystemDeviceInfo(this);
    connect(deviceInfo, SIGNAL(batteryLevelChanged(int)), this, SIGNAL(batteryLevelChanged()));
    QTimer *timer = new QTimer(this);
    timer->setSingleShot(false);
    timer->setInterval(30000);
    connect(timer, SIGNAL(timeout()), this, SIGNAL(currentTimeChanged()));
    timer->start();
    emit currentTimeChanged();
    emit batteryLevelChanged();
}

void Utils::setView(QmlApplicationViewer *view) {
    viewer = view;
}

void Utils::toggleState() {
    if (!viewer->isFullScreen()) {
        viewer->showFullScreen();
    }
    else {
        viewer->showMaximized();
    }
}

QString Utils::getDocsDirectory() const {
    return QDesktopServices::storageLocation(QDesktopServices::DocumentsLocation);
}

QString Utils::getCurrentTime() const {
    return QTime::currentTime().toString("h:mm ap");
}

int Utils::getBatteryLevel() const {
    return deviceInfo->batteryLevel();
}

QString Utils::getInstalledQtVersion() const {
    return systemInfo->version(QSystemInfo::QtCore);
}

void Utils::doNotDisturb(bool preventDisturb) {
#ifdef Q_WS_MAEMO_5
    Atom atom = XInternAtom(QX11Info::display() , "_HILDON_DO_NOT_DISTURB", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_DO_NOT_DISTURB. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    if (preventDisturb) {
        long state = 1;
        XChangeProperty(
                    QX11Info::display(),
                    viewer->winId(),
                    atom,
                    XA_INTEGER,
                    32,
                    PropModeReplace,
                    (unsigned char *) &state,
                    1);
    }
    else {
        XDeleteProperty(QX11Info::display(), viewer->winId(), atom);
    }
#else
    Q_UNUSED(preventDisturb)
#endif
}

void Utils::minimize() {
    QDBusConnection bus = QDBusConnection::sessionBus();

    QDBusMessage message = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    bus.send(message);
}

void Utils::setMediaPlayer(const QString &player) {
    mediaPlayer = player.toLower().replace(" ", "");
}

bool Utils::widgetInstalled() const {
    return QFile::exists(QString("/usr/lib/hildon-desktop/qmltube-widget"));
}

bool Utils::xTubeInstalled() const {
    return QFile::exists(QString("/opt/qmltube/qml/qmltube/scripts/xtube.js"));
}

QVariantList Utils::getInstalledMediaPlayers() const {
    QVariantList playerList;
    QVariantMap player;

    if ((QFile::exists("/usr/bin/mplayer"))) {
        player.insert("name", tr("MPlayer"));
        player.insert("value", "mplayer");
        playerList.append(player);
    }
    if ((QFile::exists("/opt/kmplayer/bin/kmplayer")) || (QFile::exists("/usr/bin/kmplayer"))) {
        player.insert("name", tr("KMPlayer"));
        player.insert("value", "kmplayer");
        playerList.append(player);
    }
    return playerList;
}

void Utils::playVideo(const QString &url) {
    if (mediaPlayer != "cutetubePlayer") {
        if (mediaPlayer == "mplayer") {
            QStringList args;
            args << "-cache" << "4096" << "-fs" << url;
            QProcess *player = new QProcess();
            connect(player, SIGNAL(finished(int, QProcess::ExitStatus)), player, SLOT(deleteLater()));
            player->start("/usr/bin/mplayer", args);
        }
        else {
            QDBusConnection bus = QDBusConnection::sessionBus();
            QDBusInterface dbus_iface("com.nokia." + mediaPlayer, "/com/nokia/" + mediaPlayer, "com.nokia." + mediaPlayer, bus);
            dbus_iface.call("mime_open", url);
        }
    }
}


bool Utils::pathExists(const QString &path) {
    QDir dir(path);
    return dir.exists();
}

void Utils::copyToClipboard(const QString &url) {
    QApplication::clipboard()->setText(url);
    emit alert(tr("Copied to clipboard"));
}
