import QtQuick 1.0
import "components"
import "scripts/createobject.js" as ObjectCreator
import "scripts/dailymotion.js" as DM
import "scripts/videolistscripts.js" as Scripts

Page {
    id: root

    property bool showingFavourites: videoFeed == _DM_FAVOURITES_FEED
    property string videoFeed
    property alias checkList: videoList.checkList

    function setVideoFeed(feed) {
        videoFeed = feed;
        DM.getVideos(feed);
    }

    function goToVideoInfo(video) {
        pageStack.push(Qt.resolvedUrl("DMInfoPage.qml"));
        pageStack.currentPage.setVideo(video);
    }


    function selectAll() {
        Scripts.selectAll()
    }

    function selectNone() {
        Scripts.selectNone();
    }

    function addVideosToFavourites() {
        DM.addVideosToFavourites();
    }

    function deleteVideosFromFavourites() {
        DM.deleteVideosFromFavourites();
    }

    function showPlaylistDialog() {
        DM.showPlaylistDialog();
    }

    function addVideosToPlaybackQueue() {
        DM.addVideosToPlaybackQueue();
    }

    function addVideosToDownloadQueue(convertToAudio) {
        DM.addVideosToDownloadQueue(convertToAudio);
    }

    tools: Menu {

        menuItems: [

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                closeMenuWhenClicked: false
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? selectAll() : selectNone()
            },

            MenuItem {
                text: showingFavourites ? qsTr("Delete from favourites") : qsTr("Add to favourites")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showingFavourites ? deleteVideosFromFavourites() : addVideosToFavourites()
            },

            MenuItem {
                text: qsTr("Add to playlist")
                visible: Dailymotion.userSignedIn
                enabled: videoList.checkList.length > 0
                onClicked: showPlaylistDialog()
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToPlaybackQueue()
            },

            MenuItem {
                text: qsTr("Download")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                enabled: videoList.checkList.length > 0
                onClicked: addVideosToDownloadQueue(true)
            }
        ]
    }

    Connections {
        target: Dailymotion
        onAddedToFavourites: if (showingFavourites) videoListTimer.restart();
        onDeletedFromFavourites: if (showingFavourites) videoListTimer.restart();
    }

    Timer {
        id: videoListTimer

        interval: 3000
        onTriggered: videoListModel.reset()
    }

    ListView {
        id: videoList

        property variant checkList : []

        anchors { fill: parent; topMargin: pageStack.currentPage == root ? titleArea.height : 0 }
        boundsBehavior: Flickable.DragOverBounds
        highlightRangeMode: ListView.StrictlyEnforceRange
        cacheBuffer: 2500
        clip: true
        onCurrentIndexChanged: {
            if ((videoList.currentIndex > 0) && (videoList.count - videoList.currentIndex == 1)
                    && (videoListModel.moreResults)
                    && (!videoListModel.loading)) {
                DM.getVideos(videoFeed);
            }
        }

        footer: Item {
            id: footer

            width: videoList.width
            height: 100
            visible: videoListModel.loading
            opacity: footer.visible ? 1 : 0

            BusyDialog {
                anchors.centerIn: footer
                opacity: footer.opacity
            }
        }

        model: ListModel {
            id: videoListModel

            property bool loading : false
            property bool error: false
            property bool moreResults: false
            property int page: 1

            function reset() {
                videoListModel.clear();
                videoListModel.page = 1;
                videoListModel.moreResults = false;
                videoListModel.error = false;
                DM.getVideos(videoFeed);
            }
        }

        delegate: DMListDelegate {
            id: delegate

            function addOrRemoveFromCheckList() {
                var cl = videoList.checkList;
                if (!delegate.checked) {
                    cl.push(index);
                }
                else {
                    for (var i = 0; i < cl.length; i++) {
                        if (cl[i] == index) {
                            cl.splice(i, 1);
                        }
                    }
                }
                videoList.checkList = cl;
            }

            checked: Scripts.indexInCheckList(index)
            onClicked: {
                if (videoList.checkList.length === 0) {
                    goToVideoInfo(ObjectCreator.cloneObject(videoListModel.get(index)));
                }
                else {
                    addOrRemoveFromCheckList();
                }
            }
            onPressAndHold: addOrRemoveFromCheckList()
            onPlayClicked: if (videoList.checkList.length === 0) playVideos([ObjectCreator.cloneObject(videoListModel.get(index))]);
        }

        ScrollBar {}
    }

    Text {
        id: noResultsText

        anchors.centerIn: videoList
        font.pixelSize: _LARGE_FONT_SIZE
        font.bold: true
        color: "#4d4d4d"
        horizontalAlignment: Text.AlignHCenter
        verticalAlignment: Text.AlignVCenter
        text: qsTr("No videos found")
        visible: (!videoListModel.loading) && (videoListModel.count == 0)
    }
}
