import QtQuick 1.0
import "components"
import "scripts/createobject.js" as ObjectCreator
import "scripts/dailymotion.js" as DM
import "scripts/dateandtime.js" as DT

Page {
    id: root

    property variant video: []
    property bool isFavourite: false

    function setVideo(videoObject) {
        video = videoObject;
        if (Dailymotion.userSignedIn) {
            checkIfFavourite(videoObject.id);
        }
    }

    function checkIfFavourite(id) {
        var request = new XMLHttpRequest();
        request.onreadystatechange = function() {
            if (request.readyState == XMLHttpRequest.DONE) {
                try {
                    var res = eval("(" + request.responseText + ")");
                    if (res.list[0].id == id) {
                        isFavourite = true;
                    }
                }
                catch(err) {
                    isFavourite = false;
                }
            }
        }
        request.open("GET", "https://api.dailymotion.com/me/favorites/" + id + "?fields=id");
        request.setRequestHeader("Authorization", "OAuth " + Dailymotion.accessToken);
        request.send();
    }

    function goToUserProfile() {
        pageStack.push(Qt.resolvedUrl("DMUserPage.qml"));
        pageStack.currentPage.getUserProfile(video.owner);
    }

    function showEditMetadataDialog() {
        var metadataDialog = ObjectCreator.createObject(Qt.resolvedUrl("DMEditMetadataDialog.qml"), appWindow);
        metadataDialog.getVideoMetadata(video.id);
        metadataDialog.metadataUpdated.connect(updateMetadata);
        metadataDialog.open();
    }

    function updateMetadata(metadata) {
        var v = video;
        v["title"] = metadata.title;
        v["description"] = metadata.description;
        v["tags"] = metadata.tags;
        video = v;
    }

    function showPlaylistDialog() {
        var playlistDialog = ObjectCreator.createObject(Qt.resolvedUrl("DMPlaylistDialog.qml"), appWindow);
        playlistDialog.playlistChosen.connect(addVideoToPlaylist);
        playlistDialog.open();
    }

    function addVideoToPlaylist(playlistId) {
        toggleBusy(true);
        Dailymotion.addToPlaylist(video.id, playlistId);
    }

    function setFavourite() {
        toggleBusy(true);
        if (isFavourite) {
            Dailymotion.deleteFromFavourites(video.id);
        }
        else {
            Dailymotion.addToFavourites(video.id);
        }
    }

    title: qsTr("Info")
    tools: Menu {

        menuItems: [

            MenuItem {
                text: isFavourite ? qsTr("Delete from favourites") : qsTr("Add to favourites")
                onClicked: setFavourite()
                visible: Dailymotion.userSignedIn
            },

            MenuItem {
                text: qsTr("Add to playlist")
                onClicked: showPlaylistDialog()
                visible: Dailymotion.userSignedIn
            },

            MenuItem {
                text: qsTr("Edit metadata")
                onClicked: showEditMetadataDialog()
                visible: (Dailymotion.userSignedIn) && (video.owner) && (video.owner == Dailymotion.username) ? true : false
            },

            MenuItem {
                text: qsTr("Download")
                onClicked: appendDownloadQueue([ObjectCreator.cloneObject(video)], false)
            },

            MenuItem {
                text: qsTr("Download (audio)")
                onClicked: appendDownloadQueue([ObjectCreator.cloneObject(video)], true)
            }
        ]
    }

    Image {
        id: frame

        width: Math.floor(root.width / 3.2)
        height: Math.floor(frame.width / (4 / 3))
        anchors { left: parent.left; leftMargin: 10; top: parent.top; topMargin: 10 }
        source: "images/frame.png"
        sourceSize.width: width
        sourceSize.height: height
        smooth: true

        Image {
            id: thumb

            z: -1
            anchors.fill: frame
            source: !video.thumbnail_large_url ? "" : video.thumbnail_large_url
            smooth: true
            onStatusChanged: if (thumb.status == Image.Error) thumb.source = Settings.appTheme == "light" ? "images/video.png" : "images/video-white.png";

            Rectangle {
                id: durationLabel

                width: durationText.width + 30
                height: durationText.height + 10
                anchors { bottom: thumb.bottom; right: thumb.right }
                color: "black"
                opacity: 0.5
                smooth: true
                visible: thumb.status == Image.Ready
            }

            Text {
                id: durationText

                anchors.centerIn: durationLabel
                text: !video.duration ? "" : DT.getDuration(video.duration)
                color: "white"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                visible: thumb.status == Image.Ready
            }

            Image {
                width: appWindow.inPortrait ? 80 : 50
                height: width
                anchors.centerIn: thumb
                source: frameMouseArea.pressed ? "images/play-button-" + Settings.activeColorString + ".png" : "images/play-button.png"
                smooth: true
                visible: thumb.status == Image.Ready
            }
        }

        MouseArea {
            id: frameMouseArea

            anchors.fill: frame
            onClicked: playVideos([video])
        }
    }

    Flickable {
        id: scrollArea

        anchors { top: frame.top; left: frame.right; leftMargin: 10; right: parent.right; rightMargin: 10; bottom: parent.bottom }
        clip: true
        contentWidth: width
        contentHeight: textColumn.height + 10
        flickableDirection: Flickable.VerticalFlick
        boundsBehavior: Flickable.DragOverBounds

        Column {
            id: textColumn

            spacing: 10
            width: parent.width
            height: childrenRect.height

            Text {
                id: titleText

                width: textColumn.width
                color: _TEXT_COLOR
                text: !video.title ? "" : video.title
                wrapMode: TextEdit.WordWrap
            }

            Row {
                width: parent.width
                spacing: 5

                Image {
                    id: avatar

                    width: 40
                    height:40
                    source: !video.owner_avatar_medium_url ? "images/avatar.png" : video.owner_avatar_medium_url
                    sourceSize.width: width
                    sourceSize.height: height
                    smooth: true
                    opacity: authorMouseArea.pressed ? 0.5 : 1
                    onStatusChanged: if (status == Image.Error) source = "images/avatar.png";

                    MouseArea {
                        id: authorMouseArea

                        anchors.fill: parent
                        enabled: (!video.owner) || (video.owner != Dailymotion.username) ? true : false
                        onClicked: goToUserProfile()
                    }
                }

                Text {
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: _TEXT_COLOR
                    wrapMode: Text.WordWrap
                    text: !video.owner ? "" : qsTr("By") + " " + video.owner_username + " " + qsTr("on") + " " + DT.getDate(video.created_time * 1000)
                }
            }

            Row {
                spacing: 10

                RatingIndicator {
                    ratingValue: !video.rating ? 0 : video.rating
                    count: !video.ratings_total ? 0 : video.ratings_total
                    maximumValue: 5
                }

                Text {
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: !video.views_total ? "0" : video.views_total
                }

                Text {
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: _TEXT_COLOR
                    text: qsTr("views")
                }
            }

            Text {
                font.pixelSize: _SMALL_FONT_SIZE
                color: _TEXT_COLOR
                text: qsTr("Description")
            }

            Text {
                id: descriptionText

                width: textColumn.width
                text: (!video.description) || (video.description === "") ? qsTr("No description") : video.description.replace(/\r/g, "")
                color: "#4d4d4d"
                font.pixelSize: _SMALL_FONT_SIZE
                wrapMode: TextEdit.WordWrap
            }

            Text {
                font.pixelSize: _SMALL_FONT_SIZE
                color: _TEXT_COLOR
                text: qsTr("Tags")
            }

            Flow {
                spacing: 10
                width: parent.width

                Text {
                    font.pixelSize: _SMALL_FONT_SIZE
                    color: "#4d4d4d"
                    text: qsTr("No tags")
                    visible: (!repeater.model.length) || (repeater.model.length === 0) ? true : false
                }

                Repeater {
                    id: repeater

                    model: (!video.tags) ? [] : video.tags.split(",")

                    Text {
                        font.pixelSize: _SMALL_FONT_SIZE
                        color: Settings.activeColor
                        opacity: children[0].pressed ? 0.5 : 1
                        text: modelData

                        MouseArea {
                            anchors.fill: parent
                            onClicked: search(parent.text, "dailymotion")
                        }

                        Text {
                            anchors.left: parent.right
                            font.pixelSize: _SMALL_FONT_SIZE
                            color: "#4d4d4d"
                            text: ","
                            visible: index < (repeater.model.length - 1)
                        }
                    }
                }
            }
        }
    }

    states: State {
        name: "portrait"
        when: appWindow.inPortrait
        PropertyChanges { target: frame; width: root.width - 20 }
        AnchorChanges { target: scrollArea; anchors { left: parent.left; top: frame.bottom } }
        PropertyChanges { target: scrollArea; anchors { topMargin: 10 } }
    }
}
