#include "downloadmanager.h"
#include <QtNetwork/QNetworkRequest>
#include <QtNetwork/QNetworkReply>
#include <QRegExp>
#include <QMap>
#include <QList>
#include <QUrl>
#include <QDir>

#define ILLEGAL_CHARS "[\"@&~=\\/:?#!|<>*^]"

DownloadManager::DownloadManager(QObject *parent) :
    QObject(parent), downloading(false), downloadReply(0) {
    dlMap["1080p"] = 37;
    dlMap["720p"] = 22;
    dlMap["480p"] = 35;
    dlMap["hq"] = 18;

    connect(this, SIGNAL(gotVideoUrl(QUrl)), this, SLOT(performDownload(QUrl)));
    converter = new QProcess(this);
    connect(converter, SIGNAL(started()), this, SIGNAL(conversionStarted()));
    connect(converter, SIGNAL(finished(int, QProcess::ExitStatus)), this, SLOT(conversionFinished(int, QProcess::ExitStatus)));
}

void DownloadManager::setNetworkAccessManager(QNetworkAccessManager *manager) {
    nam = manager;
}

void DownloadManager::setYouTubeDownloadQuality(const QString &quality) {
    downloadFormat = dlMap.value(quality, 18);
}

void DownloadManager::setDownloadPath(const QString &path) {
    downloadPath = path;
}

void DownloadManager::setXTubeDownloadPath(const QString &path) {
    xtubeDownloadPath = path;
}

void DownloadManager::getYouTubeVideoUrl(const QString &videoId) {
    QString pageUrl = "http://www.youtube.com/get_video_info?&video_id=" + videoId + "&el=detailpage&ps=default&eurl=&gl=US&hl=en";
    QNetworkAccessManager *manager = new QNetworkAccessManager(this);
    QNetworkRequest request;
    request.setUrl(QUrl(pageUrl));
    connect(manager, SIGNAL(finished(QNetworkReply*)), this, SLOT(parseYouTubeVideoPage(QNetworkReply*)));
    manager->get(request);
}

void DownloadManager::parseYouTubeVideoPage(QNetworkReply *reply) {
    QNetworkAccessManager *manager = qobject_cast<QNetworkAccessManager*>(sender());

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));
    if (!response.contains("fmt_stream_map=url=")) {
        emit downloadFailed(tr("Video not available for download"));
    }
    else {
        response = response.split("fmt_stream_map=url=").at(1);
        QStringList parts = response.split(QRegExp(",url=|&url="));
        QString part;
        QStringList keySplit;
        QString url;
        int key;
        for (int i = 0; i < parts.length(); i++) {
            part = parts[i];
            url = QByteArray::fromPercentEncoding(part.left(part.indexOf("&type=video")).toAscii()).replace("%2C", ",");
            keySplit = part.split("&itag=");
            if (keySplit.size() > 1) {
                key = keySplit.at(1).split(QRegExp("[&,]")).first().toInt();
                formats[key] = url;
            }
        }
        QList<int> flist;
        flist << dlMap.value("1080p") << dlMap.value("720p") << dlMap.value("480p") << dlMap.value("hq");
        QString videoUrl;
        int index = flist.indexOf(downloadFormat);
        while ((videoUrl.isEmpty()) && (index < flist.size())) {
            videoUrl = formats.value(flist.at(index), "");
            index++;
        }
        if (!videoUrl.startsWith("http")) {
            emit downloadFailed(tr("Video not available for download"));
        }
        else {
            emit gotVideoUrl(QUrl(videoUrl));
        }
    }
    reply->deleteLater();
    manager->deleteLater();
}

void DownloadManager::getDailymotionVideoUrl(const QString &videoId) {
    QString pageUrl = "http://iphone.dailymotion.com/video/" + videoId;
    QNetworkRequest request;
    request.setUrl(QUrl(pageUrl));
    QNetworkReply *reply = nam->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
}

void DownloadManager::parseDailymotionVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit downloadFailed(tr("Video not available for download"));
        return;
    }
    QString response(reply->readAll());
    QString videoUrl = response.split("type=\"video/x-m4v\" href=\"").at(1).split('"').at(0);
    if (!videoUrl.startsWith("http")) {
        emit downloadFailed(tr("Video not available"));
    }
    else {
        emit gotVideoUrl(QUrl(videoUrl));
    }
    reply->deleteLater();
}

void DownloadManager::getYouPornVideoUrl(const QString &url) {
    QUrl ypUrl(url);
    QNetworkRequest request(ypUrl);
    QNetworkReply *reply = nam->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseYouPornVideoPage()));
}

void DownloadManager::parseYouPornVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        emit downloadFailed(tr("Video not available for download"));
        return;
    }

    QString response(reply->readAll());
    QStringList results = response.split("\"play_video\"><a href=\"");
    results.takeFirst();
    if (!results.isEmpty()) {
        QString videoUrl = results.takeFirst().split('"').first().replace("amp;", "");
        if (videoUrl.startsWith("http")) {
            emit gotVideoUrl(QUrl(videoUrl));
        }
        else {
            emit downloadFailed(tr("Video not available for download"));
        }
    }
    else {
        emit downloadFailed(tr("Video not available for download"));
    }
    reply->deleteLater();
}

void DownloadManager::pauseDownload() {
    downloadReply->abort();
}

void DownloadManager::cancelDownload() {
    if (downloading) {
        downloadReply->abort();
    }
    output.remove();
    emit downloadCancelled();
}

void DownloadManager::deleteIncompleteDownload(QString title) {
    QString fileName(downloadPath + title.replace(QRegExp(ILLEGAL_CHARS), "_") + ".mp4.partial");
    QFile::remove(fileName);
}

void DownloadManager::downloadThumbnail() {
    QUrl url(videoToDownload.value("thumbnail").toString());
    QNetworkRequest request(url);
    QNetworkReply *reply = nam->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(thumbnailDownloadFinished()));
}

void DownloadManager::thumbnailDownloadFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());
    if (!reply) {
        return;
    }
    QString fileName(videoToDownload.value("fileName").toString());
    QFile thumbnailFile(fileName.insert(fileName.lastIndexOf("/") + 1, QString(".thumbnails/")).replace(QRegExp("\\.[a-zA-Z0-9]{3,4}$"), QString(".jpg")));
    if (thumbnailFile.open(QIODevice::WriteOnly)) {
        thumbnailFile.write(reply->readAll());
    }
    thumbnailFile.close();
    if ((videoToDownload.value("convertToAudio").toInt()) && (QFile::exists(QString("/usr/bin/ffmpeg")))) {
        convertToAudio();
    }
    else {
        emit downloadCompleted();
    }
    reply->deleteLater();
}

void DownloadManager::convertToAudio() {
    QString ffmpeg("/usr/bin/ffmpeg");
    QStringList args;
    QString input(videoToDownload.value("fileName").toString());
    QString audioFile(input.replace(QRegExp("\\.[a-zA-Z0-9]{3,4}$"), QString(".m4a")));
    videoToDownload.insert("audioFileName", audioFile);
    args << "-i" << videoToDownload.value("fileName").toString() << "-acodec" << "copy" << "-y" << "-vn" << audioFile;
    converter->start(ffmpeg, args);
}

void DownloadManager::conversionFinished(int exitCode, QProcess::ExitStatus exitStatus) {
    if ((exitCode == 0) && (exitStatus == QProcess::NormalExit)) {
        QFile output(videoToDownload.value("audioFileName").toString());
        if (output.size() > 0) {
            emit downloadCompleted();
            QFile::remove(videoToDownload.value("fileName").toString());
        }
        else {
            emit downloadFailed(tr("Conversion failed"));
            output.remove();
        }
    }
    else {
        emit downloadFailed(tr("Conversion failed"));
    }
}

void DownloadManager::startDownload(const QVariantMap &video) {
    setIsDownloading(true);
    emit downloadStarted();
    videoToDownload = video;
    QString title(videoToDownload.value("title").toString());
    QDir thumbPath;
    if ((videoToDownload.value("youtube").toInt()) || (videoToDownload.value("dailymotion").toInt())) {
        thumbPath.setPath(QString(downloadPath + ".thumbnails/"));
        output.setFileName(downloadPath + title.replace(QRegExp(ILLEGAL_CHARS), "_") + ".mp4.partial");
    }
    else {
        thumbPath.setPath(QString(xtubeDownloadPath + ".thumbnails/"));
        output.setFileName(xtubeDownloadPath + title.replace(QRegExp(ILLEGAL_CHARS), "_") + ".mp4.partial");
    }
    thumbPath.mkpath(thumbPath.path());
    if (output.exists()) {
        if (!output.open(QIODevice::Append)) {
            setIsDownloading(false);
            emit downloadFailed(tr("Unable to write to file"));
            return;                 // skip this download
        }
    }
    else if (!output.open(QIODevice::WriteOnly)) {
        setIsDownloading(false);
        emit downloadFailed(tr("Unable to write to file"));
        return;                 // skip this download
    }
    if (videoToDownload.value("youtube").toInt()) {
        getYouTubeVideoUrl(video.value("videoId").toString());
    }
    else if (videoToDownload.value("dailymotion").toInt()) {
        getDailymotionVideoUrl(video.value("id").toString());
    }
    else if (videoToDownload.value("youporn").toInt()) {
        getYouPornVideoUrl(video.value("videoId").toString());
    }
    else {
        performDownload(QUrl(video.value("videoId").toString()));
    }
}

void DownloadManager::performDownload(const QUrl &videoUrl) {
    videoToDownload.insert("videoUrl", videoUrl);
    QNetworkRequest request(videoUrl);

    if (output.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(output.size()) + "-"); // Set 'Range' header if resuming a download
    }

    downloadReply = nam->get(request);
    downloadTime.start();
    connect(downloadReply, SIGNAL(metaDataChanged()), this, SLOT(updateSize()));
    connect(downloadReply, SIGNAL(downloadProgress(qint64, qint64)), this, SLOT(updateProgress(qint64,qint64)));
    connect(downloadReply, SIGNAL(finished()), this, SLOT(downloadFinished()));
    connect(downloadReply, SIGNAL(readyRead()), this, SLOT(downloadReadyRead()));
}

void DownloadManager::updateSize() {
    emit sizeChanged(output.size() + downloadReply->header(QNetworkRequest::ContentLengthHeader).toLongLong());
}

void DownloadManager::updateProgress(qint64 bytesReceived, qint64 bytesTotal) {
    float progress = float(output.size() + bytesReceived) / float(output.size() + bytesTotal);
    int eta = int (((downloadTime.elapsed() / progress) - downloadTime.elapsed()) / 1000);
    emit progressChanged(progress, eta);
}

void DownloadManager::downloadFinished() {
    QUrl redirect = downloadReply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
    if (!redirect.isEmpty()) {
        performDownload(redirect); // Follow redirect :P
    }
    else {
        if (downloadReply->error()) {
            if (downloadReply->error() == QNetworkReply::OperationCanceledError) {
                output.close();
                setIsDownloading(false);
                emit downloadPaused();
            }
            else {
                QString statusText = downloadReply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                if ((statusText.isEmpty()) || (statusText == "OK") || (statusText == "Partial Content")) {
                    performDownload(videoToDownload.value("videoUrl").toUrl());
                }
                else {
                    output.close();
                    setIsDownloading(false);
                    emit downloadFailed(statusText);
                }
            }
        }
        else {
            output.close();
            setIsDownloading(false);
            QString fileName = output.fileName().left(output.fileName().lastIndexOf("."));
            int num = 1;
            bool fileSaved = output.rename(fileName);
            while ((!fileSaved) && (num < 10)) {
                if (num == 1) {
                    fileName = fileName.insert(fileName.lastIndexOf("."), "(" + QByteArray::number(num) + ")");
                }
                else {
                    fileName = fileName.replace(fileName.lastIndexOf("(" + QByteArray::number(num - 1) + ")"), 3, "(" + QByteArray::number(num) + ")");
                }
                fileSaved = output.rename(fileName);
                num++;
            }
            videoToDownload.insert("fileName", fileName);
            downloadThumbnail();
        }
    }
}

void DownloadManager::downloadReadyRead() {
    output.write(downloadReply->readAll());
}
