#ifndef DAILYMOTION_H
#define DAILYMOTION_H

#include <QObject>
#include <QUrl>
#include <QFile>
#include <QTime>
#include <QVariantMap>
#include <QTimer>

class QNetworkAccessManager;
class QNetworkReply;

class Dailymotion : public QObject {
    Q_OBJECT
    Q_PROPERTY(QString username
               READ getUsername
               NOTIFY usernameChanged)
    Q_PROPERTY(QString accessToken
               READ getAccessToken
               NOTIFY accessTokenChanged)
    Q_PROPERTY(QString refreshToken
               READ getRefreshToken
               NOTIFY refreshTokenChanged)
    Q_PROPERTY(QString tokenExpiry
               READ getTokenExpiry
               NOTIFY tokenExpiryChanged)
    Q_PROPERTY(QString clientId
               READ getClientId
               NOTIFY clientIdChanged)
    Q_PROPERTY(QString clientSecret
               READ getClientSecret
               NOTIFY clientSecretChanged)
    Q_PROPERTY(bool userSignedIn
               READ getUserSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(QString busyMessage
               READ getBusyMessage
               NOTIFY busyMessageChanged)


public:
    explicit Dailymotion(QObject *parent = 0);

    void setNetworkAccessManager(QNetworkAccessManager *manager);
    QString getUsername() const { return username; }
    QString getAccessToken() const { return accessToken; }
    QString getRefreshToken() const { return refreshToken; }
    int getTokenExpiry() const { return tokenExpiry; }
    QString getClientId() const { return clientId; }
    QString getClientSecret() const { return clientSecret; }
    bool isUploading() const { return uploading; }
    bool getUserSignedIn() const { return !accessToken.isEmpty(); }
    QString getBusyMessage() const { return busyMessage; }

public slots:
    void signIn(const QString &user, const QString &pass);
    void revokeAccess();
    void getVideoUrl(const QString &id);
    void setAccount(const QString &user, const QString &aToken, const QString &rToken, int expiry);
    void addToFavourites(const QString &id);
    void deleteFromFavourites(const QString &id);
    void createPlaylist(const QString &title);
    void deletePlaylist(const QString &id);
    void addToPlaylist(const QString &id, const QString &playlistId);
    void deleteFromPlaylist(const QString &id, const QString &playlistId);
    void joinGroup(const QString &id);
    void leaveGroup(const QString &id);
    void addComment(const QString &id, const QString &comment);
    void follow(const QString &userId);
    void unfollow(const QString &userId);
    void uploadVideo(const QVariantMap &video);
    void abortVideoUpload();
    void updateVideoMetadata(const QString &id, const QVariantMap &metadata);
    void setPlaybackPath(const QString &path);

private slots:
    void setUsername(const QString &user);
    void setAccessToken(const QString &token);
    void setRefreshToken(const QString &token, int expiry);
    void setTokenExpiry(int expiry);
    void refreshAccessToken(QString token = QString());
    void checkForToken();
    void checkAccessRevoked();
    void postRequest(const QUrl &url, const QByteArray &data = QByteArray());
    void deleteRequest(const QUrl &url);
    void postFinished();
    void parseVideoPage();
    void setUploadUrl();
    void performVideoUpload(const QUrl &url);
    void updateUploadProgress(qint64 bytesSent, qint64 bytesTotal);
    void uploadFinished();
    void setUploadMetadata(const QString &uploadUrl);
    void setBusyMessage(const QString &message);
    void clearBusyMessage();
    void disconnectSignals();

private:
    QNetworkAccessManager *nam;
    QString clientId;
    QString clientSecret;
    QString accessToken;
    QString refreshToken;
    QTimer *refreshTimer;
    int tokenExpiry;
    QString username;
    QFile *fileToBeUploaded;
    QVariantMap videoToUpload;
    QTime uploadTime;
    QNetworkReply *uploadReply;
    bool uploading;
    QString busyMessage;
    QString playbackPath;

signals:
    void gotVideoUrl(const QString &videoUrl);
    void videoUrlError();
    void alert(const QString &message);
    void usernameChanged();
    void userSignedInChanged();
    void accessTokenChanged();
    void clientIdChanged();
    void clientSecretChanged();
    void refreshTokenChanged();
    void tokenExpiryChanged();
    void postSuccessful(const QString &id);
    void postFailed();
    void addedToFavourites();
    void deletedFromFavourites();
    void addedToPlaylist();
    void deletedFromPlaylist();
    void playlistCreated(const QString &id);
    void playlistDeleted();
    void groupJoined();
    void groupLeft();
    void commentAdded();
    void followed();
    void unfollowed();
    void progressChanged(float progress, int eta);
    void sizeChanged(qint64 size);
    void uploadCompleted();
    void uploadStarted();
    void uploadCancelled();
    void uploadFailed(const QString &reason);
    void waitingForMetadata(const QString &id);
    void videoMetadataUpdated();
    void accessRevoked();
    void signedIn(const QString &user, const QString &aToken, const QString &rToken, int expiry);
    void busyMessageChanged();
};

#endif // DAILYMOTION_H
