import QtQuick 1.0
import QtMultimediaKit 1.2
import "scripts/settings.js" as Settings

Rectangle {
    id: window

    property variant currentVideo
    property int playlistPosition : 0

    signal playbackStopped

    function setPlaylist(videoList) {
        currentVideo = videoList[0];
        getVideoUrl(currentVideo.playerUrl);
        for (var i = 0; i < videoList.length; i++) {
            playlistModel.append(videoList[i]);
        }
    }

    function getVideoUrl(playerUrl) {
        /* Retrieve the videoUrl from the YouTube webpage */

        var videoUrl = "";
        var quality = Settings.getSetting("playbackQuality");
        var vq = { "720p": 22, "480p": 35, "360p": 34, "High Quality": 18, "Mobile": 5 };
        var formats = {};

        var request = new XMLHttpRequest();
        request.onreadystatechange = function() {
            if (request.readyState == XMLHttpRequest.DONE) {
                var response = request.responseText.replace(/[\n ]/gi, "");
                var pos = response.indexOf("\",\"fmt_url_map\":\"") + 17;
                var pos2 = response.indexOf('\"', pos);
                var parts = response.slice(pos, pos2).split("|");
                var key = parseInt(parts[0]);
                for (var i = 0; i < parts.length; i++) {
                    var part = parts[i];
                    var keyAndValue = part.split(",");
                    var url = keyAndValue[0].replace(/\\/g, "").replace(/u0026/g, "&");
                    formats[key] = url;
                    key = parseInt(keyAndValue.pop());
                }
                var sliced = false;
                var index = 0;
                var flist = [22, 35, 34, 18, 5];
                while ((index < flist.length) && (!sliced)) {
                    if (flist[index] == vq[quality]) {
                        flist = flist.slice(index);
                        sliced = true;
                    }
                    index++;
                }
                index = 0;
                while ((index < flist.length) && (videoUrl == "")) {
                    videoUrl = formats[flist[index]];
                    index++;
                }
                if (videoUrl == ""){
                    displayMessage(qsTr("Error: Unable to retrieve video"));
                }
                else {
                    videoPlayer.setVideo(videoUrl);
                }
                console.log(videoUrl);
            }
        }
        request.open("GET", playerUrl);
        request.send();
    }

    function getTime(msecs) {
        /* Convert seconds to HH:MM:SS format */

        var secs = Math.floor(msecs / 1000);
        var hours = Math.floor(secs / 3600);
        var minutes = Math.floor(secs / 60) - (hours * 60);
        var seconds = secs - (hours * 3600) - (minutes * 60);
        if (seconds < 10) {
            seconds = "0" + seconds;
        }
        var duration = minutes + ":" + seconds;
        if (hours > 0) {
            duration = hours + ":" + duration;
        }
        return duration;
    }

    color: "black"

    onPlaylistPositionChanged: {
        getVideoUrl(playlistModel.get(playlistPosition).playerUrl);
        //        videoPlayer.setVideo(playlistModel.get(playlistPosition).filePath);
        currentVideo = playlistModel.get(playlistPosition);

    }

    Video{
        id: videoPlayer

        function setVideo(videoUrl) {
            videoPlayer.source = videoUrl
            videoPlayer.play()
        }

        width: window.width
        height: window.height
        anchors.top: window.top
        z: 30

        MouseArea {
            anchors.fill: videoPlayer
            onClicked: controls.showControls = !controls.showControls
        }

        Item {
            id: controls

            property bool showControls : false

            z: 50
            anchors.fill: videoPlayer

            Item {
                id: titleBar

                anchors.top: controls.top
                width: controls.width
                height: 50

                Rectangle {
                    anchors.fill: titleBar
                    width: controls.width
                    height: 50
                    color: "black"
                    opacity: controls.showControls ? 0.5 : 0
                    smooth: true

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                ToolButton {
                    id: minimizeButton

                    anchors { verticalCenter: titleBar.verticalCenter; left: titleBar.left; leftMargin: 10 }
                    icon: "ui-images/minimizeicon.png"
                    opacity: controls.showControls ? 1 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }

                    Connections {
                        onButtonClicked: Controller.minimize()
                        onButtonHeld: Controller.toggleState()
                    }

                }

                Text {
                    id: title

                    anchors { fill: titleBar; leftMargin: 70; rightMargin: 200 }
                    font.pixelSize: _STANDARD_FONT_SIZE
                    elide: Text.ElideRight
                    verticalAlignment: Text.AlignVCenter
                    color: "white"
                    smooth: true
                    text: currentVideo == undefined ? "" : currentVideo.title
                    opacity: controls.showControls ? 1 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                Text {
                    id: time

                    anchors {right: titleBar.right; rightMargin: 70 }
                    height: titleBar.height
                    font.pixelSize: _STANDARD_FONT_SIZE
                    verticalAlignment: Text.AlignVCenter
                    color: "white"
                    smooth: true
                    text: currentVideo == undefined ? "0:00/0:00" : getTime(videoPlayer.position) + "/" + getTime(videoPlayer.duration)
                    opacity: controls.showControls ? 1 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                CloseButton {
                    id: closeButton

                    anchors { right: titleBar.right; rightMargin: 10; verticalCenter: titleBar.verticalCenter }
                    opacity: controls.showControls ? 1 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }

                    onButtonClicked: {
                        videoPlayer.stop();
                        playbackStopped();
                    }
                }

                Rectangle {
                    id: frame

                    width: 124
                    height: 94
                    anchors { left: titleBar.left; leftMargin: 70; top: titleBar.bottom }
                    color: "black"
                    smooth: true
                    opacity: (controls.showControls && window.state == "") ? 0.5 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                Image {
                    id: thumb

                    anchors { fill: frame; margins: 2 }
                    sourceSize.width: thumb.width
                    sourceSize.height: thumb.height
                    source: currentVideo.thumbnail
                    smooth: true
                    opacity: (controls.showControls && window.state == "") ? 1 : 0

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }
            }

            Row {
                id: buttonRow

                anchors.centerIn: controls
                spacing: 50

                PushButton {
                    id: previousButton

                    width: 100
                    height: 100
                    icon: "ui-images/previousicon.png"
                    opacity: controls.showControls ? 1 : 0
                    onButtonClicked: {
                        controlsTimer.restart;
                        if (videoPlayer.position > 5000) {
                            videoPlayer.position = 0;
                        }
                        else if (playlistPosition > 0) {
                            playlistPosition = playlistPosition - 1;
                        }
                    }

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                PushButton {
                    id: playButton

                    width: 100
                    height: 100
                    icon: videoPlayer.playing ? "ui-images/playicon.png" : "ui-images/pauseicon.png"
                    opacity: controls.showControls ? 1 : 0
                    onButtonClicked: {
                        controlsTimer.restart;
                        videoPlayer.paused = !videoPlayer.paused;
                    }

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }

                PushButton {
                    id: nextButton

                    width: 100
                    height: 100
                    icon: "ui-images/nexticon.png"
                    opacity: controls.showControls ? 1 : 0
                    onButtonClicked: {
                        controlsTimer.restart;
                        if (playlistPosition < playlistModel.count - 1) {
                            playlistPosition = playlistPosition + 1;
                        }
                    }

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                }
            }

            Rectangle {
                width: controls.width
                height: 20
                anchors.bottom: controls.bottom
                color: "black"
                smooth: true
                opacity: controls.showControls ? 0.5 : 0

                Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }

                Rectangle {
                    id: seekBar

                    width: currentVideo == undefined ? 0 : Math.floor(window.width * (videoPlayer.position / videoPlayer.duration))
                    height: parent.height
                    anchors.bottom: parent.bottom
                    color: _ACTIVE_COLOR_LOW
                    smooth: true
                    opacity: parent.opacity

                    Behavior on opacity { PropertyAnimation { properties: "opacity"; duration: 300 } }
                    Behavior on width { SmoothedAnimation { velocity: 1200 } }
                }

                MouseArea {
                    id: seekMouseArea

                    width: parent.width
                    height: 50
                    anchors.bottom: parent.bottom
                    onClicked: {
                        controlsTimer.restart;
                        videoPlayer.position = (Math.floor(mouseX / window.width * (videoPlayer.duration * 1000)));
                    }
                }
            }
        }
    }

    Timer {
        id: controlsTimer

        running: controls.showControls
        interval: 3000
        onTriggered: controls.showControls = false
    }

    ListModel {
        id: playlistModel
    }

    ListView {
        id: playlistView

        anchors { fill: window; topMargin: videoPlayer.height + 10 }
        model: playlistModel
        clip: true
        maximumFlickVelocity: 1000
        delegate: PlaybackDelegate {
            id: delegate

            onDelegateClicked: playlistPosition = index;
        }
    }

    states: State {
        name: "portrait"
        when: window.height > window.width
        PropertyChanges { target: videoPlayer; height: 360 }
    }
}
